/*
 * @brief Image programming code
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */
#include <string.h>
#include "chip.h"
#include "app_usbd_cfg.h"

/**********************************************************************
** Global data
**********************************************************************/

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
/* local data */
static USBD_HANDLE_T g_hUsb;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
const USBD_API_T *g_pUsbApi;

/*****************************************************************************
 * Private functions
 ****************************************************************************/
/* Initialize pin and clocks for USB port */
static void usb_pin_clk_init(void)
{
	/* enable USB main clock */
	Chip_Clock_SetUSBClockSource(SYSCTL_USBCLKSRC_PLLOUT, 1);
	/* Enable AHB clock to the USB block and USB RAM. */
	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_USB);
	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_USBRAM);
	/* power UP USB Phy */
	Chip_SYSCTL_PowerUp(SYSCTL_POWERDOWN_USBPAD_PD);
}

/* Setup system clocking */
static void SystemSetupClocking(void)
{
	volatile int i;

	/* Powerup main oscillator */
	Chip_SYSCTL_PowerUp(SYSCTL_POWERDOWN_SYSOSC_PD);

	/* Wait 200us for OSC to be stabilized, no status
	   indication, dummy wait. */
	for (i = 0; i < 0x100; i++) {}
	Chip_Clock_SetMainClockSource(SYSCTL_MAINCLKSRC_IRC);
	/* Set system PLL input to main oscillator */
	Chip_Clock_SetSystemPLLSource(SYSCTL_PLLCLKSRC_MAINOSC);

	/* Power down PLL to change the PLL divider ratio */
	Chip_SYSCTL_PowerDown(SYSCTL_POWERDOWN_SYSPLL_PD);

	/* Setup PLL for main oscillator rate (FCLKIN = 12MHz) * 4 = 48MHz
	   MSEL = 3 (this is pre-decremented), PSEL = 1 (for P = 2)
	   FCLKOUT = FCLKIN * (MSEL + 1) = 12MHz * 4 = 48MHz
	   FCCO = FCLKOUT * 2 * P = 48MHz * 2 * 2 = 192MHz (within FCCO range) */
	Chip_Clock_SetupSystemPLL(3, 1);

	/* Powerup system PLL */
	Chip_SYSCTL_PowerUp(SYSCTL_POWERDOWN_SYSPLL_PD);

	/* Wait for PLL to lock */
	while (!Chip_Clock_IsSystemPLLLocked()) {}

	/* Set system clock divider to 1 */
	Chip_Clock_SetSysClockDiv(1);

	/* Setup FLASH access to 3 clocks */
	Chip_FMC_SetFLASHAccess(FLASHTIM_50MHZ_CPU);

	/* Set main clock source to the system PLL. This will drive 48MHz
	   for the main clock and 48MHz for the system clock */
	Chip_Clock_SetMainClockSource(SYSCTL_MAINCLKSRC_PLLOUT);

	/* Set USB PLL input to main oscillator */
	Chip_Clock_SetUSBPLLSource(SYSCTL_PLLCLKSRC_MAINOSC);
	/* Setup USB PLL  (FCLKIN = 12MHz) * 4 = 48MHz
	   MSEL = 3 (this is pre-decremented), PSEL = 1 (for P = 2)
	   FCLKOUT = FCLKIN * (MSEL + 1) = 12MHz * 4 = 48MHz
	   FCCO = FCLKOUT * 2 * P = 48MHz * 2 * 2 = 192MHz (within FCCO range) */
	Chip_Clock_SetupUSBPLL(3, 1);

	/* Powerup USB PLL */
	Chip_SYSCTL_PowerUp(SYSCTL_POWERDOWN_USBPLL_PD);

	/* Wait for PLL to lock */
	while (!Chip_Clock_IsUSBPLLLocked()) {}
}

static ErrorCode_t update_device_status_patch(USBD_HANDLE_T hUsb)
{
	USB_CORE_CTRL_T *pCtrl;          
	pCtrl = (USB_CORE_CTRL_T *) hUsb;      /* convert the handle to control structure */
	if ( (((USB_CONFIGURATION_DESCRIPTOR *) (pCtrl->full_speed_desc))->bmAttributes & USB_CONFIG_POWERED_MASK) != 0 ) {
		/* This is SELF_POWERED. */
		pCtrl->device_status |= (0x01 << 0);
	}
	else {
		/* This is BUS_POWERED. */
		pCtrl->device_status &= ~(0x01 << 0);                     
	}
	if ( (((USB_CONFIGURATION_DESCRIPTOR *) (pCtrl->full_speed_desc))->bmAttributes & USB_CONFIG_REMOTE_WAKEUP) != 0 ) {
		/* This is REMOTE_WAKEUP enabled. */
		pCtrl->device_status |= (0x01 << 1);
	}
	else {
		/* This is REMOTE_WAKEUP disabled. */
		pCtrl->device_status &= ~(0x01 << 1);
	}
	return LPC_OK;
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/* Handle interrupt from USB */
void USB_IRQHandler(void)
{

	uint32_t *addr = (uint32_t *) LPC_USB->EPLISTSTART;
	/*	WORKAROUND for artf32289 ROM driver BUG:
	    As part of USB specification the device should respond
	    with STALL condition for any unsupported setup packet. The host will send
	    new setup packet/request on seeing STALL condition for EP0 instead of sending
	    a clear STALL request. Current driver in ROM doesn't clear the STALL
	    condition on new setup packet which should be fixed.
	 */
	if ( LPC_USB->DEVCMDSTAT & _BIT(8) ) {	/* if setup packet is received */
		addr[0] &= ~(_BIT(29));	/* clear EP0_OUT stall */
		addr[2] &= ~(_BIT(29) | _BIT(31));	/* clear EP0_IN stall and ACTIVE bit*/
	}
	USBD_API->hw->ISR(g_hUsb);


}

/* Handler for WCID USB device requests. */
static ErrorCode_t WCID_hdlr(USBD_HANDLE_T hUsb, void *data, uint32_t event)
{
	USB_CORE_CTRL_T *pCtrl = (USB_CORE_CTRL_T *) hUsb;
	ErrorCode_t ret = ERR_USBD_UNHANDLED;

	/* Handle Microsoft's WCID request for install less WinUSB operation.
	   Check https://github.com/pbatard/libwdi/wiki/WCID-Devices for more details.
	 */
	if (event == USB_EVT_SETUP) {
		switch (pCtrl->SetupPacket.bmRequestType.BM.Type) {
		case REQUEST_STANDARD:
			if ((pCtrl->SetupPacket.bmRequestType.BM.Recipient == REQUEST_TO_DEVICE) &&
				(pCtrl->SetupPacket.bRequest == USB_REQUEST_GET_DESCRIPTOR) &&
				(pCtrl->SetupPacket.wValue.WB.H == USB_BOS_TYPE)) {
				pCtrl->EP0Data.pData = (uint8_t *) USB_BOSDescriptor;
				pCtrl->EP0Data.Count = MIN(USB_BOSDescriptorSize, pCtrl->SetupPacket.wLength);
				USBD_API->core->DataInStage(pCtrl);
				ret = LPC_OK;
			}
            else if ((pCtrl->SetupPacket.bmRequestType.BM.Recipient == REQUEST_TO_DEVICE) &&
                     (pCtrl->SetupPacket.bRequest == USB_REQUEST_GET_DESCRIPTOR) &&
                     (pCtrl->SetupPacket.wValue.WB.H == USB_STRING_DESCRIPTOR_TYPE) &&
                     (pCtrl->SetupPacket.wValue.WB.L == 0x00EE)) {
                pCtrl->EP0Data.pData = (uint8_t *) WCID_String_Descriptor;
                pCtrl->EP0Data.Count = MIN(WCID_String_DescriptorSize, pCtrl->SetupPacket.wLength);
                USBD_API->core->DataInStage(pCtrl);
                ret = LPC_OK;
            }
			break;
        case REQUEST_VENDOR:
            if (pCtrl->SetupPacket.bRequest != WCID_VENDOR_CODE) {
                break;
            }
            switch (pCtrl->SetupPacket.bmRequestType.BM.Recipient) {
                case REQUEST_TO_DEVICE:
                    if (pCtrl->SetupPacket.wIndex.W == 0x0004) {
                        pCtrl->EP0Data.pData = (uint8_t *) WCID_CompatID_Descriptor;
                        pCtrl->EP0Data.Count = MIN(WCID_CompatID_DescriptorSize, pCtrl->SetupPacket.wLength);
                        USBD_API->core->DataInStage(pCtrl);
                        ret = LPC_OK;
                    }
                    /* Fall-through. Check note1 of
                       https://github.com/pbatard/libwdi/wiki/WCID-Devices#wiki-Defining_a_Device_Interface_GUID_or_other_device_specific_properties
                       break;
                    */

                case REQUEST_TO_INTERFACE:
                    if (pCtrl->SetupPacket.wIndex.W == 0x0005) {
                        pCtrl->EP0Data.pData = (uint8_t *) WCID_ExtProp_Descriptor;
                        pCtrl->EP0Data.Count = MIN(WCID_ExtProp_DescriptorSize, pCtrl->SetupPacket.wLength);
                        USBD_API->core->DataInStage(pCtrl);
                        ret = LPC_OK;
                    }
                    break;
            }
            break;
        }
	}

	return ret;
}

/** Main Function  */
int main(void)
{
	USBD_API_INIT_PARAM_T usb_param;
	USB_CORE_DESCS_T desc;
	ErrorCode_t ret = LPC_OK;
	volatile uint32_t i;

	/* Enable IOCON clock */
	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_IOCON);
	Chip_IOCON_PinMuxSet(LPC_IOCON, 0,  3,  (IOCON_FUNC1 | IOCON_MODE_INACT));		/* PIO0_3 used for USB_VBUS */
	Chip_IOCON_PinMuxSet(LPC_IOCON, 0,  6,  (IOCON_FUNC1 | IOCON_MODE_INACT));		/* PIO0_6 used for USB_CONNECT */

	/* Initialize board and chip */
	SystemSetupClocking();
	SystemCoreClockUpdate();
	
	/* Enable and setup SysTick Timer at a periodic rate */
	SysTick_Config(SystemCoreClock / 10);
	/* Enable AHB clock to the GPIO domain. */
  LPC_SYSCON->SYSAHBCLKCTRL |= (1<<6);
   /* Set port 1_26 to output */
  LPC_GPIO->DIR[1] |= (1<<26);

	/* enable clocks and USB PHY/pads */
	usb_pin_clk_init();

	/* Init USB API structure */
	g_pUsbApi = (const USBD_API_T *) LPC_ROM_API->usbdApiBase;

	/* initialize call back structures */
	memset((void *) &usb_param, 0, sizeof(USBD_API_INIT_PARAM_T));
	usb_param.usb_reg_base = LPC_USB0_BASE;
	/*	WORKAROUND for artf44835 ROM driver BUG:
	    Code clearing STALL bits in endpoint reset routine corrupts memory area
	    next to the endpoint control data. For example When EP0, EP1_IN, EP1_OUT,
	    EP2_IN are used we need to specify 3 here. But as a workaround for this
	    issue specify 4. So that extra EPs control structure acts as padding buffer
	    to avoid data corruption. Corruption of padding memory doesn�t affect the
	    stack/program behavior.
	 */
	usb_param.max_num_ep = 1 + 1;
	usb_param.mem_base = USB_STACK_MEM_BASE;
	usb_param.mem_size = USB_STACK_MEM_SIZE;
	usb_param.USB_Reset_Event = update_device_status_patch;
	// usb_param.USB_Configure_Event = USB_Configure_Event;

	/* Set the USB descriptors */
	desc.device_desc = (uint8_t *) &USB_DeviceDescriptor;
	desc.string_desc = (uint8_t *) &USB_StringDescriptor[0];
	/* Note, to pass USBCV test full-speed only devices should have both
	 * descriptor arrays point to same location and device_qualifier set
	 * to 0.
	 */
	desc.high_speed_desc = (uint8_t *) &USB_FsConfigDescriptor[0];
	desc.full_speed_desc = (uint8_t *) &USB_FsConfigDescriptor[0];
	desc.device_qualifier = 0;

	/* USB Initialization */
	ret = USBD_API->hw->Init(&g_hUsb, &desc, &usb_param);
	if (ret == LPC_OK) {
		
		/* Link Power Management is supported. */
    LPC_USB->DEVCMDSTAT |= (0x1<<11);
    LPC_USB->LPM |= (0x2<<4);/* RESUME duration. */

		/*	WORKAROUND for artf32219 ROM driver BUG:
		    The mem_base parameter part of USB_param structure returned
		    by Init() routine is not accurate causing memory allocation issues for
		    further components.
		 */
		usb_param.mem_base = USB_STACK_MEM_BASE + (USB_STACK_MEM_SIZE - usb_param.mem_size);
		/* register WCID handler */
	ret = USBD_API->core->RegisterClassHandler(g_hUsb, WCID_hdlr, 0);
		
		ret = usb_dfu_init(g_hUsb,
						   (USB_INTERFACE_DESCRIPTOR *) &USB_FsConfigDescriptor[sizeof(USB_CONFIGURATION_DESCRIPTOR)],
						   &usb_param.mem_base, &usb_param.mem_size);
		
		if (ret == LPC_OK) {
				/* enable USB interrupts */
	NVIC_EnableIRQ(USB0_IRQn);
			/* now connect */
			USBD_API->hw->Connect(g_hUsb, 1);
		}
	}

	while (1) {
				if (dfu_detach_sig)
				{
					for (i = 0; i < 0x4000; i++) {
						/* wait before detach */
					}
					enter_DFU_SL(g_hUsb);
				}
				else {
					__WFI();
				}
			
		}
}

/**********************************************************************
** Function name:
**
** Description:
**
** Parameters:
**
** Returned value:
**********************************************************************/


/* SysTick interrupt happens every 10 ms */
void SysTick_Handler(void)
{
  	LPC_GPIO->NOT[1] = 1<<26;
}
