/*
 * @brief Heart-rate Monitor Software Example (Cortex-M0+ Core)
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "board.h"

/** @defgroup PERIPH_M0SLAVE_5410X Multicore blinky example (M0 as slave)
 * @ingroup EXAMPLES_CORE_5410X
 * @include "multicore\m0slave_blinky\readme.txt"
 */

#define PDOWNMODE	POWER_POWER_DOWN
#define SAMPLE_FREQUENCY 20

/* Heart rate monitor related variables */
volatile int BPM;                  					/* Holds last BPM value calculated 														*/
volatile int Signal;                				/* Holds the current raw sample												  			*/
volatile int temp_data[SAMPLE_FREQUENCY];		/* Array that holds the set of incoming data 							 		*/
volatile int IBI = 600;             				/* Holds the time between beats, must be seeded! 		 	  			*/
volatile int Pulse = 0;     								/* True when pulse wave is high, false when it's low  				*/
volatile int QS = 0;        								/* Becomes true when a heart beat is detected 								*/
volatile int rate[10];                      /* Array to hold last ten IBI values													*/
volatile float sampleCounter = 0;           /* Used to determine pulse timing										 					*/
volatile float lastBeatTime = 0;            /* Used to find IBI  																					*/
volatile int P = 2548;                      /* Used to find peak in pulse wave, seeded										*/
volatile int T = 2548;                      /* Used to find trough in pulse wave, seeded									*/
volatile int thresh = 2548;                 /* Used to find instant moment of heart beat, seeded  				*/
volatile int amp = 100;                     /* Used to hold amplitude of pulse waveform, seeded   				*/
volatile int firstBeat = 1;                 /* Used to seed rate array so we startup with reasonable BPM  */
volatile int secondBeat = 0;                /* Used to seed rate array so we startup with reasonable BPM  */

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/* Saved clock source */
static volatile CHIP_SYSCON_MAINCLKSRC_T saved_clksrc;
/* Get 32 samples for 3 channels CH0, CH3 and CH4 */
#define NUM_SAMPLES_A    1
#define NUM_CHANNELS_A    1
#define SAMPLE_COUNT_A    1 /* Run 1 time */

/* SEQ_A enables channels 0, 3 and 4; Uses software trigger; doesn't use BURST */
#define ADC_SEQ_A_CONFIG  \
	TRIG_SOFT |		/* Software trigger for SEQ_A */ \
	TRIG_POL_POS |	/* UM recommends this for S/W trigger */ \
	MODE_EOS |		/* Event generated after Sequence done */ \
	ENABLE_CH(3)	/* Associate channels 0, 2 and 3 to SEQ_A */

/* General ADC configuration */
#define ADC_CONFIG \
	MODE_SYNC |	/* Enable synchronous mode */ \
	RESOL_12BIT |	/* Use 12-Bit resolution */	\
	SAMPLE_TIME(0)	/* No extra clocks */

/* Buffer pointers */
static uint16_t buff_A[NUM_SAMPLES_A][NUM_CHANNELS_A];	/* Memory to hold samples for SEQ_A */
static uint16_t(*volatile pBufA)[NUM_CHANNELS_A];	/* Pointer to current active SEQ_A buffer */

/* ADC Driver context memory */
#define RAMBLOCK_H          60
static uint32_t  start_of_ram_block0[RAMBLOCK_H];

/* ADC ROM Driver Handle */
static ADC_HANDLE_T *hADC;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/


/*****************************************************************************
 * Private functions
 ****************************************************************************/

STATIC const PINMUX_GRP_T pinmuxing_lowpower[] = {
	//{0, 0,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /* UART0 RX */
	//{0, 1,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /* UART0 TX */
	{0, 2,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)},
	{0, 3,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)},
	{0, 4,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)},
	{0, 5,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)},
	{0, 6,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)},
	{0, 7,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)},
	{0, 8,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 9,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 10,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /* */
	{0, 11,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 12,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 13,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 14,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 15,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	//{0, 16,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  swd */
	//{0, 17,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  swd */
	{0, 18,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 19,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 20,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 21,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 22,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 23,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 24,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 25,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 26,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 27,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{0, 28,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{1, 9,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)},  /*  */
	{1, 10,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{1, 11,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{1, 12,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{1, 13,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{1, 14,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{1, 15,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{1, 16,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
	{1, 17,  (IOCON_FUNC0 | IOCON_MODE_INACT | IOCON_DIGITAL_EN)}, /*  */
};

void SetupUnusedPins(void)
{
Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 1);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 1, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 2);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 2, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 3);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 3, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 4);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 4, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 5);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 5, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 6);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 6, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 7);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 7, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 8);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 8, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 9);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 9, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 10);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 10, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 11);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 11, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 12);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 12, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 13);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 13, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 14);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 14, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 15);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 15, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 18);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 18, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 19);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 19, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 20);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 20, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 21);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 21, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 22);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 22, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 23);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 23, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 24);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 24, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 25);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 25, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 26);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 26, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 27);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 27, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 28);
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 28, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 1, 9);
	Chip_GPIO_SetPinState(LPC_GPIO, 1, 9, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 1, 10);
	Chip_GPIO_SetPinState(LPC_GPIO, 1, 10, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 1, 11);
	Chip_GPIO_SetPinState(LPC_GPIO, 1, 11, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 1, 12);
	Chip_GPIO_SetPinState(LPC_GPIO, 1, 12, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 1, 13);
	Chip_GPIO_SetPinState(LPC_GPIO, 1, 13, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 1, 14);
	Chip_GPIO_SetPinState(LPC_GPIO, 1, 14, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 1, 15);
	Chip_GPIO_SetPinState(LPC_GPIO, 1, 15, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 1, 16);
	Chip_GPIO_SetPinState(LPC_GPIO, 1, 16, 0);
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 1, 17);
	Chip_GPIO_SetPinState(LPC_GPIO, 1, 17, 0);
}

void ConfigureUnusedPins(void)
{
	Chip_Clock_EnablePeriphClock(SYSCON_CLOCK_IOCON);
	Chip_Clock_EnablePeriphClock(SYSCON_CLOCK_GPIO0);
	Chip_Clock_EnablePeriphClock(SYSCON_CLOCK_GPIO1);
	Chip_IOCON_SetPinMuxing(LPC_IOCON, pinmuxing_lowpower, sizeof(pinmuxing_lowpower) / sizeof(PINMUX_GRP_T));
	SetupUnusedPins();
}

void RTC_Init(void)
{
	/* Turn on the RTC 32K Oscillator */
	Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_32K_OSC);

	/* Enable the RTC oscillator, oscillator rate can be determined by
	   calling Chip_Clock_GetRTCOscRate()	*/
	Chip_Clock_EnableRTCOsc();

	/* Initialize RTC driver (enables RTC clocking) */
	Chip_RTC_Init(LPC_RTC);

	/* Enable RTC as a peripheral wakeup event */
	Chip_SYSCON_EnableWakeup(SYSCON_STARTER_RTC);

	/* RTC reset */
	Chip_RTC_Reset(LPC_RTC);

	/* Start RTC at a count of 0 when RTC is disabled. If the RTC is enabled, you
	   need to disable it before setting the initial RTC count. */
	Chip_RTC_Disable(LPC_RTC);
	Chip_RTC_SetCount(LPC_RTC, 0);

	/* Enable RTC and high resolution timer - this can be done in a single
	   call with Chip_RTC_EnableOptions(LPC_RTC, (RTC_CTRL_RTC1KHZ_EN | RTC_CTRL_RTC_EN)); */
	Chip_RTC_Enable1KHZ(LPC_RTC);
	Chip_RTC_Enable(LPC_RTC);

	/* Clear latched RTC interrupt statuses */
	Chip_RTC_ClearStatus(LPC_RTC, (RTC_CTRL_OFD | RTC_CTRL_WAKE1KHZ));

	/* Enable RTC interrupt */
	NVIC_EnableIRQ(RTC_IRQn);

	/* save the clock source, power down the PLL */
	saved_clksrc = Chip_Clock_GetMainClockSource();

	/* Enable RTC interrupt */
	Chip_RTC_EnableWakeup(LPC_RTC, RTC_CTRL_WAKEDPD_EN);
	Chip_SYSCON_EnableWakeup(SYSCON_STARTER_RTC);

	/* Turn on the RED LED */
	Board_LED_Set(0, true);
}

void RTC_LowPowerSleep(void)
{
	/* Disable PLL, if previously enabled, prior to sleep */
	if (saved_clksrc == SYSCON_MAINCLKSRC_PLLOUT) {
		Chip_Clock_SetMainClockSource(SYSCON_MAINCLKSRC_IRC);
		Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SYS_PLL);
	}

	Chip_POWER_SetVoltage(POWER_LOW_POWER_MODE, Chip_Clock_GetMainClockRate());

	/* Before going to sleep or power down,
		make sure that the UART has completed transmission */
	while ((LPC_USART0->STAT & UART_TXIDLE) == 0) ;

	/* Go to sleep leaving SRAM and the 32K oscillator powered during sleep.
		Use lower voltage during sleep. */
	Chip_POWER_EnterPowerMode(PDOWNMODE, (SYSCON_PDRUNCFG_PD_SRAM1 | SYSCON_PDRUNCFG_PD_32K_OSC | SYSCON_PDRUNCFG_PD_VDDA_ENA));

	/* On wakeup, restore PLL power if needed */
	if (saved_clksrc == SYSCON_MAINCLKSRC_PLLOUT) {
		Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_SYS_PLL);

		/* Wait for PLL lock */
		while (!Chip_Clock_IsSystemPLLLocked());

		Chip_POWER_SetVoltage(POWER_LOW_POWER_MODE, Chip_Clock_GetSystemPLLOutClockRate(false));

		/* Use PLL for system clock */
		Chip_Clock_SetMainClockSource(SYSCON_MAINCLKSRC_PLLOUT);
	}
}

void Compute_Heartrate(void)
{
	int i, N, Signal, runningTotal, current_sample = 0;

	while(current_sample < SAMPLE_FREQUENCY/2){
		Signal = temp_data[current_sample];

		/* Keep track of time in milliseconds with sampleCounter variable */
		sampleCounter = sampleCounter + 50;

		/* Monitor the time since last beat to avoid noise */
		N = sampleCounter - lastBeatTime;

		/* Find the peak and trough of the pulse wave, avoid dichrotic noise by waiting 3/5 of last IBI */
		if(Signal < thresh && N > (IBI/5)*3){
			if (Signal < T){
				/* Keep track of lowest point in pulse wave in the T variable */
				T = Signal;
			}
		}

		/* Use threshold condition to filter out noise, store peak in P */
		if(Signal > thresh && Signal > P){
			P = Signal;
		}

		/* Analyze the data to find heartbeat */
		if (N > 500){
			/* Avoid high frequency noise */
			if ( (Signal > thresh) && (Pulse == 0) && (N > (IBI/5)*3) ){
				Pulse = 1;
				IBI = sampleCounter - lastBeatTime;
				lastBeatTime = sampleCounter;

				if(secondBeat){
					secondBeat = 0;
					for(i=0; i<=9; i++){
						rate[i] = IBI;
					}
				}
				if(firstBeat){
					firstBeat = 0;
					secondBeat = 1;
					continue;
				}

				/* Keep a running total of the last 10 IBI values */
				runningTotal = 0;

				for(i=0; i<=8; i++){
					rate[i] = rate[i+1];
					runningTotal += rate[i];
				}

				/* Average the latest IBI values and calculate the BPM */
				rate[9] = IBI;
				runningTotal += rate[9];
				runningTotal /= 10;
				BPM = 60000/runningTotal;
				QS = 1;
			}
		}
		/* Once the beat is over, reset values */
		if (Signal < thresh && Pulse == 1){
			Pulse = 0;
			amp = P - T;
			thresh = amp/2 + T;
			P = thresh;
			T = thresh;
		}
		/* If we do not detect a heart beat in 2.5 seconds, reset all values */
		if (N > 2500){
			thresh = 2548;
			P = 2548;
			T = 2548;
			lastBeatTime = sampleCounter;
			firstBeat = 1;
			secondBeat = 0;
		}
		current_sample++;
	}
}

void RTC_IRQHandler(void)
{
	uint32_t rtcStatus;
	/* Get RTC status register */
	rtcStatus = Chip_RTC_GetStatus(LPC_RTC);

	/* Set RTC timer */
	Chip_RTC_SetWake(LPC_RTC, 50);

	/* Clear only latched RTC status */
	Chip_RTC_ClearStatus(LPC_RTC, (rtcStatus & RTC_CTRL_WAKE1KHZ));
}

static void ADC_PinMuxSetup(void)
{
#if defined(BOARD_NXP_LPCXPRESSO_54102)
	/* All pins to inactive, neither pull-up nor pull-down. */
	Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 0, IOCON_MODE_INACT | IOCON_FUNC1 | IOCON_ANALOG_EN);
#endif
}

/* Initialize the ADC ROM Driver */
static int adcrom_init(void)
{
	volatile int size_in_bytes;

	ADC_PinMuxSetup();
	Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_ADC0 | SYSCON_PDRUNCFG_PD_VDDA_ENA | SYSCON_PDRUNCFG_PD_VREFP);
	Chip_Clock_EnablePeriphClock(SYSCON_CLOCK_ADC0);

	Chip_Clock_SetADCClockSource(SYSCON_ADCCLKSELSRC_MAINCLK);
	Chip_Clock_SetADCClockDiv(0x1);

	size_in_bytes =  ROM_ADC_GetMemSize();

	if (RAMBLOCK_H < (size_in_bytes / 4)) {
		return 1;
	}

	hADC = ROM_ADC_Init(start_of_ram_block0, LPC_ADC_BASE, 0);

	return 0;
}

/* Configure ADC ROM Driver and pripheral */
static int adcrom_config(void)
{
	ADC_CFG_T cfg = {
		ADC_SEQ_A_CONFIG,
		0,
		ADC_CONFIG,
		0	/* Divider will be calculated during run time */
	};

	cfg.clkDiv = 0xFF;

	/* Configure the ADC */
	ROM_ADC_Configure(hADC, &cfg);

	/* Calibrate the ADC */
	if (ROM_ADC_Calibrate(hADC, Chip_Clock_GetSystemClockRate()) != LPC_OK) {
		DEBUGSTR("ERROR: Calibrating ADC\r\n");
		while (1) {}
	}
	DEBUGSTR("ADC Initialized and Calibrated successfully!\r\n");

	/* Channel configurations */
	ROM_ADC_ConfigureCh(hADC, 0, ADC_CH_THRES_DATA | ADC_CH_THRES_SEL1 | ADC_CH_THRES_CROSSING);
	ROM_ADC_ConfigureCh(hADC, 1, ADC_CH_THRES_DATA);

	return 0;
}

void configure_m0_master(){
	int temp;

	/* Configure M0 to be in control of entering low power modes */
	temp = (*(volatile int*)0x40000300 & 0xFFBF) | (0xc0c4 << 16);
	*(volatile int*)0x40000300 = temp;
}
/*****************************************************************************
 * Public functions
 ****************************************************************************/

/**
 * @brief	main routine for M0 core heart-rate example
 * @return	Function should not exit.
 */
int main(void)
{
	int infinity = 1, sample_num = 0;

	/* Switch main system clock to IRC and power down PLL if needed */
	saved_clksrc = Chip_Clock_GetMainClockSource();
	if (saved_clksrc == SYSCON_MAINCLKSRC_PLLOUT) {
		if(Chip_Clock_GetSystemClockRate() <= 12000000){
			Chip_Clock_SetAsyncSysconClockSource(SYSCON_ASYNC_IRC);
			Chip_Clock_SetMainClockSource(SYSCON_MAINCLKSRC_IRC);
			Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SYS_PLL);
		}
	}
	SystemCoreClockUpdate();

	/* Initialize UART for this core */
	DEBUGINIT();

	/* Configure M0+ core as master and in control of reduced power modes */
	configure_m0_master();

	/* Configure pins to minimize power consumption */
	ConfigureUnusedPins();

	/* Initialize RTC */
	RTC_Init();

	/* Initialize the PinMux and setup the memory for ROM driver */
	if (adcrom_init()) {
		return 1;
	}

	/* Configure the ADC */
	if (adcrom_config()) {
		return 1;
	}

	Chip_RTC_SetWake(LPC_RTC, 50);

	while (infinity) {
		/* Enter POWERDOWN mode with the ADC OFF */
		RTC_LowPowerSleep();

		/* Re-calibrate the ADC upon wake up */
		if (ROM_ADC_Calibrate(hADC, Chip_Clock_GetSystemClockRate()) != LPC_OK) {
			DEBUGSTR("ERROR: Calibrating ADC\r\n");
			while (1) {}
		}

		/* Sample the ADC */
		if (ROM_ADC_StartConversion(hADC, ADC_SEQ_A, &buff_A[0][0], NUM_SAMPLES_A * NUM_CHANNELS_A) != LPC_OK) {
			DEBUGSTR("ERROR: Starting conversion in SEQ_A\r\n");
		}

		/* Wait for buffer to be filled up with required number of samples */
		ROM_ADC_Handler(hADC, ADC_EV_SEQ_A_POLL);

		/* Store current sample */
		temp_data[sample_num] = buff_A[0][0];
		sample_num++;

		/* If we have half a seconds worth of samples, check if there was a heartbeat */
		if(sample_num >= SAMPLE_FREQUENCY/2){

			/* Execute heart rate algorithm to see if there's a heartbeat! */
			Compute_Heartrate();

			/* Print out approximate BPM if heart rate was detected */
			if( QS == 1){
				DEBUGOUT("Found heartbeat. BPM: %i\r\n\r\n", BPM);
				QS = 0;
				/* Reset heart rate variables if BPM is ridiculous in value */
				if(BPM > 100 || BPM < 30){
					P = 2548;
					T = 2548;
					thresh = 2548;
					amp = 100;
					firstBeat = 1;
					secondBeat = 0;
				}
			}
		sample_num = 0;
		}

	}
	return 0;
}
