/****************************************************************************
 *   $Id:: main.c 8974 2014-03-23					$
 *   Project: NXP LPC8xx Capacitive Touch example
 *
 *   Description:
 *     This file contains the Main Loop and some initializations
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.

 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors'
 * relevant copyright in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers. This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
****************************************************************************/

#include "chip.h"

#include "lpc8xx_comp.h"
#include "lpc8xx_gpio.h"

#define cRedState 1
#if	cRedState
#include "sct_fsm.h"
#else
#include "sct_ctouch.h"
#endif

#include "main.h"

/******************************************
 * SCT Interrupt Handler
 *
 * The following interrupts are used
 * SCT_IRQ_EVENT_Interval_timer_IRQ (0) - defines the sampling period
 *******************************************/

#if 1
enum {	eLed1 = 13,	/* work set	*/
	eLed2 = 12,
	eLed3 = 11,
	eLed4 = 10 } ;
#else
enum {	eLed1 = 13,	/* test set	*/
	eLed2 = 11,
	eLed3 = 12,
	eLed4 = 10 } ;
#endif

enum {	eTimTest = 4 } ;

enum {  eNop  = 0, eOn, eOff,	/* Led toggle states	*/
	eLow  = 0,
	eHigh = 1 } ;

CapTouch_t CapTouch[eBtnMax] ;

//#define ISR_TIME(x, y)
  #define ISR_TIME(x, y)	Chip_GPIO_SetPinState(LPC_GPIO_PORT, 0, x, y)

void SCT_IRQHandler (void)
{
  ISR_TIME(eTimTest, eLow) ;		/* on timing measurement	*/

  static uint32_t button = 0 ;		/* currently sampled button (either button 0 or button 1) */

  static CapTouch_t *p = CapTouch ;

  uint32_t status = LPC_SCT->EVFLAG ;
  if	  (status & 0x00000001) {		/* SCT_IRQ_EVENT_Interval_timer_IRQ	*/

    uint32_t led_x = eNop ;
    uint32_t led_t = false ;
    uint32_t width = sct_fsm_CAPTURE_Oscillator_period ;

    if	    (width < p->PressDat) {	 	/* button not pressed		*/
      if    (width > p->Max) p->Max = width ;	/* record the maximum value	*/

	  p->Count++ ;
      if (p->Count == 500) {
	  p->Count = 0 ;
	  p->PressDat = p->Max + 2 ;	/* set the button pressed value greater than	*/
	  p->Max = 0 ;					/* the Max value of unpressed	*/
      }
      if (p->PressCnt > 0) {		/* if not pressed, then decrement counter	*/
	  p->PressCnt-- ;

      } else {
	  p->State = eLow ;			/* button not pressed, or off	*/
	  led_x	   = eOff ;
      }
    }	else {					/* button pressed		*/

      p->PressCnt++ ;				/* button pressed, so increment counter	*/
      if (width < p->Min) p->Min = width ;

      if (p->PressCnt >= 10) {		/* if we have received 10 samples above the val_button_press1 value */
	  p->PressCnt  = 10 ;			/* limit the maximum number of samples to 10	*/
	  led_x = eOn ;

	if (p->State == eLow) {			/* rising edge detected */
	    p->State  = eHigh ;
	    led_t     = true ;			/* toggle LED		*/
	}
      }
    }

    switch (button) {
    case 0 :		/* change comparator input from button 0 to button 1 */
      if (led_x) Chip_GPIO_SetPinState(LPC_GPIO_PORT, 0, eLed1, led_x-1) ;	/* on/off LED0		*/
      if (led_t) Chip_GPIO_SetPortToggle(LPC_GPIO_PORT, 0, 1 << eLed2) ;	/* toggle LED1		*/

      Chip_ACMP_SetNegVoltRef(LPC_CMP, ACMP_NEGIN_ACMP_I2) ;	/* COMP_VM -> ACMP_I2		*/
      break ;

    case 1 :		/* change comparator input from button 1 to button 0 */
      if (led_x) Chip_GPIO_SetPinState(LPC_GPIO_PORT, 0, eLed3, led_x-1) ;	/* on/off LED2		*/
      if (led_t) Chip_GPIO_SetPortToggle(LPC_GPIO_PORT, 0, 1 << eLed4) ;	/* toggle LED3		*/

      Chip_ACMP_SetNegVoltRef(LPC_CMP, ACMP_NEGIN_ACMP_I1) ;	/* COMP_VM -> ACMP_I1		*/
      break ;
    }

    if  (++button >= eBtnMax) {
	   button  = eBtn0 ;
	   p = CapTouch ;
    } else p++ ;

    LPC_SCT->STATE_L = 0 ;		/* make sure that the LOW counter starts in STATE 0	*/
    uint16_t tmp  = LPC_SCT->CTRL_L ;
#if cRedState
    tmp &= ~(1 << 2) ;		/* un-halt counter_L	*/
    tmp |=  (1 << 1) ;		/*    stop counter_L	*/
#else
    tmp &= ~eSct_Halt ;		/* un-halt counter_L	*/
    tmp |=  eSct_Stop ;		/*    stop counter_L	*/
#endif
    LPC_SCT->CTRL_L = tmp ;	/* The STOP and HALT bit need to be updated in a single command */
  }
    LPC_SCT->EVFLAG = status ;		/* Acknowledge interrupts */

    ISR_TIME(eTimTest, eHigh) ;		/* off timing measurement	*/
}

/******************************************************************************
**   Main Function  main()
******************************************************************************/

int main (void)
{
  SystemCoreClockUpdate() ;

  GPIOInit() ;

  COMP_Init() ;

	/* Enable AHB clock to the SCT. */
  LPC_SYSCTL->SYSAHBCLKCTRL |= (0x1 << 8) ;

#if cRedState
  sct_fsm_init() ;
#else
  sct_ctouch_init() ;
#endif

  for (CapTouch_t *p =  CapTouch,
		  *q = &p[eBtnMax] ; p < q ; p++) {
    p->Min	= eMin ;
    p->Max	= eMax ;
    p->PressDat	= eIdle ;
    p->PressCnt	= 0 ;
    p->Count	= 0 ;
    p->State	= eLow ;
  }

#if cRedState
  LPC_SCT->EVFLAG = 0x3F ; 		/* clear all SCT interrupt flags	*/
#else
  LPC_SCT->EVFLAG = bEvts ; 		/* clear all SCT interrupt flags	*/
#endif
  NVIC_EnableIRQ(SCT_IRQn) ; 		/* Enable SCT interrupt			*/

  uint32_t tmp = LPC_SCT->CTRL_U ;
#if cRedState
  tmp &= ~((1 << 2+16) |		/* un-halt counter_H	*/
  	   (1 << 2)) ;			/* un-halt counter_L	*/
  tmp |=   (1 << 1) ;			/*    stop counter_L	*/
#else
  tmp &= ~(eSct_HaltL | eSct_HaltH) ;	/* un-halt counter_L,H	*/
  tmp |=   eSct_StopL ;			/*    stop counter_L	*/
#endif
  LPC_SCT->CTRL_U = tmp ;	/* The STOP and HALT bit need to be updated in a single command */

  while ( 1 )  {
    __WFI() ;
  } ;
}

/******************************************************************************
**                            End Of File
******************************************************************************/

