/****************************************************************************
 *   $Id:: iap_test.c 7171 2011-04-26 20:23:50Z nxp28548                      $
 *   Project: NXP LPC11Axx IAP example
 *
 *   Description:
 *     This example demostrate how to use the IAP for the LPC11Axx
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/

#include "LPC11Axx.h"
#include "uart.h"
#include <stdio.h>
#include <string.h>
#include "iap.h"
#include "gpio.h"

void menu(void);
void CopyToSRAM(void);
void CopyInterruptToSRAM(void);
void WriteToFlash(void);
void ClearStringBuffer(void);

#define MENU_SIZE 7

uint8_t* menu_array[MENU_SIZE] = {
(uint8_t*)"\r\nLPC11Axx IAP Test\r\n",
(uint8_t*)"'r' - Read Part ID\r\n",
(uint8_t*)"'b' - Read Boot Code Version\r\n",	
(uint8_t*)"'s' - Read Serial Number\r\n",
(uint8_t*)"'f' - Fill Flash Sector 1 - 7 with 0xAA\r\n",
(uint8_t*)"'0' - Erase Flash Sector 1 - 7\r\n",
(uint8_t*)"'i' - Invoke ISP\r\n"	
};

const char ascii[] = "0123456789ABCDEF";

#define SYSTICK_DELAY		(SystemCoreClock/1000)
#define TEST_INTERRUPT 1

typedef enum {
	CMD_READ_ID = 'r',
	CMD_READ_BOOT = 'b',
	CMD_READ_SERIAL = 's',
	CMD_AA = 'f',
	CMD_00 = '0',
	CMD_INVOKE_ISP = 'i'
}commandEnum;

char lineBuff[80], valBuff[80];
IAP_STATUS_CODE iap_ret;

int main (void)
{
	int i;
	uint32_t *partID, iapTemp, *serialPtr, serialTemp[4];
	uint8_t *bootFirstByte, *bootSecondByte, bootTemp, bootTemp1;
	static char cmd;
	iap_ret = CMD_SUCCESS;
	
	SystemCoreClockUpdate();
	
#if TEST_INTERRUPT
	CopyInterruptToSRAM();		//remap interrupt vector to SRAM
	LPC_SYSCON->SYSMEMREMAP = 0x1;	//change memory map
	
	//Setup PIO0_16 as GPIO to toggle LED
	LPC_IOCON->PIO0_16 &= ~(3<<3);
	LPC_GPIO_PORT->DIR[0] |= (1<<16);
	LPC_GPIO_PORT->SET[0] = 1<<16;
	//Enable GPIO domain
	LPC_SYSCON->SYSAHBCLKCTRL |= (1<<6);
	/* Called for system library in core_cmx.h(x=0 or 3). */
  SysTick_Config( SYSTICK_DELAY );
#endif
	
	UARTInit(9600);
	NVIC_DisableIRQ(USART_IRQn);

	menu();
	while(1)
	{
		ClearStringBuffer();
		cmd = get_key();
		print_string((uint8_t*)"\r\n");
		switch(cmd)
		{
			case CMD_READ_ID:
				partID = &iapTemp;
				iap_ret = ReadPartID(partID);
				if (iap_ret == CMD_SUCCESS)
				{
					sprintf(lineBuff, "Part ID is %c%c%c%c%c%c%c%c\r\n", ascii[((uint8_t)(iapTemp>>24))>>4],ascii[((uint8_t)(iapTemp>>24))& 0x0F],
					ascii[((uint8_t)(iapTemp>>16))>>4], ascii[((uint8_t)(iapTemp>>16))& 0x0F],
					ascii[((uint8_t)(iapTemp>>8))>>4], ascii[((uint8_t)(iapTemp>>8))& 0x0F],
					ascii[((uint8_t)iapTemp>>4)], ascii[((uint8_t)iapTemp& 0x0F)]);
					print_string((uint8_t*) lineBuff);
				}
				break;
			case CMD_READ_BOOT:
				bootFirstByte = &bootTemp;
				bootSecondByte = &bootTemp1;
				iap_ret = ReadBootCodeVer(bootFirstByte, bootSecondByte);
				if (iap_ret == CMD_SUCCESS)
				{
					sprintf(lineBuff, "Part boot code # is %c.%c\r\n", ascii[bootTemp], ascii[bootTemp1]);
					print_string((uint8_t*) lineBuff);
				}
				else
				{
					print_string((uint8_t*)"Read Boot Version Failed!!\r\n");
				}
				break;
			
			case CMD_READ_SERIAL:
				serialPtr = &serialTemp[0];
				iap_ret = ReadDeviceSerialNum(serialPtr);
				if (iap_ret == CMD_SUCCESS)
				{
					sprintf(lineBuff, "");
					
					for (i =0; i<4; i++)
					{
							sprintf(valBuff, "%c%c%c%c%c%c%c%c ",ascii[((uint8_t)(serialTemp[i]>>24))>>4],ascii[((uint8_t)(serialTemp[i]>>24))& 0x0F],
							ascii[((uint8_t)(serialTemp[i]>>16))>>4], ascii[((uint8_t)(serialTemp[i]>>16))& 0x0F],
							ascii[((uint8_t)(serialTemp[i]>>8))>>4], ascii[((uint8_t)(serialTemp[i]>>8))& 0x0F],
							ascii[((uint8_t)serialTemp[i]>>4)], ascii[((uint8_t)serialTemp[i]& 0x0F)]);
							strcat(lineBuff, valBuff);
					}
					strcat(lineBuff, "\r\n");
					sprintf(valBuff, "Serial # is ");
					strcat(valBuff,lineBuff);
					print_string((uint8_t*) valBuff);
				}
				else
				{
					print_string((uint8_t*)"Read serial number failed\r\n");
				}
				break;
			case CMD_AA:
				CopyToSRAM();		//load RAM with fill pattern of 0xAA
				iap_ret = EraseSector(2, 7);
				if (iap_ret == CMD_SUCCESS)
				{
					WriteToFlash();
					print_string((uint8_t*)"Fill flash successful\r\n");
				}
				else
				{
					print_string((uint8_t*)"Fill flash failed\r\n");
				}					
				break;
			
			case CMD_00:
				iap_ret = EraseSector(2,7);
				if (iap_ret == CMD_SUCCESS)
				{
					print_string((uint8_t*)"Erase completed\r\n");
				}
				else
				{
					print_string((uint8_t*)"Erase failed\r\n");
				}
				break;
			
			case CMD_INVOKE_ISP:
				print_string((uint8_t*)"Revoking ISP\r\n\r\n");
				print_string((uint8_t*)"Type ? to synchronize system\r\n\r\n");
				//  enable Timer 1
				LPC_SYSCON->SYSAHBCLKCTRL |= (1<<10);
				//	enable GPIO domain
				LPC_SYSCON->SYSAHBCLKCTRL |= (1<<6);
				//	enable IO configuration clock
				LPC_SYSCON->SYSAHBCLKCTRL |= (1<<16);
				//	set AHB clock divider is 1:1
				LPC_SYSCON->SYSAHBCLKDIV = 1;
				//	set the stack pointer to ROM value (reset default)
				__set_MSP(*((uint32_t *)0x1FFF0000));
				InvokeISP();
				break;
			default:
				print_string((uint8_t*)"Invalid command!\r\n\r\n");
			break;
		}
		menu();
	}
}

/*********************************************************************//**
 * @brief        Clear the char buffer to output to UART
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void ClearStringBuffer(void)
{
	int i;
	for (i =0; i<80; i++)
	{
		valBuff[i] = 0x00;
		lineBuff[i] = 0x00;
	}
}

/*********************************************************************//**
 * @brief        Display the menu through UART
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void menu()
{
	uint32_t i;
	for (i=0; i<MENU_SIZE; i++)
	{	
		print_string((uint8_t*) menu_array[i]);
	}
}


/*********************************************************************//**
 * @brief        Initiate IAP write to flash
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void WriteToFlash(void)
{
	int i;
	unsigned int flash_addr=0x2000;	//start of sector 2
	unsigned int ram_addr=0x10000800; //4kB of SRAM reserve for data storage
	
	for (i=1; i<=7; i++)
	{
		//The CopyRAM2Flash function will automatically prepare the sector
		iap_ret = CopyRAM2Flash((uint8_t*)flash_addr,(uint8_t*)ram_addr, IAP_WRITE_4096);
		flash_addr+=4096;
	}
}


/*********************************************************************//**
 * @brief        Copy the interrupt vector on the Flash to SRAM
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void CopyInterruptToSRAM(void)
{
	unsigned int * flashPtr, * ramPtr;
  unsigned int * uLimit = (unsigned int *) 0x200; 		
		
		ramPtr = (unsigned int *)0x10000000;	//load RAM starting at 0x10000000, 
		flashPtr = (unsigned int *)0x00;			//start of interrupt vector table
	  while(flashPtr < uLimit)
			{
				*ramPtr = *flashPtr;
				ramPtr++;
				flashPtr++; 
			}
}

/*********************************************************************//**
 * @brief        Fill SRAM with fill pattern to write to flash
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void CopyToSRAM(void)
{
		unsigned int * pointer;
    unsigned int * uLimit = (unsigned int *) 0x100017FF; 		
		
		pointer = (unsigned int *)0x10000800;	//load RAM starting at 0x10000800, 
	  while(pointer < uLimit)
			{
				*pointer=0xAAAAAAAA;
				 pointer++;
			}

}


