/**********************************************************************
* $Id$		lpc18xx_i2c.c		2012-08-31
*//**
* @file		lpc18xx_i2c.c
* @brief	Contains all functions support for I2C firmware library
* 			on LPC18xx
* @version	1.1
* @date		31. Aug. 2012
* @author	NXP MCU SW Application Team
*
* Copyright(C) 2012, NXP Semiconductor
* All rights reserved.
*
***********************************************************************
* Software that is described herein is for illustrative purposes only
* which provides customers with programming information regarding the
* products. This software is supplied "AS IS" without any warranties.
* NXP Semiconductors assumes no responsibility or liability for the
* use of the software, conveys no license or title under any patent,
* copyright, or mask work right to the product. NXP Semiconductors
* reserves the right to make changes in the software without
* notification. NXP Semiconductors also make no representation or
* warranty that such application will be suitable for the specified
* use without further testing or modification.
* Permission to use, copy, modify, and distribute this software and its
* documentation is hereby granted, under NXP Semiconductors'
* relevant copyright in the software, without fee, provided that it
* is used in conjunction with NXP Semiconductors microcontrollers.  This
* copyright, permission, and disclaimer notice must appear in all copies of
* this code.
**********************************************************************/

/* Peripheral group ----------------------------------------------------------- */
/** @addtogroup I2C
 * @{
 */

/* Includes ------------------------------------------------------------------- */
#include "lpc18xx_i2c.h"
#include "lpc18xx_cgu.h"
#include "lpc18xx_scu.h"

/* If this source file built with example, the LPC18xx FW library configuration
 * file in each example directory ("lpc18xx_libcfg.h") must be included,
 * otherwise the default FW library configuration file must be included instead
 */
#ifdef __BUILD_WITH_EXAMPLE__
#include "lpc18xx_libcfg.h"
#else
#include "lpc18xx_libcfg_default.h"
#endif /* __BUILD_WITH_EXAMPLE__ */

#ifdef _I2C


/* Private Types -------------------------------------------------------------- */
/** @defgroup I2C_Private_Types I2C Private Types
 * @{
 */
#define SFSP2_3_CONFIGURE_I2C1_SDA					(0x00000001 | MD_ZI | MD_EZI)
#define SFSP2_4_CONFIGURE_I2C1_SCL					(0x00000001 | MD_ZI | MD_EZI)

#define SFSPE_13_CONFIGURE_I2C1_SDA					(0x00000002 | MD_ZI | MD_EZI)
#define SFSPE_15_CONFIGURE_I2C1_SCL					(0x00000002 | MD_ZI | MD_EZI)

#define SFSI2C0_CONFIGURE_STANDARD_FAST_MODE		(1<<3 | 1<<11)
#define SFSI2C0_CONFIGURE_FASTPLUS_HIGHSPEED_MODE	(2<<1 | 1<<3 | 1<<7 | 1<<10 | 1<<11)
#define BLOCKING_TIMEOUT (0x000FFFFFUL)
#define RESTRANSMISSION_MAX (0x000000FFUL)

#define MAX_TX_BUFFER_SIZE 10 // Maximum transmit buffer size in I2C_MasterWriteReg() function


/**
 * @brief I2C device configuration structure type
 */
typedef struct
{
	union {
		I2C_M_SETUP_Type	txrx_setup_master; 						/* Transmission setup */
		I2C_S_SETUP_Type	txrx_setup_slave; 						/* Transmission setup */
	};
	int32_t		dir;								/* Current direction phase, 0 - write, 1 - read */
} I2C_CFG_T;

/**
 * @}
 */

/* Private Variables ---------------------------------------------------------- */
/**
 * @brief II2C driver data for I2C0, I2C1
 */
static I2C_CFG_T i2cdat[2];

static Bool I2C_MasterComplete[2];
static Bool I2C_SlaveComplete[2];

static uint32_t I2C_MonitorBufferIndex;

/* Private Functions ---------------------------------------------------------- */

/* Get I2C number */
static int32_t I2C_getNum(LPC_I2Cn_Type *I2Cx);

/* Generate a start condition on I2C bus (in master mode only) */
static uint32_t I2C_Start (LPC_I2Cn_Type *I2Cx, I2C_TRANSFER_OPT_Type Opt);

/* Generate a stop condition on I2C bus (in master mode only) */
static Status I2C_Stop (LPC_I2Cn_Type *I2Cx, I2C_TRANSFER_OPT_Type Opt);

/* I2C send byte subroutine */
static uint32_t I2C_SendByte (LPC_I2Cn_Type *I2Cx, uint8_t databyte);

/* I2C get byte subroutine */
static uint32_t I2C_GetByte (LPC_I2Cn_Type *I2Cx, uint8_t *retdat, Bool ack);

/*--------------------------------------------------------------------------------*/
/********************************************************************//**
 * @brief		Convert from I2C peripheral to number
 * @param[in]	I2Cx I2C peripheral selected, should be:
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @return 		I2C number or error code, could be:
 * 					- 0		:I2C0
 * 					- 1		:I2C1
 * 					- (-1)	:Error
 *********************************************************************/
static int32_t I2C_getNum(LPC_I2Cn_Type *I2Cx){
	if (I2Cx == LPC_I2C0) {
		return (0);
	} else if (I2Cx == LPC_I2C1) {
		return (1);
	}
	return (-1);
}


/********************************************************************//**
 * @brief		Generate a start condition on I2C bus (in master mode only)
 * @param[in]	I2Cx I2C peripheral selected, should be:
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	Opt	  a I2C_TRANSFER_OPT_Type type that selected for
 * 					interrupt or polling mode.
 * @return 		value of I2C status register after generate a start condition
 *********************************************************************/
static uint32_t I2C_Start (LPC_I2Cn_Type *I2Cx, I2C_TRANSFER_OPT_Type Opt)
{
	uint32_t cnt = 0;
	// Reset STA, STO, SI
	I2Cx->CONCLR = I2C_I2CONCLR_SIC|I2C_I2CONCLR_STOC|I2C_I2CONCLR_STAC;

	// Enter to Master Transmitter mode
	I2Cx->CONSET = I2C_I2CONSET_STA;

    if(Opt == I2C_TRANSFER_POLLING)
    {
		// Wait for complete
    	while (!(I2Cx->CONSET & I2C_I2CONSET_SI))
    	{
    		if(++cnt > BLOCKING_TIMEOUT){
    			return I2C_STAT_CODE_ERROR;
    		}
    	}
    }

	return (I2Cx->STAT & I2C_STAT_CODE_BITMASK);
}


/********************************************************************//**
 * @brief		Generate a stop condition on I2C bus (in master mode only)
 * @param[in]	I2Cx I2C peripheral selected, should be:
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	Opt	  a I2C_TRANSFER_OPT_Type type that selected for
 * 					interrupt or polling mode.
 * @return 		None
 *********************************************************************/
static Status I2C_Stop (LPC_I2Cn_Type *I2Cx, I2C_TRANSFER_OPT_Type Opt)
{
	uint32_t cnt = 0;
	/* Make sure start bit is not active */
	if (I2Cx->CONSET & I2C_I2CONSET_STA)
	{
		I2Cx->CONCLR = I2C_I2CONCLR_STAC;
	}

	I2Cx->CONSET = I2C_I2CONSET_STO;

	I2Cx->CONCLR = I2C_I2CONCLR_SIC;

    if(Opt == I2C_TRANSFER_POLLING)
    {
        // wait for stop is sent
        while(I2Cx->CONSET & I2C_I2CONSET_STO)
        {
            if(I2Cx->CONSET & I2C_I2CONSET_SI)
                I2Cx->CONCLR = I2C_I2CONCLR_SIC;
    		if(++cnt > BLOCKING_TIMEOUT){
    			return ERROR;
    		}
        }
    }

    return SUCCESS;
}

/********************************************************************//**
 * @brief		Send a byte
 * @param[in]	I2Cx I2C peripheral selected, should be:
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	databyte	sent data
 * @return 		value of I2C status register after sending
 *********************************************************************/
static uint32_t I2C_SendByte (LPC_I2Cn_Type *I2Cx, uint8_t databyte)
{
	uint32_t CodeStatus = I2Cx->STAT & I2C_STAT_CODE_BITMASK;

	if((CodeStatus != I2C_I2STAT_M_TX_START) &&
		(CodeStatus != I2C_I2STAT_M_TX_RESTART) &&
		(CodeStatus != I2C_I2STAT_M_TX_SLAW_ACK)  &&
		(CodeStatus != I2C_I2STAT_M_TX_DAT_ACK)  )
	{
		return CodeStatus;
	}

	I2Cx->DAT = databyte & I2C_I2DAT_BITMASK;

	I2Cx->CONSET = I2C_I2CONSET_AA;

	I2Cx->CONCLR = I2C_I2CONCLR_SIC;

	return (I2Cx->STAT & I2C_STAT_CODE_BITMASK);
}

/********************************************************************//**
 * @brief		Get a byte
 * @param[in]	I2Cx I2C peripheral selected, should be:
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[out]	retdat	pointer to return data
 * @param[in]	ack		assert acknowledge or not, should be: TRUE/FALSE
 * @return 		value of I2C status register after sending
 *********************************************************************/
static uint32_t I2C_GetByte (LPC_I2Cn_Type *I2Cx, uint8_t *retdat, Bool ack)
{
	*retdat = (uint8_t) (I2Cx->DAT & I2C_I2DAT_BITMASK);
	
	if (ack == TRUE)
	{
		I2Cx->CONSET = I2C_I2CONSET_AA;
	}
	else
	{
		I2Cx->CONCLR = I2C_I2CONCLR_AAC;
	}

	I2Cx->CONCLR = I2C_I2CONCLR_SIC;

	return (I2Cx->STAT & I2C_STAT_CODE_BITMASK);
}

/* End of Private Functions --------------------------------------------------- */


/* Public Functions ----------------------------------------------------------- */
/** @addtogroup I2C_Public_Functions
 * @{
 */

/********************************************************************//**
 * @brief		Initializes the I2Cx peripheral with specified parameter.
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	clockrate Target clock rate value to initialized I2C
 * 				peripheral (Hz)
 * @return 		None
 *********************************************************************/
void I2C_Init(LPC_I2Cn_Type *I2Cx, uint32_t clockrate)
{
	uint32_t tem;

	CHECK_PARAM(PARAM_I2Cx(I2Cx));

	if (I2Cx==LPC_I2C0)
	{
		/* Set up clock for I2C0 module */
		//LPC_CGU->BASE_VPB1_CLK = (SRC_PL160M_0<<24) | (1<<11);
		CGU_EntityConnect(CGU_CLKSRC_PLL1, CGU_BASE_APB1);
		/* Select weather standard, fast, fast plus mode*/
		if(clockrate>=1000000)// Fast mode plus: 1MHz, high speed 3.4MHz
			LPC_SCU->SFSI2C0 = SFSI2C0_CONFIGURE_FASTPLUS_HIGHSPEED_MODE;
		else				  // standard 100KHz, fast 400KHz
			LPC_SCU->SFSI2C0 = SFSI2C0_CONFIGURE_STANDARD_FAST_MODE;
	}
	else if (I2Cx==LPC_I2C1)
	{
		/* Set up clock for I2C1 module */
		//LPC_CGU->BASE_VPB3_CLK = (SRC_PL160M_0<<24) | (1<<11);
		CGU_EntityConnect(CGU_CLKSRC_PLL1, CGU_BASE_APB3);
		/* Configure pin function for I2C1*/
		LPC_SCU->SFSPE_13 = SFSPE_13_CONFIGURE_I2C1_SDA;			/* SDA */
		LPC_SCU->SFSPE_15 = SFSPE_15_CONFIGURE_I2C1_SCL;			/* SCL */
		/* Check if I2C1 run fast mode*/
  		if(clockrate > 400000)
  			return;
	}
	else {
		// Up-Support this device
		return;
	}

    /* Set clock rate */
	if(clockrate<1000)	//make sure SCLH,SCLL not exceed its 16bit value
		return;
	tem = CGU_GetPCLKFrequency(CGU_PERIPHERAL_M3CORE) / clockrate;
	I2Cx->SCLH = (uint32_t)(tem / 2);
	I2Cx->SCLL = (uint32_t)(tem - I2Cx->SCLH);
    /* Set I2C operation to default */
    I2Cx->CONCLR = (I2C_I2CONCLR_AAC |I2C_I2CONCLR_SIC | I2C_I2CONCLR_STAC | I2C_I2CONCLR_I2ENC);
}


/*********************************************************************//**
 * @brief		De-initializes the I2C peripheral registers to their
 *                  default reset values.
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @return 		None
 **********************************************************************/
void I2C_DeInit(LPC_I2Cn_Type* I2Cx)
{
	CHECK_PARAM(PARAM_I2Cx(I2Cx));

	/* Disable I2C control */
	I2Cx->CONCLR = I2C_I2CONCLR_I2ENC;

}

/*********************************************************************//**
 * @brief		Enable or disable I2C peripheral's operation
 * @param[in]	I2Cx I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	NewState New State of I2Cx peripheral's operation, should be:
 * 					- ENABLE	:enable I2C operation
 * 					- DISABLE	:disable I2C operation
 * @return 		none
 **********************************************************************/
void I2C_Cmd(LPC_I2Cn_Type* I2Cx, FunctionalState NewState)
{
	CHECK_PARAM(PARAM_FUNCTIONALSTATE(NewState));
	CHECK_PARAM(PARAM_I2Cx(I2Cx));

	if (NewState == ENABLE)
	{
		I2Cx->CONSET = I2C_I2CONSET_I2EN;
	}
	else
	{
		I2Cx->CONCLR = I2C_I2CONCLR_I2ENC;
	}
}


/*********************************************************************//**
 * @brief 		Enable/Disable interrupt for I2C peripheral
 * @param[in]	I2Cx	I2C peripheral selected, should be:
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	NewState	New State of I2C peripheral interrupt in NVIC core
 * 				should be:
 * 					- ENABLE: enable interrupt for this I2C peripheral
 * 					- DISABLE: disable interrupt for this I2C peripheral
 * @return 		None
 **********************************************************************/
void I2C_IntCmd (LPC_I2Cn_Type *I2Cx, Bool NewState)
{
	if (NewState)
	{
		if(I2Cx == LPC_I2C0)
		{
			NVIC_EnableIRQ(I2C0_IRQn);
		}
		else if (I2Cx == LPC_I2C1)
		{
			NVIC_EnableIRQ(I2C1_IRQn);
		}
	}
	else
	{
		if(I2Cx == LPC_I2C0)
		{
			NVIC_DisableIRQ(I2C0_IRQn);
		}
		else if (I2Cx == LPC_I2C1)
		{
			NVIC_DisableIRQ(I2C1_IRQn);
		}
	}
    return;
}

/*********************************************************************//**
 * @brief 		Handle I2C Master states.
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	CodeStatus	I2C state
 * @param[in]	TransferCfg	Pointer to a I2C_M_SETUP_Type structure that
 * 								contains specified information about the
 * 								configuration for master transfer.
 * @param[in]	Opt	I2C_TRANSFER_OPT_Type type that selected for
 * 								interrupt or polling mode.
 * @return 		Current state, should be:
 *				- I2C_OK
 *				- I2C_BYTE_RECV
 *				- I2C_BYTE_SENT
 *				- I2C_SEND_END
 *				- I2C_RECV_END
 *				- I2C_ERR
 *				- I2C_NAK_RECV
 **********************************************************************/
int32_t I2C_MasterHanleStates(LPC_I2Cn_Type* I2Cx, uint32_t CodeStatus, I2C_M_SETUP_Type *TransferCfg, I2C_TRANSFER_OPT_Type Opt)
{
	uint8_t *txdat;
	uint8_t *rxdat;
	uint8_t tmp;
	int32_t Ret = I2C_OK;

	//get buffer to send/receive
	txdat = (uint8_t *) &TransferCfg->tx_data[TransferCfg->tx_count];
	rxdat = (uint8_t *) &TransferCfg->rx_data[TransferCfg->rx_count];

	switch(CodeStatus)
	{
		case I2C_I2STAT_M_TX_START:
		case I2C_I2STAT_M_TX_RESTART:
		//case I2C_I2STAT_M_RX_START:
		//case I2C_I2STAT_M_RX_RESTART
			// Send data first
			if(TransferCfg->tx_count < TransferCfg->tx_length)
			{
				/* Send slave address + WR direction bit = 0 ----------------------------------- */
				I2C_SendByte(I2Cx, (TransferCfg->sl_addr7bit << 1));
				Ret = I2C_BYTE_SENT;
			}
			else if (TransferCfg->rx_count  < TransferCfg->rx_length)
			{
				/* Send slave address + RD direction bit = 1 ----------------------------------- */
				I2C_SendByte(I2Cx, ((TransferCfg->sl_addr7bit << 1) | 0x01));
				Ret = I2C_BYTE_SENT;
			}
            // Clear STA bit after the slave address is sent
            I2Cx->CONCLR = I2C_I2CONCLR_STAC;
			break;

		case I2C_I2STAT_M_TX_SLAW_ACK:
		case I2C_I2STAT_M_TX_DAT_ACK:

			if(TransferCfg->tx_count < TransferCfg->tx_length)
			{
				I2C_SendByte(I2Cx, *txdat);
				txdat++;
				TransferCfg->tx_count++;
				Ret = I2C_BYTE_SENT;
			}
			else
			{
				if(TransferCfg->rx_count >= TransferCfg->rx_length)
                {
                    I2C_Stop(I2Cx, Opt);
                }
				Ret = I2C_SEND_END;


			}
			break;
		case I2C_I2STAT_M_TX_DAT_NACK:
            if(TransferCfg->rx_count >= TransferCfg->rx_length)
            {
                I2C_Stop(I2Cx, Opt);
            }
			Ret = I2C_SEND_END;
			break;

		case I2C_I2STAT_M_RX_ARB_LOST:
		case I2C_I2STAT_S_RX_ARB_LOST_M_GENCALL:
		case I2C_I2STAT_S_TX_ARB_LOST_M_SLA:
        //case I2C_I2STAT_M_TX_ARB_LOST:
			I2C_Stop(I2Cx, Opt);
            Ret = I2C_ERR;
			break;

		case I2C_I2STAT_M_RX_SLAR_ACK:
            if(TransferCfg->rx_length > 1)
                I2Cx->CONSET = I2C_I2CONSET_AA;
            else
                I2Cx->CONCLR = I2C_I2CONCLR_AAC;
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;

			Ret = I2C_BYTE_RECV;
			break;

		case I2C_I2STAT_M_RX_DAT_ACK:
			if (TransferCfg->rx_count <TransferCfg->rx_length)
			{
				if ((TransferCfg->rx_length > 1) && (TransferCfg->rx_count < (TransferCfg->rx_length - 2)))
				{
					I2C_GetByte(I2Cx, &tmp, TRUE);
					Ret = I2C_BYTE_RECV;

				}
				else  // the next byte is the last byte, send NACK instead.
				 {
					I2C_GetByte(I2Cx, &tmp, FALSE);
					Ret = I2C_BYTE_RECV;
				 }
				*rxdat++ = tmp;

				TransferCfg->rx_count++;
			}
			else
			{
				I2C_Stop(I2Cx, Opt);
				Ret = I2C_RECV_END;
			}
			break;

		case I2C_I2STAT_M_RX_DAT_NACK:
			I2C_GetByte(I2Cx, &tmp, FALSE);
            if (TransferCfg->rx_count < TransferCfg->rx_length)
            {
                *rxdat++ = tmp;
                TransferCfg->rx_count++;
            }
			I2C_Stop(I2Cx, Opt);
			Ret = I2C_RECV_END;
			break;

		case I2C_I2STAT_M_RX_SLAR_NACK:
		case I2C_I2STAT_M_TX_SLAW_NACK:
		case I2C_I2STAT_BUS_ERROR:
			// Send STOP condition
			I2C_Stop(I2Cx, Opt);
			Ret = I2C_ERR;
			break;

        /* No status information */
		case I2C_I2STAT_NO_INF:
          if ((TransferCfg->tx_count <TransferCfg->tx_length)||
               (TransferCfg->rx_count <TransferCfg->rx_length))
          {
            I2C_Stop(I2Cx, Opt);
            Ret = I2C_ERR;
          }
          else
          {
            Ret = I2C_RECV_END;
          }
          break;

        default:
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			break;
	}

	return Ret;
}

/*********************************************************************//**
 * @brief 		Handle I2C Slave states.
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	CodeStatus	I2C state
 * @param[in]	TransferCfg	Pointer to a I2C_S_SETUP_Type structure that
 * 								contains specified information about the
 * 								configuration for slave transfer.
 * @return 		Current state, should be:
 *				- I2C_OK
 *				- I2C_BYTE_RECV
 *				- I2C_BYTE_SENT
 *				- I2C_SEND_END
 *				- I2C_RECV_END
 *				- I2C_ERR
 *				- I2C_NAK_RECV
 **********************************************************************/
int32_t I2C_SlaveHanleStates(LPC_I2Cn_Type* I2Cx, uint32_t CodeStatus, I2C_S_SETUP_Type *TransferCfg)
{
	int32_t Ret = I2C_OK;
	uint8_t *txdat;
	uint8_t *rxdat;

	//get buffer to send/receive
	txdat = (uint8_t *) &TransferCfg->tx_data[TransferCfg->tx_count];
	rxdat = (uint8_t *) &TransferCfg->rx_data[TransferCfg->rx_count];

	switch (CodeStatus)
	{
		/* Reading phase -------------------------------------------------------- */
		/* Own SLA+R has been received, ACK has been returned */
		case I2C_I2STAT_S_RX_SLAW_ACK:

		/* General call address has been received, ACK has been returned */
		case I2C_I2STAT_S_RX_GENCALL_ACK:
			I2Cx->CONSET = I2C_I2CONSET_AA;
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			break;
		/* Arbitration has been lost in Slave Address + R/W bit as bus Master. General Call has
			been received and ACK has been returned.*/
		case I2C_I2STAT_S_RX_ARB_LOST_M_GENCALL:
			I2Cx->CONSET = I2C_I2CONSET_AA|I2C_I2CONSET_STA;
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			break;
		/* Previously addressed with own SLA;
		 * DATA byte has been received;
		 * ACK has been returned */
		case I2C_I2STAT_S_RX_ARB_LOST_M_SLA:
		case I2C_I2STAT_S_RX_PRE_SLA_DAT_ACK:

			/*
			 * All data bytes that over-flow the specified receive
			 * data length, just ignore them.
			 */
			if ((TransferCfg->rx_count < TransferCfg->rx_length) && (TransferCfg->rx_data != NULL))
			{
				*rxdat++ = (uint8_t)I2Cx->DAT;

				TransferCfg->rx_count++;

				Ret = I2C_BYTE_RECV;
			}
			if(TransferCfg->rx_count == (TransferCfg->rx_length) ) {
				I2Cx->CONCLR = I2C_I2CONCLR_AAC|I2C_I2CONCLR_SIC;
				Ret = I2C_BYTE_RECV;
			}
			else {
				I2Cx->CONSET = I2C_I2CONSET_AA;
				I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			}

			break;
		/* DATA has been received, Only the first data byte will be received with ACK. Additional
				data will be received with NOT ACK. */
		case I2C_I2STAT_S_RX_PRE_GENCALL_DAT_ACK:
			if ((TransferCfg->rx_count < TransferCfg->rx_length) && (TransferCfg->rx_data != NULL))
			{
				*rxdat++ = (uint8_t)I2Cx->DAT;

				TransferCfg->rx_count++;

				Ret = I2C_BYTE_RECV;
			}
			I2Cx->CONCLR = I2C_I2CONCLR_AAC|I2C_I2CONCLR_SIC;
			break;

		/* Writing phase -------------------------------------------------------- */
		/* Own SLA+R has been received, ACK has been returned */
		case I2C_I2STAT_S_TX_SLAR_ACK:

		/* Data has been transmitted, ACK has been received */
		case I2C_I2STAT_S_TX_DAT_ACK:
			/*
			 * All data bytes that over-flow the specified receive
			 * data length, just ignore them.
			 */
			if ((TransferCfg->tx_count < TransferCfg->tx_length) && (TransferCfg->tx_data != NULL))
			{
				I2Cx->DAT = *txdat++;

				TransferCfg->tx_count++;

				Ret = I2C_BYTE_SENT;
			}

			I2Cx->CONSET = I2C_I2CONSET_AA;
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			break;
		/* Arbitration lost in Slave Address and R/W bit as bus Master. Own Slave Address + Read
				has been received, ACK has been returned. */
		case I2C_I2STAT_S_TX_ARB_LOST_M_SLA:
			if ((TransferCfg->tx_count < TransferCfg->tx_length) && (TransferCfg->tx_data != NULL))
			{
				I2Cx->DAT = *txdat++;

				TransferCfg->tx_count++;

				Ret = I2C_BYTE_SENT;
			}
			I2Cx->CONSET = I2C_I2CONSET_AA|I2C_I2CONSET_STA;
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			break;

		case I2C_I2STAT_S_TX_LAST_DAT_ACK:
		/* Data has been transmitted, NACK has been received,
		 * that means there's no more data to send, exit now */
		/*
		 * Note: Don't wait for stop event since in slave transmit mode,
		 * since there no proof lets us know when a stop signal has been received
		 * on slave side.
		 */
		case I2C_I2STAT_S_TX_DAT_NACK:
			I2Cx->CONSET = I2C_I2CONSET_AA;
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			Ret = I2C_SEND_END;
			break;

		/* Previously addressed with own SLA;
		 * DATA byte has been received;
		 * NOT ACK has been returned */
		case I2C_I2STAT_S_RX_PRE_SLA_DAT_NACK:

		/* DATA has been received, NOT ACK has been returned */
		case I2C_I2STAT_S_RX_PRE_GENCALL_DAT_NACK:
			I2Cx->CONSET = I2C_I2CONSET_AA;
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			Ret = I2C_RECV_END;
			break;

		/*
		 * Note that: Return code only let us know a stop condition mixed
		 * with a repeat start condition in the same code value.
		 * So we should provide a time-out. In case this is really a stop
		 * condition, this will return back after time out condition. Otherwise,
		 * next session that is slave receive data will be completed.
		 */

		/* A Stop or a repeat start condition */
		case I2C_I2STAT_S_RX_STA_STO_SLVREC_SLVTRX:
			I2Cx->CONSET = I2C_I2CONSET_AA;
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			Ret = I2C_STA_STO_RECV;
			break;

		/* No status information */
		case I2C_I2STAT_NO_INF:
		/* Other status must be captured */
		default:
			I2Cx->CONSET = I2C_I2CONSET_AA;
			I2Cx->CONCLR = I2C_I2CONCLR_SIC;
			break;

	}

	return Ret;
}

/*********************************************************************//**
 * @brief 		General Master Interrupt handler for I2C peripheral
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @return 		None
 **********************************************************************/
void I2C_Interrupt_MasterHandler(LPC_I2Cn_Type  *I2Cx)
{
	int32_t tmp;
	uint32_t returnCode;
	I2C_M_SETUP_Type *txrx_setup;
	int32_t Ret = I2C_OK;

	tmp = I2C_getNum(I2Cx);
	txrx_setup = (I2C_M_SETUP_Type *)&i2cdat[tmp].txrx_setup_master;

	while (!(I2Cx->CONSET & I2C_I2CONSET_SI));

	returnCode = (uint32_t)(I2Cx->STAT & I2C_STAT_CODE_BITMASK);

	// Save current status
	txrx_setup->status = returnCode;

	Ret = I2C_MasterHanleStates(I2Cx, returnCode, txrx_setup, I2C_TRANSFER_INTERRUPT);

	if(I2C_CheckError(Ret))
	{
		if(txrx_setup->retransmissions_count < txrx_setup->retransmissions_max)
		{
			// Retry
			txrx_setup->retransmissions_count ++;
			txrx_setup->tx_count = 0;
			txrx_setup->rx_count = 0;
			// Reset STA, STO, SI
	        I2C_Start(I2Cx, I2C_TRANSFER_INTERRUPT);
			return;
		}
		else
		{
			goto s_int_end;
		}
	}
	else if (Ret & I2C_SEND_END)
	{
		// If no need to wait for data from Slave
		if(txrx_setup->rx_count >= (txrx_setup->rx_length))
		{
			goto s_int_end;
		}
		else	// Start to wait for data from Slave
		{
			// Reset STA, STO, SI
			I2C_Start(I2Cx, I2C_TRANSFER_INTERRUPT);
			return;
		}
	}
	else if (Ret & I2C_RECV_END)
	{
		goto s_int_end;
	}
	else
	{
		return;
	}

s_int_end:

	// Disable interrupt
	I2C_IntCmd(I2Cx, FALSE);
	I2Cx->CONCLR = I2C_I2CONCLR_AAC | I2C_I2CONCLR_SIC | I2C_I2CONCLR_STAC;

	I2C_MasterComplete[tmp] = TRUE;
}

/*********************************************************************//**
 * @brief 		General Slave Interrupt handler for I2C peripheral
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @return 		None
 **********************************************************************/
void I2C_Interrupt_SlaveHandler (LPC_I2Cn_Type  *I2Cx)
{
	int32_t tmp;
	uint32_t returnCode;
	I2C_S_SETUP_Type *txrx_setup;
	int32_t Ret = I2C_OK;

	tmp = I2C_getNum(I2Cx);

	txrx_setup = (I2C_S_SETUP_Type *)&i2cdat[tmp].txrx_setup_slave;

	while (!(I2Cx->CONSET & I2C_I2CONSET_SI));

	returnCode = (uint32_t)(I2Cx->STAT & I2C_STAT_CODE_BITMASK);
	// Save current status
	txrx_setup->status = returnCode;

	Ret = I2C_SlaveHanleStates(I2Cx, returnCode, txrx_setup);

	if((I2C_CheckError(Ret)) || (Ret & I2C_STA_STO_RECV) || (Ret & I2C_SEND_END))
	{
		goto s_int_end;
	}
	else
	{
		return;
	}

s_int_end:

	// Disable interrupt
	I2C_IntCmd(I2Cx, FALSE);
	I2Cx->CONCLR = I2C_I2CONCLR_AAC | I2C_I2CONCLR_SIC | I2C_I2CONCLR_STAC;

	I2C_SlaveComplete[tmp] = TRUE;
}

/*********************************************************************//**
 * @brief 		Transmit and Receive data in master mode
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	TransferCfg		Pointer to a I2C_M_SETUP_Type structure that
 * 								contains specified information about the
 * 								configuration for master transfer.
 * @param[in]	Opt				a I2C_TRANSFER_OPT_Type type that selected for
 * 								interrupt or polling mode.
 * @return 		SUCCESS or ERROR
 *
 * Note:
 * - In case of using I2C to transmit data only, either transmit length set to 0
 * or transmit data pointer set to NULL.
 * - In case of using I2C to receive data only, either receive length set to 0
 * or receive data pointer set to NULL.
 * - In case of using I2C to transmit followed by receive data, transmit length,
 * transmit data pointer, receive length and receive data pointer should be set
 * corresponding.
 **********************************************************************/
Status I2C_MasterTransferData(LPC_I2Cn_Type* I2Cx, I2C_M_SETUP_Type *TransferCfg, I2C_TRANSFER_OPT_Type Opt)
{
	int32_t tmp;
	uint32_t CodeStatus;
	int32_t Ret = I2C_OK;

	// Reset I2C setup value to default state
	TransferCfg->tx_count = 0;
	TransferCfg->rx_count = 0;
	TransferCfg->status = 0;

	if (Opt == I2C_TRANSFER_POLLING)
	{
		/* First Start condition -------------------------------------------------------------- */
		TransferCfg->retransmissions_count = 0;
retry:
		// Reset I2C setup value to default state
		TransferCfg->tx_count = 0;
		TransferCfg->rx_count = 0;

		// Start command
		CodeStatus = I2C_Start(I2Cx, I2C_TRANSFER_POLLING);

		while(1)	// send data first and then receive data from Slave.
		{
			Ret = I2C_MasterHanleStates(I2Cx, CodeStatus, TransferCfg, I2C_TRANSFER_POLLING);
			if(I2C_CheckError(Ret))
			{
				TransferCfg->retransmissions_count++;
				if (TransferCfg->retransmissions_count > TransferCfg->retransmissions_max){
						// save status
						TransferCfg->status = CodeStatus | I2C_SETUP_STATUS_NOACKF;
						goto error;
					} else {
						goto retry;
					}
			}
			else if( (Ret & I2C_BYTE_SENT) ||
					(Ret & I2C_BYTE_RECV))
			{
				// Wait for sending ends/ Wait for next byte
				while (!(I2Cx->CONSET & I2C_I2CONSET_SI));
			}
			else if (Ret & I2C_SEND_END) // already send all data
			{
				// If no need to wait for data from Slave
				if(TransferCfg->rx_count >= (TransferCfg->rx_length))
				{
					break;
				}
				else
				{
					I2C_Start(I2Cx, I2C_TRANSFER_POLLING);
				}
			}
			else if (Ret & I2C_RECV_END) // already receive all data
			{
				break;
			}
			CodeStatus = I2Cx->STAT & I2C_STAT_CODE_BITMASK;
		}
		return SUCCESS;
error:
		return ERROR;
	}

	else if (Opt == I2C_TRANSFER_INTERRUPT)
	{
		tmp = I2C_getNum(I2Cx);
		I2C_MasterComplete[tmp] = FALSE;
		// Setup tx_rx data, callback and interrupt handler
		i2cdat[tmp].txrx_setup_master = *TransferCfg;

		// Set direction phase, write first
		i2cdat[tmp].dir = 0;

		/* First Start condition -------------------------------------------------------------- */
		// Reset STA, STO, SI
		I2Cx->CONCLR = I2C_I2CONCLR_SIC|I2C_I2CONCLR_STOC|I2C_I2CONCLR_STAC;
		I2Cx->CONSET = I2C_I2CONSET_STA;
		I2C_IntCmd(I2Cx, TRUE);

		return (SUCCESS);
	}

	return ERROR;
}

/*********************************************************************//**
 * @brief 		Receive and Transmit data in slave mode
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	TransferCfg		Pointer to a I2C_S_SETUP_Type structure that
 * 								contains specified information about the
 * 								configuration for master transfer.
 * @param[in]	Opt				I2C_TRANSFER_OPT_Type type that selected for
 * 								interrupt or polling mode.
 * @return 		SUCCESS or ERROR
 *
 * Note:
 * The mode of slave's operation depends on the command sent from master on
 * the I2C bus. If the master send a SLA+W command, this sub-routine will
 * use receive data length and receive data pointer. If the master send a SLA+R
 * command, this sub-routine will use transmit data length and transmit data
 * pointer.
 * If the master issue an repeat start command or a stop command, the slave will
 * enable an time out condition, during time out condition, if there's no activity
 * on I2C bus, the slave will exit, otherwise (i.e. the master send a SLA+R/W),
 * the slave then switch to relevant operation mode. The time out should be used
 * because the return status code can not show difference from stop and repeat
 * start command in slave operation.
 * In case of the expected data length from master is greater than data length
 * that slave can support:
 * - In case of reading operation (from master): slave will return I2C_I2DAT_IDLE_CHAR
 * value.
 * - In case of writing operation (from master): slave will ignore remain data from master.
 **********************************************************************/
Status I2C_SlaveTransferData(LPC_I2Cn_Type* I2Cx, I2C_S_SETUP_Type *TransferCfg, I2C_TRANSFER_OPT_Type Opt)
{
	int32_t tmp;
	int32_t Ret = I2C_OK;
	uint32_t CodeStatus = 0;

	// Reset I2C setup value to default state
	TransferCfg->tx_count = 0;
	TransferCfg->rx_count = 0;
	TransferCfg->status = 0;

	// Polling option
	if (Opt == I2C_TRANSFER_POLLING)
	{
		/* Set AA bit to ACK command on I2C bus */
		I2Cx->CONSET = I2C_I2CONSET_AA;

		/* Clear SI bit to be ready ... */
		I2Cx->CONCLR = (I2C_I2CONCLR_SIC | I2C_I2CONCLR_STAC|I2C_I2CONCLR_STOC);

		while (1)
		{
			/* Check SI flag ready */
			if (I2Cx->CONSET & I2C_I2CONSET_SI)
			{
				CodeStatus = (I2Cx->STAT & I2C_STAT_CODE_BITMASK);

				Ret = I2C_SlaveHanleStates(I2Cx, CodeStatus, TransferCfg);
				if(I2C_CheckError(Ret))
				{
					goto s_error;
				}
				else if((Ret & I2C_STA_STO_RECV) || (Ret & I2C_SEND_END))
				{
					goto s_end_stage;
				}
			}
		}

s_end_stage:
		/* Clear AA bit to disable ACK on I2C bus */
		I2Cx->CONCLR = I2C_I2CONCLR_AAC;

		// Check if there's no error during operation
		// Update status
		TransferCfg->status = CodeStatus | I2C_SETUP_STATUS_DONE;
		return SUCCESS;

s_error:
		/* Clear AA bit to disable ACK on I2C bus */
		I2Cx->CONCLR = I2C_I2CONCLR_AAC;

		// Update status
		TransferCfg->status = CodeStatus;
		return ERROR;
	}

	else if (Opt == I2C_TRANSFER_INTERRUPT)
	{
		tmp = I2C_getNum(I2Cx);
		I2C_SlaveComplete[tmp] = FALSE;
		// Setup tx_rx data, callback and interrupt handler
		i2cdat[tmp].txrx_setup_slave = *TransferCfg;

		// Set direction phase, read first
		i2cdat[tmp].dir = 1;

		// Enable AA
		I2Cx->CONSET = I2C_I2CONSET_AA;
		I2Cx->CONCLR = I2C_I2CONCLR_SIC | I2C_I2CONCLR_STAC;
		I2C_IntCmd(I2Cx, TRUE);

		return (SUCCESS);
	}

	return ERROR;
}

/********************************************************************//**
 * @brief		Transmit an array of bytes in Master mode
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	TransferCfg   Pointer to a I2C_M_SETUP_Type structure that
 * 								contains specified information about the
 * 								configuration for master transfer.
 * @param[in]	Opt				a I2C_TRANSFER_OPT_Type type that selected for
 * 								interrupt or polling mode.
 * @return 		SUCCESS or ERROR
 *********************************************************************/
Status	I2C_MasterTransmitData(LPC_I2Cn_Type* I2Cx, I2C_M_SETUP_Type *TransferCfg, I2C_TRANSFER_OPT_Type Opt)
{
	TransferCfg->rx_data = NULL;
	TransferCfg->rx_length = 0;
	TransferCfg->retransmissions_max = 3;
	TransferCfg->tx_count = 0;
	TransferCfg->rx_count = 0;
	TransferCfg->retransmissions_count = 0;

	return I2C_MasterTransferData(I2Cx, TransferCfg, Opt);
}

/********************************************************************//**
 * @brief		Receive an array of bytes in Master mode
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	TransferCfg   Pointer to a I2C_M_SETUP_Type structure that
 * 								contains specified information about the
 * 								configuration for master transfer.
 * @param[in]	Opt				a I2C_TRANSFER_OPT_Type type that selected for
 * 								interrupt or polling mode.
 * @return 		SUCCESS or ERROR
 *********************************************************************/
Status I2C_MasterReceiveData(LPC_I2Cn_Type* I2Cx, I2C_M_SETUP_Type *TransferCfg, I2C_TRANSFER_OPT_Type Opt)
{
	TransferCfg->tx_data = NULL;
	TransferCfg->tx_length = 0;
	TransferCfg->retransmissions_max = 3;
	TransferCfg->tx_count = 0;
	TransferCfg->rx_count = 0;
	TransferCfg->retransmissions_count = 0;

	return I2C_MasterTransferData (I2Cx, TransferCfg, Opt);
}

/********************************************************************//**
 * @brief		Transmit an array of bytes in Slave mode
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	TransferCfg   Pointer to a I2C_S_SETUP_Type structure that
 * 								contains specified information about the
 * 								configuration for slave transfer.
 * @param[in]	Opt				a I2C_TRANSFER_OPT_Type type that selected for
 * 								interrupt or polling mode.
 * @return 		SUCCESS or ERROR
 *********************************************************************/
Status	I2C_SlaveTransmitData(LPC_I2Cn_Type* I2Cx, I2C_S_SETUP_Type *TransferCfg, I2C_TRANSFER_OPT_Type Opt)
{
	TransferCfg->tx_count = 0;
	TransferCfg->rx_data = NULL;
	TransferCfg->rx_length = 0;
	TransferCfg->rx_count = 0;

	return I2C_SlaveTransferData(I2Cx, TransferCfg, Opt);
}

/********************************************************************//**
 * @brief		Receive an array of bytes in Slave mode
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	TransferCfg   Pointer to a I2C_S_SETUP_Type structure that
 * 								contains specified information about the
 * 								configuration for slave transfer.
 * @param[in]	Opt				a I2C_TRANSFER_OPT_Type type that selected for
 * 								interrupt or polling mode.
 * @return 		SUCCESS or ERROR
 *********************************************************************/
Status	I2C_SlaveReceiveData(LPC_I2Cn_Type* I2Cx, I2C_S_SETUP_Type *TransferCfg, I2C_TRANSFER_OPT_Type Opt)
{
	TransferCfg->tx_data = NULL;
	TransferCfg->tx_length = 0;
	TransferCfg->tx_count = 0;
	TransferCfg->rx_count = 0;

	return I2C_SlaveTransferData(I2Cx, TransferCfg, Opt);
}

/*********************************************************************//**
 * @brief		Set Own slave address in I2C peripheral corresponding to
 * 				parameter specified in OwnSlaveAddrConfigStruct.
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	OwnSlaveAddrConfigStruct	Pointer to a I2C_OWNSLAVEADDR_CFG_Type
 * 				structure that contains the configuration information for the
 *              specified I2C slave address.
 * @return 		None
 **********************************************************************/
void I2C_SetOwnSlaveAddr(LPC_I2Cn_Type *I2Cx, I2C_OWNSLAVEADDR_CFG_Type *OwnSlaveAddrConfigStruct)
{
	uint32_t tmp;
	CHECK_PARAM(PARAM_I2Cx(I2Cx));
	CHECK_PARAM(PARAM_I2C_SLAVEADDR_CH(OwnSlaveAddrConfigStruct->SlaveAddrChannel));
	CHECK_PARAM(PARAM_FUNCTIONALSTATE(OwnSlaveAddrConfigStruct->GeneralCallState));

	tmp = (((uint32_t)(OwnSlaveAddrConfigStruct->SlaveAddr_7bit << 1)) \
			| ((OwnSlaveAddrConfigStruct->GeneralCallState == ENABLE) ? 0x01 : 0x00))& I2C_I2ADR_BITMASK;
	switch (OwnSlaveAddrConfigStruct->SlaveAddrChannel)
	{
	case 0:
		I2Cx->ADR0 = tmp;
		I2Cx->MASK[0] = I2C_I2MASK_MASK((uint32_t) \
				(OwnSlaveAddrConfigStruct->SlaveAddrMaskValue));
		break;
	case 1:
		I2Cx->ADR1 = tmp;
		I2Cx->MASK[1] = I2C_I2MASK_MASK((uint32_t) \
				(OwnSlaveAddrConfigStruct->SlaveAddrMaskValue));
		break;
	case 2:
		I2Cx->ADR2 = tmp;
		I2Cx->MASK[2] = I2C_I2MASK_MASK((uint32_t) \
				(OwnSlaveAddrConfigStruct->SlaveAddrMaskValue));
		break;
	case 3:
		I2Cx->ADR3 = tmp;
		I2Cx->MASK[3] = I2C_I2MASK_MASK((uint32_t) \
				(OwnSlaveAddrConfigStruct->SlaveAddrMaskValue));
		break;
	}
}


/*********************************************************************//**
 * @brief		Configures functionality in I2C monitor mode
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	MonitorCfgType Monitor Configuration type, should be:
 * 					- I2C_MONITOR_CFG_SCL_OUTPUT	:I2C module can 'stretch'
 * 				the clock line (hold it low) until it has had time to respond
 * 				to an I2C interrupt.
 * 					- I2C_MONITOR_CFG_MATCHALL		:When this bit is set to '1'
 * 				and the I2C is in monitor mode, an interrupt will be generated
 * 				on ANY address received.
 * @param[in]	NewState New State of this function, should be:
 * 					- ENABLE	:Enable this function.
 * 					- DISABLE	:Disable this function.
 * @return		None
 **********************************************************************/
void I2C_MonitorModeConfig(LPC_I2Cn_Type *I2Cx, uint32_t MonitorCfgType, FunctionalState NewState)
{
	CHECK_PARAM(PARAM_I2Cx(I2Cx));
	CHECK_PARAM(PARAM_I2C_MONITOR_CFG(MonitorCfgType));
	CHECK_PARAM(PARAM_FUNCTIONALSTATE(NewState));

	if (NewState == ENABLE)
	{
		I2Cx->MMCTRL |= MonitorCfgType;
	}
	else
	{
		I2Cx->MMCTRL &= (~MonitorCfgType) & I2C_I2MMCTRL_BITMASK;
	}
}


/*********************************************************************//**
 * @brief		Enable/Disable I2C monitor mode
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	NewState New State of this function, should be:
 * 					- ENABLE	:Enable monitor mode.
 * 					- DISABLE	:Disable monitor mode.
 * @return		None
 **********************************************************************/
void I2C_MonitorModeCmd(LPC_I2Cn_Type *I2Cx, FunctionalState NewState)
{
	CHECK_PARAM(PARAM_I2Cx(I2Cx));
	CHECK_PARAM(PARAM_FUNCTIONALSTATE(NewState));

	if (NewState == ENABLE)
	{
		I2Cx->MMCTRL |= I2C_I2MMCTRL_MM_ENA;
		I2Cx->CONSET = I2C_I2CONSET_AA;
		I2Cx->CONCLR = I2C_I2CONCLR_SIC | I2C_I2CONCLR_STAC;
	}
	else
	{
		I2Cx->MMCTRL &= (~I2C_I2MMCTRL_MM_ENA) & I2C_I2MMCTRL_BITMASK;
		I2Cx->CONCLR = I2C_I2CONCLR_SIC | I2C_I2CONCLR_STAC | I2C_I2CONCLR_AAC;
	}
	I2C_MonitorBufferIndex = 0;
}


/*********************************************************************//**
 * @brief		Get data from I2C data buffer in monitor mode.
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @return		None
 * Note:	In monitor mode, the I2C module may lose the ability to stretch
 * the clock (stall the bus) if the ENA_SCL bit is not set. This means that
 * the processor will have a limited amount of time to read the contents of
 * the data received on the bus. If the processor reads the I2DAT shift
 * register, as it ordinarily would, it could have only one bit-time to
 * respond to the interrupt before the received data is overwritten by
 * new data.
 **********************************************************************/
uint8_t I2C_MonitorGetDatabuffer(LPC_I2Cn_Type *I2Cx)
{
	CHECK_PARAM(PARAM_I2Cx(I2Cx));
	return ((uint8_t)(I2Cx->DATA_BUFFER));
}


/*********************************************************************//**
 * @brief		Get data from I2C data buffer in monitor mode.
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @return		None
 * Note:	In monitor mode, the I2C module may lose the ability to stretch
 * the clock (stall the bus) if the ENA_SCL bit is not set. This means that
 * the processor will have a limited amount of time to read the contents of
 * the data received on the bus. If the processor reads the I2DAT shift
 * register, as it ordinarily would, it could have only one bit-time to
 * respond to the interrupt before the received data is overwritten by
 * new data.
 **********************************************************************/
BOOL_8 I2C_MonitorHandler(LPC_I2Cn_Type *I2Cx, uint8_t *buffer, uint32_t size)
{
	BOOL_8 ret=FALSE;

	I2Cx->CONCLR = I2C_I2CONCLR_SIC;

	buffer[I2C_MonitorBufferIndex] = (uint8_t)(I2Cx->DATA_BUFFER);
	I2C_MonitorBufferIndex++;
	if(I2C_MonitorBufferIndex >= size)
	{
		ret = TRUE;
	}
	return ret;
}


/*********************************************************************//**
 * @brief 		Get status of Master Transfer
 * @param[in]	I2Cx	I2C peripheral selected, should be:
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @return 		Master transfer status, could be:
 * 					- TRUE		:master transfer completed
 * 					- FALSE 	:master transfer have not completed yet
 **********************************************************************/
uint32_t I2C_MasterTransferComplete(LPC_I2Cn_Type *I2Cx)
{
	uint32_t retval, tmp;
	tmp = I2C_getNum(I2Cx);
	retval = I2C_MasterComplete[tmp];
	I2C_MasterComplete[tmp] = FALSE;
	return retval;
}


/*********************************************************************//**
 * @brief 		Get status of Slave Transfer
 * @param[in]	I2Cx	I2C peripheral selected, should be:
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @return 		Complete status, could be: TRUE/FALSE
 **********************************************************************/
uint32_t I2C_SlaveTransferComplete(LPC_I2Cn_Type *I2Cx)
{
	uint32_t retval, tmp;
	tmp = I2C_getNum(I2Cx);
	retval = I2C_SlaveComplete[tmp];
	I2C_SlaveComplete[tmp] = FALSE;
	return retval;
}



/**
 * @}
 */

#endif /* _I2C */

/**
 * @}
 */

/* --------------------------------- End Of File ------------------------------ */
