/****************************************************************************
 *   Project: ANxxxxx Implementing Sigma Delta ADC using LPC800 Analog
 *            Comparator
 *
 *   Description:
 *     This file contains sigma delta implementation using LPC800 comparator
 *     and SCT
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.

 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors'
 * relevant copyright in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers. This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 ****************************************************************************/

#include <LPC8xx.h>

#include <cr_section_macros.h>
#include <NXP/crp.h>

#include "lpc8xx_sd_adc.h"
#include "circular_buffer.h"
#include "simple_iir.h"
extern void uart_init(void);

extern int printf(const char *format, ...);

#define FEEDBACK_PIN       1  /*< Use PIO0_1 as feedback pin */
#define SD_ADC_DEBUG_PIN   15 /*< Use PIO0_15 as Analog Comparator output and
                                  CTIN_0 */

// Variable to store CRP value in. Will be placed automatically
// by the linker when "Enable Code Read Protect" selected.
// See crp.h header for more information
__CRP const unsigned int CRP_WORD = CRP_NO_CRP;

/* SCT Interrupt Handler
 *
 * Required by the SD ADC module
 */
void SCT_IRQHandler(void) {
	sdadc_sct_irqhandler();
}

void sdadc_user_irqhandler(uint32_t adc_val) {
	cbuf_write(adc_val);
}

int main(void) {
	SystemCoreClockUpdate();
	uart_init();
	cbuf_clear();

	/* Init the SD ADC pins */
	LPC_IOCON->PIO0_1  = 0x80; /* Disable pull up / pull down resistor for
	                              feedback pin */
	LPC_IOCON->PIO0_15 = 0x80; /* Disable pull up / pull down resistor for
	                              acmp_out_pin */
	sdadc_init(FEEDBACK_PIN, SD_ADC_DEBUG_PIN);

	NVIC_EnableIRQ(SCT_IRQn);

	printf("%s\r\n", "Welcome to SD ADC Demo");

	while(1) {
		uint32_t adc_reading;
		if(cbuf_read(&adc_reading) >= 0) {
#if defined (ENABLE_IIR_LPF)
			uint32_t result = iir_lpf(adc_reading);
			printf("%d\r\n", result);
#else
			printf("%d\r\n", adc_reading);
#endif
		}
	}

	return 0;
}
