/*********************************************************************
*                SEGGER MICROCONTROLLER GmbH & Co. KG                *
*        Solutions for real time microcontroller applications        *
**********************************************************************
*                                                                    *
*        (c) 2003-2012     SEGGER Microcontroller GmbH & Co KG       *
*                                                                    *
*        Internet: www.segger.com    Support:  support@segger.com    *
*                                                                    *
**********************************************************************

----------------------------------------------------------------------
File    : Main.c
Purpose : Calls hardware initialization and application.
--------  END-OF-HEADER  ---------------------------------------------
*/
#include <stdio.h>
#include <stdint.h>
#ifndef _WINDOWS
	#include "GUI.h"
	#include "bsp.h"
	#include "LCDConf.h"
	#include "spifi_rom_api.h"
	extern void SPIFI_Init (void);
#else
	#include "WinConf.h"
#endif

#ifdef __CROSSWORKS_ARM
extern void __low_level_init(); // hwconf.c
#endif

/*********************************************************************
*
*       Defines
*
**********************************************************************
*/
#define TOUCH_TIMER_INTERVAL  10
//
// EEPROM (I2C)
//
#define EEPROM_MAGIC_NUMBER     0xEA01CDAE
#define EEPROM_MAX_SEQ_VER      1
//
// SSP
//
#define SSP_MODE_LCD   0
#define SSP_MODE_TS    1

//
// Physical display size
//
#ifndef _WINDOWS
  #define XSIZE_PHYS  _LcdParams.PPL
  #define YSIZE_PHYS  _LcdParams.LPP
#else
  #define XSIZE_PHYS  FB_XSIZE
  #define YSIZE_PHYS  FB_YSIZE
#endif

/*********************************************************************
*
*       Extern data
*
**********************************************************************
*/
extern volatile int TimeMS;  // Defined in GUI_X.c
extern volatile uint32_t  time_to_move;
/*********************************************************************
*
*       Static data
*
**********************************************************************
*/
#ifndef _WINDOWS
static EEPROM_CONFIG_OFFS _EepromConfigOffs;
static U8                 _acBuffer[256];  // Buffer to read EEPROM initstring
static U8  PenIsDown;
static U8  IsInited = 0;
#endif
/*********************************************************************
*
*       Local functions
*
**********************************************************************
*/
/*********************************************************************
*
*       CheckUpdateTouch
*
* Function description
*   Reads touch values from ADC on Truly LCD display
*   controller IC on Embedded Artists QVGA Base Board.
*   Checks if a touch event has occurred. If we found an event the
*   static variables for x and y will be filled with new values that
*   can be processed via GUI touch routines.
*
* Return value:
*      0  No touch event, x, y have not been updated
*   != 0  Touch event occurred, x, y have been updated and return value is read z value
*/
#ifndef _WINDOWS
static int CheckUpdateTouch(void) {
  int x;
  int y;
  int z;
  int z1;
  int z2;
  int LastX;
  int LastY;
  int LastZ1;
  int LastZ2;
  int i;

  LastY   = SSP_SendCmd(READ_Y(0));
  LastY >>= 3;

  LastX   = SSP_SendCmd(READ_X(0));
  LastX >>= 3;

  LastZ1   = SSP_SendCmd(READ_Z1(0));
  LastZ1 >>= 3;

  LastZ2   = SSP_SendCmd(READ_Z2(0));
  LastZ2 >>= 3;
  //
  // Get debounced y value
  //
  for (i = 0; i < TS_DEBOUNCE_MAX; i++) {
    y   = SSP_SendCmd(READ_Y(0));
    y >>= 3;
    if (ABS(LastY - y) <= TS_DEBOUNCE_TOL) {
      break;
    }
    LastY = y;
  }
  //
  // Get debounced x value
  //
  for (i = 0; i < TS_DEBOUNCE_MAX; i++) {
    x   = SSP_SendCmd(READ_X(0));
    x >>= 3;
    if (ABS(LastX - x) <= TS_DEBOUNCE_TOL) {
      break;
    }
    LastX = x;
  }
  //
  // Get debounced z1 value
  //
  for (i = 0; i < TS_DEBOUNCE_MAX; i++) {
    z1   = SSP_SendCmd(READ_Z1(0));
    z1 >>= 3;
    if (ABS(LastZ1 - z1) <= TS_DEBOUNCE_TOL) {
      break;
    }
    LastZ1 = z1;
  }
  //
  // Get debounced z2 value
  //
  for (i = 0; i < TS_DEBOUNCE_MAX; i++) {
    z2   = SSP_SendCmd(READ_Z1(0));
    z2 >>= 3;
    if (ABS(LastZ2 - z2) <= TS_DEBOUNCE_TOL) {
      break;
    }
    LastZ2 = z2;
  }
  //
  // Update static variables if we have a valid touch
  //
  z = ((x * z2) - z1) / z1;
  if (z > 40000) {
    return 0;   // No update
  }
  _TouchX = y;
  _TouchY = x;
  return z;
}
#endif
/*********************************************************************
*
*       ExecTouch
*
* Function description
*   Check for new touch event. Static x, y coordinates will be updated
*   by the _CheckUpdateTouch() routine. If no touch event has occurred
*   we do nothing. Is called by SysTickHandler().
*/
#ifndef _WINDOWS
static void ExecTouch(void) {
  GUI_PID_STATE State;
  int           HasUpdate;

  if (IsInited == 0) {
    return;
  }
  HasUpdate = CheckUpdateTouch();
  SSP_SendCmd(PWRDOWN);
  if (HasUpdate) {
    PenIsDown = 1;
    GUI_TOUCH_Exec();
  } else if (PenIsDown) {
    //
    // No further touch event after at least one touch event means we have
    // lift the pen from the touch screen which means a click.
    //
    PenIsDown = 0;
    GUI_PID_GetState(&State);
    State.Pressed = 0;
    GUI_PID_StoreState(&State);
  }
}
#endif

/*********************************************************************
*
*       EatWhite
*/
static void EatWhite(char ** ps) {
  while ((**ps == ' ') && (**ps != '\0')) {
    (*ps)++;
  }
}

/*********************************************************************
*
*       ParseDec
*
* Return value:
*   Number of digits parsed
*/
#ifndef _WINDOWS
static int ParseDec(char ** ps, int Len, int * pNumDigits) {
  U32 Data;
  int NumDigits;

  Data      = 0;
  NumDigits = 0;

  EatWhite(ps);
  while (Len--) {
    int v;
    char c;

    c = **ps;
    v =  ((c >= '0') && (c <= '9')) ? c - '0' : -1;
    if (v >= 0) {
      Data = (Data * 10) + v;
      (*ps)++;
      NumDigits++;
    } else {
      if (pNumDigits != NULL) {
        *pNumDigits = NumDigits;
      }
      break;
    }
  }
  return Data;
}
#endif

/*********************************************************************
*
*       InitLCDC
*
* Function description
*   Inits the LCD controller, backlight and sets the frame buffer.
*/
#ifndef _WINDOWS
static void InitLCDC(void) {
  U32 TIMH, TIMV, POL, CTRL;

  //
  // Translate and check params from EEPROM values to device values
  //
  if (_LcdParams.BPP == 16) {
    _LcdParams.BPP = LCDC_BPP_16_565;
  } else {
    while (1);  // Error, display mode not yet supported
  }
  if (_LcdParams.LCD == 0) {
    _LcdParams.LCD = LCDC_TFT;
  } else {
    while (1);  // Error, display type not yet supported
  }
  //
  // Init LCDC
  //
  TIMH  = 0             // Configure horizontal axis
			   | ((((U32)_LcdParams.PPL / 16) - 1) <<  2)
			   |  (((U32)_LcdParams.HSW - 1)       <<  8)
			   |  (((U32)_LcdParams.HFP - 1)       << 16)
			   |  (((U32)_LcdParams.HBP - 1)       << 24)
			   ;
  TIMV  = 0             // Configure vertical axis
			   | (((U32)_LcdParams.LPP - 1) <<  0)
			   | (((U32)_LcdParams.VSW - 1) << 10)
			   | (((U32)_LcdParams.VFP)     << 16)
			   | (((U32)_LcdParams.VBP)     << 24)
			   ;
  POL   = 0             // Configure clock and signal polarity
			   | (FindClockDivisor(_LcdParams.Clk) <<  0)
			   | (((U32)_LcdParams.ACB - 1)         <<  6)
			   | (((U32)_LcdParams.IVS)             << 11)
			   | (((U32)_LcdParams.IHS)             << 12)
			   | (((U32)_LcdParams.IPC)             << 13)
			   | (((U32)_LcdParams.PPL - 1)         << 16)
			   ;
  CTRL  = 0             // Configure operating mode and panel parameters
			   | ((U32)_LcdParams.BPP << 1)
			   | ((U32)BGR            << 8)
			   | ((U32)_LcdParams.LCD << 5)
			   ;
  BSPInitLCDC(TIMH, TIMV, POL, CTRL, (U32)&_aVRAM[0]);
}
#endif

/*********************************************************************
*
*       ExecInitSequence
*
* Function description
*   Executes the LCD init sequence found in EEPROM.
*/
#ifndef _WINDOWS
static void ExecInitSequence(char * s) {
  char * pCmd;
  int    ParamLen;  // Len of command parameters

  //
  // Init PCA9532 multiplexer
  //
  I2C_PCA9532_BSP_Init(100000);
  //
  // Process init string
  //
  while (*s != '\0') {
    EatWhite(&s);
    //
    // Calc len of next command
    //
    pCmd = s;
    while ((*s != ',') && (*s != '\0')) {
      s++;  // Skip chars until at end of command
    }
    ParamLen = (s - pCmd) - 1;
    if (*s == ',') {
      s++;  // Skip separator if any
    }
    //
    // Execute command
    //
    switch (*pCmd++) {
    case 'c':  // Exec PCA9532 action
      ExecutePCA9532Cmd(pCmd, ParamLen);
      break;
    case 'd':  // Delay for a given number of ms
      DelayMs(ParseDec(&pCmd, ParamLen, NULL));
      break;
    case 'o':  // Open LCD, for us this means init the LCDC
      InitLCDC();
      break;
    case 'v':  // Check if init string format is supported
      if (ParseDec(&pCmd, ParamLen, NULL) > EEPROM_MAX_SEQ_VER) {
        while (1);  // Init string format not supported
      }
      break;
    }
  }
}
#endif

/*********************************************************************
*
*       InitTouch
*
* Function description
*   Initializes the touch screen.
*/
#ifndef _WINDOWS
void InitTouch(void) {
  U32 TouchOrientation;
  U8  Config[3];
  U8  TouchHasSwapXY;
  int TouchADLeft;
  int TouchADRight;
  int TouchADTop;
  int TouchADBottom;

  //
  // Init ports and SSP interface as needed for touch
  //
  SetTouchPin();
  InitSSP(SSP_MODE_TS);
  Config[0] = REF_ON;
  Config[1] = (READ_12BIT_SER(ADS_A2A1A0_vaux) | ADS_PD10_ALL_ON);
  Config[2] = PWRDOWN;
  TS_CS_SET();
  SSP_Send(Config, 3);
  TS_CS_CLR();
  //
  // Set touch orientation and calibrate touch
  //
  if (_Display == DISPLAY_TRULY_240_320) {
    //
    // 3.2" Truly display
    // Swap XY for touch as touch does not have the same orientation as the display
    //
    if (LCD_GetSwapXYEx(0)) {
      TouchHasSwapXY = 0;
    } else {
      TouchHasSwapXY = 1;
    }
    //
    // Calibrate touch
    //
    TouchOrientation = (GUI_MIRROR_X * LCD_GetMirrorYEx(0)) |  // XY swapped
                       (GUI_MIRROR_Y * LCD_GetMirrorXEx(0)) |  // XY swapped
                       (GUI_SWAP_XY  * TouchHasSwapXY) ;
    GUI_TOUCH_SetOrientation(TouchOrientation);
    if (TouchHasSwapXY) {
      GUI_TOUCH_Calibrate(GUI_COORD_X, 0, YSIZE_PHYS, TOUCH_TRULY_240_320_AD_LEFT, TOUCH_TRULY_240_320_AD_RIGHT);
      GUI_TOUCH_Calibrate(GUI_COORD_Y, 0, XSIZE_PHYS, TOUCH_TRULY_240_320_AD_TOP , TOUCH_TRULY_240_320_AD_BOTTOM);
    } else {
      GUI_TOUCH_Calibrate(GUI_COORD_Y, 0, XSIZE_PHYS, TOUCH_TRULY_240_320_AD_LEFT, TOUCH_TRULY_240_320_AD_RIGHT);
      GUI_TOUCH_Calibrate(GUI_COORD_X, 0, YSIZE_PHYS, TOUCH_TRULY_240_320_AD_TOP , TOUCH_TRULY_240_320_AD_BOTTOM);
    }
  } else {
    //
    // LCD board other than truly
    //
    if        (_Display == DISPLAY_BOARD_480_272) {
      TouchADLeft   = TOUCH_BOARD_480_272_AD_LEFT;
      TouchADRight  = TOUCH_BOARD_480_272_AD_RIGHT;
      TouchADTop    = TOUCH_BOARD_480_272_AD_TOP;
      TouchADBottom = TOUCH_BOARD_480_272_AD_BOTTOM;
    } else if (_Display == DISPLAY_BOARD_800_480) {
      TouchADLeft   = TOUCH_BOARD_800_480_AD_LEFT;
      TouchADRight  = TOUCH_BOARD_800_480_AD_RIGHT;
      TouchADTop    = TOUCH_BOARD_800_480_AD_TOP;
      TouchADBottom = TOUCH_BOARD_800_480_AD_BOTTOM;
    } else {
      while (1);  // Touch not supported
    }
    //
    // Calibrate touch
    //
    TouchOrientation = (GUI_MIRROR_X * LCD_GetMirrorXEx(0)) |
                       (GUI_MIRROR_Y * LCD_GetMirrorYEx(0)) |
                       (GUI_SWAP_XY  * LCD_GetSwapXYEx (0)) ;
    GUI_TOUCH_SetOrientation(TouchOrientation);
    GUI_TOUCH_Calibrate(GUI_COORD_X, 0, XSIZE_PHYS, TouchADLeft, TouchADRight);
    GUI_TOUCH_Calibrate(GUI_COORD_Y, 0, YSIZE_PHYS, TouchADTop , TouchADBottom);
  }
  IsInited = 1;
}
#endif

/*********************************************************************
*
*       UpdateDisplayConfig
*
* Function description
*   Checks for attached display and updates the LCDC parameters.
*/
#ifndef _WINDOWS
static void UpdateDisplayConfig(void) {

  U32 v;

  ConfigureI2C();
  //
  // Check if we have an EEPROM that contains display settings and
  // save the decision for later usage.
  //
  EEPROM_Read(0, (U8*)&v, 4);
  if (v == EEPROM_MAGIC_NUMBER) {
    //
    // Read LCD settings from EEPROM
    //
    EEPROM_Read(64, (U8*)&_EepromConfigOffs, sizeof(EEPROM_CONFIG_OFFS));           // Offset of LCD params is stored at offset 64 in the EEPROM
    EEPROM_Read(_EepromConfigOffs.LcdParams, (U8*)&_LcdParams, sizeof(LCD_PARAM));  // Read LCD params
    if        (_LcdParams.PPL == 480) {
      _Display = DISPLAY_BOARD_480_272;
    } else if (_LcdParams.PPL == 800) {
      _Display = DISPLAY_BOARD_800_480;
    } else {
      while (1);  // Error, display type not supported
    }
  } else {
    _Display = DISPLAY_TRULY_240_320;
    //
    // Setup display settings for Truly display manually as there is no EEPROM on this display module
    //
    _LcdParams.HBP  = HBP_TRULY_3_2;
    _LcdParams.HFP  = HFP_TRULY_3_2;
    _LcdParams.HSW  = HSW_TRULY_3_2;
    _LcdParams.PPL  = PPL_TRULY_3_2;
    _LcdParams.VBP  = VBP_TRULY_3_2;
    _LcdParams.VFP  = VFP_TRULY_3_2;
    _LcdParams.VSW  = VSW_TRULY_3_2;
    _LcdParams.LPP  = LPP_TRULY_3_2;
    _LcdParams.IOE  = IOE_TRULY_3_2;
    _LcdParams.IPC  = IPC_TRULY_3_2;
    _LcdParams.IHS  = IHS_TRULY_3_2;
    _LcdParams.IVS  = IVS_TRULY_3_2;
    _LcdParams.ACB  = ACB_TRULY_3_2;
    _LcdParams.BPP  = BPP_TRULY_3_2;
    _LcdParams.Clk  = CLK_TRULY_3_2;
    _LcdParams.LCD  = LCD_TRULY_3_2;
    _LcdParams.Dual = DUAL_TRULY_3_2;
  }
}
#endif

/*********************************************************************
*
*       InitController
*
* Function description
*   Initializes the LCD controller and touch screen
*/
#ifndef _WINDOWS
static void InitController(unsigned LayerIndex) {
    InitLCD();
    //
    // Check which display is connected and initialize it
    //
    if (_Display == DISPLAY_TRULY_240_320) {
      //
      // No configuration EEPROM found, display is Truly (if any connected).
      // LCDC is enabled after display init.
      //
      InitSSP(SSP_MODE_LCD);
      InitTrulyLCD();
      InitLCDC();
    } else {
      //
      // EEPROM with settings was found. Either 4.3" or 7" display is connected.
      // LCDC is enabled upon request of the stored initialization sequence in EEPROM.
      //
      //_InitOtherLCD();
      EEPROM_Read(_EepromConfigOffs.LcdInit, _acBuffer, _EepromConfigOffs.PowerDown - _EepromConfigOffs.LcdInit);  // Read LCD init sequence string
      ExecInitSequence((char*)_acBuffer);
    }
  //
  // Init touch
  //
#if GUI_SUPPORT_TOUCH  // Used when touch screen support is enabled
  InitTouch();
#endif
}
#endif

void MainTask(void);  // Defined in SEGGERDEMO.c

/*********************************************************************
*
*       main
*/
int main(void) {
	uint32_t i;
  #ifdef __CROSSWORKS_ARM
  __low_level_init();
  #endif
  #ifndef _WINDOWS
	SPIFI_Init();
  HW_X_Config();      // Initialization of Hardware
	UpdateDisplayConfig();
	
	//Fill part of Video buffer with  with black color
	for (i=0;i<(XSIZE_PHYS*YSIZE_PHYS*PIXEL_WIDTH)/4;i++){
	_aVRAM[i]=0x00;
	}
	InitController(0);
	#endif
  MainTask();         // emWin application
}

/*********************************************************************
*
*       SysTick_Handler()
*
* Function description
*   This is the code that gets called when the processor receives a
*   _SysTick exception.
*/
#ifdef __cplusplus
extern "C" {
#endif
  void SysTick_Handler(void);      // Avoid warning, Systick_Handler is not prototyped in any CMSIS header
#ifdef __cplusplus
}
#endif

#ifndef _WINDOWS
void SysTick_Handler(void) {
  static int TouchCnt = 0;

  if (TouchCnt == TOUCH_TIMER_INTERVAL) {
    ExecTouch();
    TouchCnt = 0;
  } else {
    TouchCnt++;
  }
  TimeMS++;
	
	time_to_move++;
}

#endif
