/*
* Copyright(C) NXP Semiconductors, 2011
* All rights reserved.
*
* Software that is described herein is for illustrative purposes only
* which provides customers with programming information regarding the
* LPC products.  This software is supplied "AS IS" without any warranties of
* any kind, and NXP Semiconductors and its licensor disclaim any and 
* all warranties, express or implied, including all implied warranties of 
* merchantability, fitness for a particular purpose and non-infringement of 
* intellectual property rights.  NXP Semiconductors assumes no responsibility
* or liability for the use of the software, conveys no license or rights under any
* patent, copyright, mask work right, or any other intellectual property rights in 
* or to any products. NXP Semiconductors reserves the right to make changes
* in the software without notification. NXP Semiconductors also makes no 
* representation or warranty that such application will be suitable for the
* specified use without further testing or modification.
* 
* Permission to use, copy, modify, and distribute this software and its 
* documentation is hereby granted, under NXP Semiconductors' and its 
* licensor's relevant copyrights in the software, without fee, provided that it 
* is used in conjunction with NXP Semiconductors microcontrollers.  This 
* copyright, permission, and disclaimer notice must appear in all copies of 
* this code.
*/


#include "../bsp_internal.h"

#if (BOARD == BOARD_EAOEMBase_RevA)

/*********************************************************************
*
*       Defines for HW_Conf
*
**********************************************************************
*/

#define SDRAM_SIZE_MB    32
#define SDRAM_SIZE       (SDRAM_SIZE_MB * 1024 * 1024)
#define SDRAM_BASE_ADDR  0xA0000000

#define NVIC_VTOR           (*(volatile uint32_t*)(0xE000ED08))

/*********************************************************************
*
*       Defines for emWin_Conf
*
**********************************************************************
*/

#ifdef LPC4088
#define LPC_SSP LPC_SSP2
#else
#define LPC_SSP LPC_SSP0
#endif

//
// EEPROM (I2C)
//

#define EEPROM_I2C_ADDR         0x56  // Address of the EEPROM that holds the display configurations

//
// LCDC
//
#define LCD_BACKLIGHT_BIT  18    // P1.18
#define LCD_REG_BIT        19    // P0.19
#define BCD_BIT            26    // Bypass pixel clock divider
#define LCD_DATA()         (LPC_GPIO0->SET = (1 << LCD_REG_BIT))
#define LCD_CMD()          (LPC_GPIO0->CLR = (1 << LCD_REG_BIT))

/*********************************************************************
*
*       Defines, sfrs
*
**********************************************************************
*/
#define MATRIX_ARB  (*(volatile uint32_t*)(0x400FC188))

//
// SSP
//
#define SSPSR_TNF      (1 << 1)  // Transmit FIFO not full
#define SSPSR_RNE      (1 << 2)  // Receive FIFO not empty
#define SSPSR_BSY      (1 << 4)  // Busy

#define SSP_CLOCK      3000000
#define SSP_FIFO_SIZE  8

#define SSP_MODE_LCD   0
#define SSP_MODE_TS    1

//
// I2C
//
#define I2C0_BASE_ADDR  0x4001C000

//
// PCA9532 (I2C)
//
#define PCA9532_I2C_ADDR         0x64  // Address of the PCA9532 on the display PCB
#define PCA9532_DISP_3V3_EN_BIT  0
#define PCA9532_DISP_5V_EN_BIT   1
#define PCA9532_DISP_EN_BIT      4
#define PCA9532_DISP_BL_BIT      8

//
// Touch screen
//
#define TOUCH_CS_BIT                   20
//#define TS_CS_SET()                    LPC_GPIO0->CLR = (1 << TOUCH_CS_BIT)
//#define TS_CS_CLR()                    LPC_GPIO0->SET = (1 << TOUCH_CS_BIT)

/*********************************************************************
*
*       Macros, function replacement
*/
#define GUI_COUNTOF(a)          (sizeof(a) / sizeof(a[0]))

/*********************************************************************
*
*       Local functions for emWin_Conf
*
**********************************************************************
*/

/*********************************************************************
*
*       _EatWhite
*/
static void _EatWhite(char ** ps) {
  while ((**ps == ' ') && (**ps != '\0')) {
    (*ps)++;
  }
}
/*********************************************************************
*
*       _ParseDec
*
* Return value:
*   Number of digits parsed
*/
static int _ParseDec(char ** ps, int Len, int * pNumDigits) {
  uint32_t Data;
  int NumDigits;

  Data      = 0;
  NumDigits = 0;

  _EatWhite(ps);
  while (Len--) {
    int v;
    char c;

    c = **ps;
    v =  ((c >= '0') && (c <= '9')) ? c - '0' : -1;
    if (v >= 0) {
      Data = (Data * 10) + v;
      (*ps)++;
      NumDigits++;
    } else {
      if (pNumDigits != NULL) {
        *pNumDigits = NumDigits;
      }
      break;
    }
  }
  return Data;
}

/*********************************************************************
*
*       _InitLCDPorts
*
* Function description
*   Initializes the port pins as needed for the LCD.
*/
#ifndef _WINDOWS
static void InitLCDPorts(void) {
  LPC_IOCON->P0_4   = 7;  // LCD_VD_0
  LPC_IOCON->P0_5   = 7;  // LCD_VD_1
  LPC_IOCON->P0_6   = 7;  // LCD_VD_8
  LPC_IOCON->P0_7   = 7;  // LCD_VD_9
  LPC_IOCON->P0_8   = 7;  // LCD_VD_16
  LPC_IOCON->P0_9   = 7;  // LCD_VD_17
#ifdef LPC4088
	LPC_IOCON->P0_10  = 7;  // LCD_VD_5
#endif

  LPC_IOCON->P1_20  = 7;  // LCD_VD_10
  LPC_IOCON->P1_21  = 7;  // LCD_VD_11
  LPC_IOCON->P1_22  = 7;  // LCD_VD_12
  LPC_IOCON->P1_23  = 7;  // LCD_VD_13
  LPC_IOCON->P1_24  = 7;  // LCD_VD_14
  LPC_IOCON->P1_25  = 7;  // LCD_VD_15
  LPC_IOCON->P1_26  = 7;  // LCD_VD_20
  LPC_IOCON->P1_27  = 7;  // LCD_VD_21
  LPC_IOCON->P1_28  = 7;  // LCD_VD_22
  LPC_IOCON->P1_29  = 7;  // LCD_VD_23

  LPC_IOCON->P2_0   = 7;  // LCD_PWR
  LPC_IOCON->P2_1   = 7;  // LCD_LE
  LPC_IOCON->P2_2   = 7;  // LCD_DCLK
  LPC_IOCON->P2_3   = 7;  // LCD_FP
  LPC_IOCON->P2_4   = 7;  // LCD_ENAB_M
  LPC_IOCON->P2_5   = 7;  // LCD_LP
  LPC_IOCON->P2_6   = 7;  // LCD_VD_4
#ifndef LPC4088
  LPC_IOCON->P2_7   = 7;  // LCD_VD_5
#endif  /* LPC4088 */
  LPC_IOCON->P2_8   = 7;  // LCD_VD_6
  LPC_IOCON->P2_9   = 7;  // LCD_VD_7

  LPC_IOCON->P2_11  = 7;  // LCD_CLKIN
  LPC_IOCON->P2_12  = 7;  // LCD_VD_18
  LPC_IOCON->P2_13  = 7;  // LCD_VD_19

  LPC_IOCON->P4_28  = 7;  // LCD_VD_2
  LPC_IOCON->P4_29  = 7;  // LCD_VD_3

  LPC_GPIO1->DIR   |= (1 << LCD_BACKLIGHT_BIT);
  LPC_GPIO1->CLR   |= (1 << LCD_BACKLIGHT_BIT);  // Initially set backlight to off
}
#endif

/*********************************************************************
*
*       _GetSspDividers
*
* Function description
*   Find best SSP dividers for the desired clock rate.
*/
#ifndef _WINDOWS
static void GetSspDividers(uint32_t TargetClock, uint8_t* Scr, uint8_t* Cpsr) {
  uint16_t tScr  = 1;
  uint8_t  tCpsr = 2;
  uint32_t Clock = PeripheralClock;

  while (Clock > TargetClock) {
    Clock = PeripheralClock / ((tScr + 1) * tCpsr);
    if (Clock > TargetClock) {
      tScr++;
      if (tScr > 0xFF) {
        tScr   = 0;
        tCpsr += 2;
      }
    }
  }
  *Scr  = (uint8_t)tScr;
  *Cpsr = tCpsr;
}
#endif

/*********************************************************************
*
*       _WriteLcdReg
*
* Function description
*   Write to display register. Used to configure Truly LCD display
*   controller IC on Embedded Artists QVGA Base Board.
*/
#ifndef _WINDOWS
static void WriteLcdReg(uint16_t Addr, uint16_t Data) {
  uint8_t Buffer[2];

  LCD_CMD();
  Buffer[0] = 0;
  Buffer[1] = Addr & 0xFF;
  SSP_Send(Buffer, 2);

  LCD_DATA();
  Buffer[0] = Data >> 8;
  Buffer[1] = Data & 0xFF;
  SSP_Send(Buffer, 2);

  LCD_CMD();
  Buffer[0] = 0;
  Buffer[1] = 0x22;
  SSP_Send(Buffer, 2);
}
#endif

/*********************************************************************
*
*       Global functions for emWin_Conf
*
**********************************************************************
*/

/*********************************************************************
*
*       _FindClockDivisor
*
* Function description
*   Find closest clock divider to get the desired clock rate.
*/
#ifndef _WINDOWS
uint32_t FindClockDivisor(uint32_t TargetClock) {
  uint32_t Divider;
  uint32_t r;

  Divider = 1;
  while (((SystemCoreClock / Divider) > TargetClock) && (Divider <= 0x3F)) {
    Divider++;
  }
  if (Divider <= 1) {
    r = (1 << BCD_BIT);  // Skip divider logic if clock divider is 1
  } else {
    //
    // Use found divider
    //
    Divider -= 2;
    r = 0
        | (((Divider >> 0) & 0x1F)
        | (((Divider >> 5) & 0x1F) << 27))
        ;
  }
  return r;
}
#endif

/*********************************************************************
*
*       DelayMs()
*
* Function description
*   Starts a timer and waits for the given delay in ms.
*/
void DelayMs(uint32_t ms) {
  LPC_TIM0->TCR = 0x02;  // Reset timer
  LPC_TIM0->PR  = 0x00;  // Set prescaler to zero
  LPC_TIM0->MR0 = ms * (SystemCoreClock / (LPC_SC->PCLKSEL & 0x1F) / 1000 - 1);
  LPC_TIM0->IR  = 0xFF;  // Reset all interrrupts
  LPC_TIM0->MCR = 0x04;  // Stop timer on match
  LPC_TIM0->TCR = 0x01;  // Start timer
  //
  // Wait until delay time has elapsed
  //
  while (LPC_TIM0->TCR & 1);
}

void ConfigureI2C (void)
{
	  //
	  // Init I2C for access to configuration EEPROM
	  //
	  LPC_SC->PCONP |= (1UL << 7);                       // Power I2C
	  LPC_IOCON->P0_27 = (LPC_IOCON->P0_27 & ~7UL) | 1;  // I2C0_SDA
	  LPC_IOCON->P0_28 = (LPC_IOCON->P0_27 & ~7UL) | 1;  // I2C0_SCL
	  //I2C_Init(I2C0_BASE_ADDR, PeripheralClock, 100000);
	  I2C_Init(I2C_0, 100000);
	  /* Enable I2C0 operation */
	  I2C_Cmd(I2C_0, ENABLE);
}

/*********************************************************************
*
*       SSP_Send
*
* Function description
*   Sends data via the SSP0 interface. Used to configure Truly LCD
*   display controller IC on Embedded Artists QVGA Base Board.
*/
#ifndef _WINDOWS
void SSP_Send(uint8_t* pData, uint32_t NumBytes) {
  volatile uint8_t  Dummy;

  do {
    //
    // Wait for not busy and TX FIFO not full
    //
    while ((LPC_SSP->SR & (SSPSR_TNF|SSPSR_BSY)) != SSPSR_TNF);
    //
    // Send next byte
    //
    LPC_SSP->DR = *pData;
    pData++;
    //
    // Whenever a byte is written, MISO FIFO counter is incremented.
    // Therefore we have to clear the byte from FIFO as otherwise we would
    // get old data on next read.
    //
    while ((LPC_SSP->SR & (SSPSR_BSY|SSPSR_RNE)) != SSPSR_RNE);
    Dummy = LPC_SSP->DR;
  }
  while (--NumBytes);
}
#endif

/*********************************************************************
*
*       SSP_Recv
*
* Function description
*   Sends data via the SSP0 interface. Used to configure Truly LCD
*   display controller IC on Embedded Artists QVGA Base Board.
*/
#ifndef _WINDOWS
void SSP_Recv(uint8_t* pData, uint32_t NumBytes) {
  do {
    LPC_SSP->DR = 0xFF;                                            // Using peer-to-peer communication, SSPDR has to be written to receive data
    while ((LPC_SSP->SR & (SSPSR_BSY | SSPSR_RNE)) != SSPSR_RNE);  // Wait for not busy and receive buffer not empty
    *pData++ = LPC_SSP->DR;
  } while (--NumBytes);
}
#endif

/*********************************************************************
*
*       EEPROM_Read
*
* Function description
*   Reads from EEPROM connected via I2C.
*
* Return value:
*      0: O.K.
*   != 0: Error
*/
#ifndef _WINDOWS
uint8_t EEPROM_Read(uint16_t Offs, uint8_t * pData, uint16_t NumBytes) {
	  uint8_t acOffs[2];
	  I2C_M_SETUP_Type i2cData;
	  acOffs[0] = (Offs >> 8);
	  acOffs[1] = (Offs & 0xFF);

	  i2cData.sl_addr7bit = EEPROM_I2C_ADDR;
	  i2cData.tx_data = acOffs;
	  i2cData.tx_length = 2;
	  i2cData.rx_data = pData;
	  i2cData.rx_length = NumBytes;
	  i2cData.retransmissions_max = 3;

	  return I2C_MasterTransferData(I2C_0, &i2cData, I2C_TRANSFER_POLLING);
}
#endif

/*********************************************************************
*
*       InitLCD
*
* Function description
*   Initializes the port pins as needed for the LCD and selects
*   an available LCD.
*/
#ifndef _WINDOWS
void InitLCD(void) {
  InitLCDPorts();
  //
  // Setup BUS priorities
  //
  MATRIX_ARB = 0            // Set AHB Matrix priorities [0..3] with 3 being highest priority
               | (1 <<  0)  // PRI_ICODE : I-Code bus priority. Should be lower than PRI_DCODE for proper operation.
               | (2 <<  2)  // PRI_DCODE : D-Code bus priority.
               | (0 <<  4)  // PRI_SYS   : System bus priority.
               | (0 <<  6)  // PRI_GPDMA : General Purpose DMA controller priority.
               | (0 <<  8)  // PRI_ETH   : Ethernet: DMA priority.
               | (3 << 10)  // PRI_LCD   : LCD DMA priority.
               | (0 << 12)  // PRI_USB   : USB DMA priority.
               ;
}
#endif

/*********************************************************************
*
*       InitSSP
*
* Function description
*   Initialize the SSP interface. Used to configure Truly LCD display
*   controller IC on Embedded Artists QVGA Base Board.
*/
#ifndef _WINDOWS
void InitSSP(uint8_t Mode) {
  uint8_t Cpsr;
  uint8_t Scr;
  uint8_t i;
  volatile uint32_t v;

  //
  // Init ports
  //
#ifdef LPC4088
  LPC_SC->PCONP |=  (1UL << 20);  // Power the SSP2 port
  LPC_IOCON->P5_2 = 2;  // SSP2_SCK
  LPC_IOCON->P5_3 = 2;  // SSP2_SSEL
  LPC_IOCON->P5_1 = 2;  // SSP2_MISO
  LPC_IOCON->P5_0 = 2;  // SSP2_MOSI
#else
  LPC_IOCON->P0_15 = 2;  // SSP0_SCK
  LPC_IOCON->P0_16 = 2;  // SSP0_SSEL
  LPC_IOCON->P0_17 = 2;  // SSP0_MISO
  LPC_IOCON->P0_18 = 2;  // SSP0_MOSI
#endif
  //
  // Init SSP parameters
  //
  GetSspDividers(SSP_CLOCK, &Scr, &Cpsr);
  v               = 0
                    | (7   << 0)  // DSS : 8-bit transfer
                    | (0   << 4)  // FRF : Frame Format -> 0 = SPI
                    | (Scr << 8)  // SCR : Serial Clock Rate
                    ;
  if        (Mode == SSP_MODE_LCD) {
    v            |= 0
                    | (0   << 6)  // CPOL: Clock out polarity
                    | (0   << 7)  // CPHA: Clock out phase
                    ;
  } else if (Mode == SSP_MODE_TS) {
    v            |= 0
                    | (1   << 6)  // CPOL: Clock out polarity
                    | (1   << 7)  // CPHA: Clock out phase
                    ;
  }
  LPC_SSP->CR0   = v;
  LPC_SSP->CPSR  = Cpsr;         // CPSR: Clock prescale register, master mode, minimum divisor is 0x02
  for (i = 0; i < SSP_FIFO_SIZE; i++) {
    v = LPC_SSP->DR;             // Clear the RxFIFO
  }
  LPC_SSP->CR1   = 0
                    | (1 << 1)    // SSE : Enable SSP
                    | (0 << 2)    // MS  : Master mode
                    ;
}
#endif

/*********************************************************************
*
*       InitTrulyLCD
*
* Function description
*   Initialize the LCD display. Used to configure Truly LCD display
*   controller IC on Embedded Artists QVGA Base Board.
*/
#ifndef _WINDOWS
void InitTrulyLCD(void) {
  //
  // Power LCD
  //
  LPC_GPIO2->DIR |= 1;  // Set to output
  LPC_GPIO2->SET  = 1;  // Output LCD power
  //
  // Set display CMD/DATA register select pin to output high
  //
  LPC_GPIO0->DIR |= (1 << LCD_REG_BIT);
  //
  // Display init sequence
  //
  WriteLcdReg(0x00,0x0001);
  DelayMs(15 * 2);
  WriteLcdReg(0x03,0x6E3E);  // AAAC
  WriteLcdReg(0x0C,0x0007);  // 0002
  WriteLcdReg(0x0D,0x000E);  // 000A
  WriteLcdReg(0x0E,0x2C00);  // 2C00
  WriteLcdReg(0x1E,0x00AE);  // 00B8
  DelayMs(15 * 2);

  WriteLcdReg(0x07,0x0021);
  DelayMs(50 * 2);
  WriteLcdReg(0x07,0x0023);
  DelayMs(50 * 2);
  WriteLcdReg(0x07,0x0033);
  DelayMs(50 * 2);

  WriteLcdReg(0x01,0x2B3F);
  WriteLcdReg(0x02,0x0600);
  WriteLcdReg(0x10,0x0000);
  DelayMs(15 * 2);
  WriteLcdReg(0x11,0xC5B0);  // 60B0: RGB I/R
  DelayMs(20 * 2);
  WriteLcdReg(0x15,0x00D0);

  WriteLcdReg(0x05,0x0000);
  WriteLcdReg(0x06,0x0000);
  WriteLcdReg(0x16,0xEF1C);
  WriteLcdReg(0x17,0x0003);
  WriteLcdReg(0x07,0x0233);
  WriteLcdReg(0x0B,0x5310);
  WriteLcdReg(0x0F,0x0000);
  WriteLcdReg(0x25,0xE000);
  DelayMs(20 * 2);

  WriteLcdReg(0x41,0x0000);
  WriteLcdReg(0x42,0x0000);
  WriteLcdReg(0x48,0x0000);
  WriteLcdReg(0x49,0x013F);
  WriteLcdReg(0x44,0xEF00);
  WriteLcdReg(0x45,0x0000);
  WriteLcdReg(0x46,0x013F);
  WriteLcdReg(0x4A,0x0000);
  WriteLcdReg(0x4B,0x0000);
  DelayMs(20 * 2);

  WriteLcdReg(0x30,0x0707);
  WriteLcdReg(0x31,0x0600);  // 0704
  WriteLcdReg(0x32,0x0005);  // 0204
  WriteLcdReg(0x33,0x0402);  // 0201
  WriteLcdReg(0x34,0x0203);
  WriteLcdReg(0x35,0x0204);
  WriteLcdReg(0x36,0x0204);
  WriteLcdReg(0x37,0x0401);  // 0502
  WriteLcdReg(0x3A,0x0302);
  WriteLcdReg(0x3B,0x0500);
  DelayMs(20 * 2);
  WriteLcdReg(0x22,0x0000);
  DelayMs(20 * 2);
}
#endif

/*********************************************************************
*
*       InitLCDC
*
* Function description
*   Inits the LCD controller, backlight and sets the frame buffer.
*/
#ifndef _WINDOWS
void BSPInitLCDC(uint32_t TIMH, uint32_t TIMV,uint32_t POL,uint32_t CTRL,uint32_t VRAM_ADDR_PHYS) {
  uint32_t i;
  //
  // Init LCDC
  //
  LPC_SC->PCONP |=  (1UL << 0);  // Power the LCDC
  LPC_LCD->CTRL &= ~(1UL << 0);  // Disable the LCDC

  LPC_LCD->TIMH = TIMH;
  LPC_LCD->TIMV = TIMV;
  LPC_LCD->POL  = POL;
  LPC_LCD->CTRL = CTRL;

  for (i = 0; i < GUI_COUNTOF(LPC_LCD->PAL); i++) {
    LPC_LCD->PAL[i] = 0;  // Clear the color palette with black
  }
  LPC_SC->LCD_CFG = 0x0;  // No panel clock prescaler
  //
  // Enable LCDC
  //
  LPC_LCD->UPBASE  = VRAM_ADDR_PHYS;
  LPC_LCD->CTRL   |= (1 <<  0);                 // Enable LCD signals
  LPC_LCD->CTRL   |= (1 << 11);                 // Enable LCD power
  LPC_GPIO1->SET  |= (1 << LCD_BACKLIGHT_BIT);  // Set backlight to on
}
#endif

/*********************************************************************
*
*       I2C_PCA9532_BSP_Init
*
* Function description
*   Init I2C_PCA9532 for emWin_Conf
*/
void I2C_PCA9532_BSP_Init(uint32_t ClockRate)
{
	I2C_PCA9532_Init(ClockRate);
}

/*********************************************************************
*
*       SetTouchPin
*
* Function description
*   Set Touch Pin for emWin_Conf
*/
void SetTouchPin(void)
{
	LPC_GPIO0->DIR |= (1 << TOUCH_CS_BIT);  // P0.20 as output (touch controller CS)
}

/*********************************************************************
*
*       TS_CS_SET
*
* Function description
*   
*/
void TS_CS_SET(void)
{
	LPC_GPIO0->CLR = (1 << TOUCH_CS_BIT);
}

/*********************************************************************
*
*       TS_CS_CLR
*
* Function description
*   
*/
void TS_CS_CLR(void)
{
	LPC_GPIO0->SET = (1 << TOUCH_CS_BIT);
}

/*********************************************************************
*
*       ExecutePCA9532Cmd
*
* Function description
*   Executes a command on the multiplexer.
*/
#ifndef _WINDOWS
void ExecutePCA9532Cmd(char * s, int Len) {
  uint32_t  v;
  char Cmd;
  uint8_t   PwmDuty;

  Cmd = *s++;
  //
  // Parse all parameters
  //
  switch (Cmd) {
  case 'c':
  case 'd':
  case '3':
  case '5':
    v = _ParseDec(&s, Len - 1, NULL);
    break;
  default:
    return;
  }
  //
  // Exec cmd
  //
  switch (Cmd) {
  case 'c':  // Enable/disable backlight
    v = 100 - v;
    if (v > 100) {
      v = 100;
    }
    PwmDuty = (255 * v) / 100;
    I2C_PCA9532_SetPwm(I2C0_BASE_ADDR, PCA9532_I2C_ADDR, PCA9532_DISP_BL_BIT, 0, PwmDuty, 0);
    break;
  case 'd':  // Enable/disable display
    if (v) {
      v = 0;
    } else {
      v = 1;
    }
    I2C_PCA9532_SetPinSelect(I2C0_BASE_ADDR, PCA9532_I2C_ADDR, PCA9532_DISP_EN_BIT, v);
    break;
  case '3':  // Enable/disable display 3v3
    I2C_PCA9532_SetPinSelect(I2C0_BASE_ADDR, PCA9532_I2C_ADDR, PCA9532_DISP_3V3_EN_BIT, v);
    break;
  case '5':  // Enable/disable display 5v
    I2C_PCA9532_SetPinSelect(I2C0_BASE_ADDR, PCA9532_I2C_ADDR, PCA9532_DISP_5V_EN_BIT, v);
    break;
  }
}
#endif

/*********************************************************************
*
*       SSP_SendCmd
*
* Function description
*   Sends a cmd via the SSP0 interface and returns the answer.
*   Used to configure Truly LCD display controller IC on
*   Embedded Artists QVGA Base Board.
*/
#ifndef _WINDOWS
uint16_t SSP_SendCmd(uint8_t Cmd) {
  uint8_t v[2];

  v[0] = Cmd;

  TS_CS_SET();
  SSP_Send(v, 1);
  SSP_Recv(v, 2);
  TS_CS_CLR();

  return ((v[0] << 8) | v[1]);
}
#endif
/*********************************************************************
*
*       HW_X_Config()
*
* Function description
*   Initializes the hardware (timer).
*   May be modified, if an other timer should be used.
*/

#ifdef __ICCARM__
  typedef void( *intfunc )( void );
  typedef union { intfunc __fun; void * __ptr; } intvec_elem;
  extern const intvec_elem __vector_table[];
  #define __Vectors    __vector_table
#else
  extern unsigned char __Vectors;
#endif

#ifdef __cplusplus
extern "C" {
#endif
int __low_level_init(void);  // Avoid "no ptototype" warning
#ifdef __cplusplus
  }
#endif
void HW_X_Config(void) {
  uint32_t  TickPrio;

  //
  // Assuming PLL and core clock were already set by SystemInit() (called
  // from the startup code), initializing any hardware is not necessary.
  // Just ensure that the system clock variable is updated and then
  // set the periodic system timer tick for embOS.
  //
  SystemCoreClockUpdate();                      // Ensure, the SystemCoreClock is set
  if (SysTick_Config(SystemCoreClock / 1000)) { // Setup SysTick Timer for 1 msec interrupts
    while (1);                                  // Handle Error
  }
  //
  // Initialize NVIC vector base address. Might be necessary for RAM targets
  // or application not running from 0.
  //
  // NVIC_VTOR = (uint32_t)&__Vectors;
  //
  // Set the interrupt priority for the system timer to 2nd lowest level to
  // ensure the timer can preempt PendSV handler
  //
  NVIC_SetPriority(SysTick_IRQn, (uint32_t) -1);
  TickPrio  = NVIC_GetPriority(SysTick_IRQn);
  TickPrio -=1;
  NVIC_SetPriority(SysTick_IRQn, (uint32_t)TickPrio);

  // Call low-level HW initialisation (EMC, etc.) in BSP.c
  __low_level_init();
}

/*********************************************************************
*
*       _TestSDRAM()
*
* Function description
*   Simple SDRAM test for testing timing setup
*
* Return value
*   1: Error
*   0: O.K.
*/
static int _TestSDRAM(void) {
  volatile uint32_t * pWriteLong;
  volatile uint16_t * pWriteShort;
  uint32_t            Data;
  uint32_t            i;
  uint32_t            j;

  pWriteLong  = (uint32_t *)SDRAM_BASE_ADDR;
  pWriteShort = (uint16_t *)SDRAM_BASE_ADDR;
  //
  // Fill 16 bit wise
  //
  for (i = 0; i < (SDRAM_SIZE / 0x40000); i++) {
    for (j = 0; j < 0x100; j++) {
      *pWriteShort++ = (i + j);
      *pWriteShort++ = (i + j) + 1;
    }
  }
  //
  // Verifying
  //
  pWriteLong = (uint32_t*)SDRAM_BASE_ADDR;
  for (i = 0; i < (SDRAM_SIZE / 0x40000); i++) {
    for (j = 0; j < 0x100; j++) {
      Data = *pWriteLong++;
      if (Data != (((((i + j) + 1) & 0xFFFF) << 16) | ((i + j) & 0xFFFF))) {
        return 1;  // Error
      }
    }
  }
  return 0;  // O.K.
}

/*********************************************************************
*
*       _FindDelay()
*
* Function description:
*   Test for min./max. CMDDLY or FBCLKDLY values
*
* Parameter:
*   DelayType: 0 = CMDDLY, 1 = FBCLKDLY
*/
static void _FindDelay(int DelayType) {
  uint32_t Delay;
  uint32_t Min;
  uint32_t Max;
  uint32_t v;

  //
  // Init start values
  //
  Delay = 0x00;
  Min   = 0xFF;
  Max   = 0xFF;
  //
  // Test for DLY min./max. values
  //
  while (Delay < 32) {
    //
    // Setup new DLY value to test
    //
    if (DelayType == 0) {
      v                 = LPC_SC->EMCDLYCTL & ~0x001Ful;
      LPC_SC->EMCDLYCTL = v | Delay;
    } else {
      v                 = LPC_SC->EMCDLYCTL & ~0x1F00ul;
      LPC_SC->EMCDLYCTL = v | (Delay << 8);
    }
    //
    // Test configured DLY value and find out min./max. values that will work
    //
    if (_TestSDRAM() == 0) {
      //
      // Test passed, remember min. DLY value if not done yet
      //
      if (Min == 0xFF) {
        Min = Delay;
      }
    } else {
      //
      // Test failed, if a min. value has been found before, remember the current value for max.
      //
      if (Min != 0xFF) {
        Max = Delay;
      }
    }
    Delay++;
  }
  //
  // Calc DLY value
  //
  if        (Max != 0xFF) {  // If we found a min. and max. value we use the average of the min. and max. values to get an optimal DQSIN delay
    Delay = (Min + Max) / 2;
  } else if (Min != 0xFF) {  // If we found only a min. value we use the average of the min. value and the longest DLY value to get an optimal DQSIN delay
    Delay = (Min + 0x1F) / 2;
  } else {                   // No working max. and/or min. values found
    while (1);  // Fatal error
  }
  //
  // Setup DLY value to work with
  //
  if (DelayType == 0) {
    v                 = LPC_SC->EMCDLYCTL & ~0x001Ful;
    LPC_SC->EMCDLYCTL = v | Delay;
  } else {
    v                 = LPC_SC->EMCDLYCTL & ~0x1F00ul;
    LPC_SC->EMCDLYCTL = v | (Delay << 8);
  }
}

/*********************************************************************
*
*       _CalibrateOsc()
*
* Function description:
*   Calibrate ring oscillator.
*
* Return value:
*   Current ring oscillator count
*/
static uint32_t _CalibrateOsc(void) {
  uint32_t Cnt;
  uint32_t v;
  uint32_t i;

  //
  // Init start values
  //
  Cnt = 0;
  //
  // Calibrate osc.
  //
  for (i = 0; i < 10; i++) {
    LPC_SC->EMCCAL = (1 << 14);     // Start calibration
    v = LPC_SC->EMCCAL;
    while ((v & (1 << 15)) == 0) {  // Wait for calibration done
      v = LPC_SC->EMCCAL;
    }
    Cnt += (v & 0xFF);
  }
  return (Cnt / 10);
}

/*********************************************************************
*
*       _AdjustEMCTiming()
*
* Function description:
*   Adjust timings for EMC.
*/
static void _AdjustEMCTiming(uint32_t Delay) {
  uint32_t FBClkDly;
  uint32_t FBDelay;
  uint32_t CmdDly;
  uint32_t v;

  FBDelay           = _CalibrateOsc();
  v                 = LPC_SC->EMCDLYCTL;
  CmdDly            = ((v &  0x001Ful) * Delay / FBDelay) & 0x1F;
  FBClkDly          = ((v &  0x1F00ul) * Delay / FBDelay) & 0x1F00;
  LPC_SC->EMCDLYCTL =  (v & ~0x1F1Ful) | FBClkDly | CmdDly;
}

/*********************************************************************
*
*       _EMC_Init()
*
*  Purpose:
*    Initializes external memory controller for SDRAM
*/
static void _EMC_Init(void) {
  volatile uint32_t CmdDly;
  volatile uint32_t Dummy;
  volatile uint32_t i;

  LPC_SC->PCONP     |= (1 << 11);   // Turn on EMC peripheral clock
  LPC_SC->EMCDLYCTL  = 0x00001010;
  LPC_EMC->Control   = 1;           // EMC enable
  LPC_EMC->Config    = 0;
  //
  // Port init
  //
  LPC_IOCON->P3_0  = 1;  // D0
  LPC_IOCON->P3_1  = 1;  // D1
  LPC_IOCON->P3_2  = 1;  // D2
  LPC_IOCON->P3_3  = 1;  // D3
  LPC_IOCON->P3_4  = 1;  // D4
  LPC_IOCON->P3_5  = 1;  // D5
  LPC_IOCON->P3_6  = 1;  // D6
  LPC_IOCON->P3_7  = 1;  // D7
  LPC_IOCON->P3_8  = 1;  // D8
  LPC_IOCON->P3_9  = 1;  // D9
  LPC_IOCON->P3_10 = 1;  // D10
  LPC_IOCON->P3_11 = 1;  // D11
  LPC_IOCON->P3_12 = 1;  // D12
  LPC_IOCON->P3_13 = 1;  // D13
  LPC_IOCON->P3_14 = 1;  // D14
  LPC_IOCON->P3_15 = 1;  // D15
  LPC_IOCON->P3_16 = 1;  // D16
  LPC_IOCON->P3_17 = 1;  // D17
  LPC_IOCON->P3_18 = 1;  // D18
  LPC_IOCON->P3_19 = 1;  // D19
  LPC_IOCON->P3_20 = 1;  // D20
  LPC_IOCON->P3_21 = 1;  // D21
  LPC_IOCON->P3_22 = 1;  // D22
  LPC_IOCON->P3_23 = 1;  // D23
  LPC_IOCON->P3_24 = 1;  // D24
  LPC_IOCON->P3_25 = 1;  // D25
  LPC_IOCON->P3_26 = 1;  // D26
  LPC_IOCON->P3_27 = 1;  // D27
  LPC_IOCON->P3_28 = 1;  // D28
  LPC_IOCON->P3_29 = 1;  // D29
  LPC_IOCON->P3_30 = 1;  // D30
  LPC_IOCON->P3_31 = 1;  // D31

  LPC_IOCON->P4_0  = 1;  // A0
  LPC_IOCON->P4_1  = 1;  // A1
  LPC_IOCON->P4_2  = 1;  // A2
  LPC_IOCON->P4_3  = 1;  // A3
  LPC_IOCON->P4_4  = 1;  // A4
  LPC_IOCON->P4_5  = 1;  // A5
  LPC_IOCON->P4_6  = 1;  // A6
  LPC_IOCON->P4_7  = 1;  // A7
  LPC_IOCON->P4_8  = 1;  // A8
  LPC_IOCON->P4_9  = 1;  // A9
  LPC_IOCON->P4_10 = 1;  // A10
  LPC_IOCON->P4_11 = 1;  // A11
  LPC_IOCON->P4_12 = 1;  // A12
  LPC_IOCON->P4_13 = 1;  // A13
  LPC_IOCON->P4_14 = 1;  // A14
  LPC_IOCON->P4_15 = 1;  // A15
  LPC_IOCON->P4_16 = 1;  // A16
  LPC_IOCON->P4_17 = 1;  // A17
  LPC_IOCON->P4_18 = 1;  // A18
  LPC_IOCON->P4_19 = 1;  // A19
  LPC_IOCON->P4_20 = 1;  // A20
  LPC_IOCON->P4_21 = 1;  // A21
  LPC_IOCON->P4_22 = 1;  // A22
  LPC_IOCON->P4_23 = 1;  // A23

  LPC_IOCON->P4_24 = 1;  // OE
  LPC_IOCON->P4_25 = 1;  // WE
  LPC_IOCON->P4_26 = 1;  // BLS0
  LPC_IOCON->P4_27 = 1;  // BLS1
  LPC_IOCON->P4_28 = 1;  // BLS2
  LPC_IOCON->P4_29 = 1;  // BLS3
  LPC_IOCON->P4_30 = 1;  // CS0
  LPC_IOCON->P4_31 = 1;  // CS1
  LPC_IOCON->P2_14 = 1;  // CS2
  LPC_IOCON->P2_15 = 1;  // CS3
  LPC_IOCON->P2_16 = 1;  // CAS
  LPC_IOCON->P2_17 = 1;  // RAS
  LPC_IOCON->P2_18 = 1;  // CLKOUT0
  LPC_IOCON->P2_19 = 1;  // CLKOUT1
  LPC_IOCON->P2_20 = 1;  // DYCS0
  LPC_IOCON->P2_21 = 1;  // DYCS1
  LPC_IOCON->P2_22 = 1;  // DYCS2
  LPC_IOCON->P2_23 = 1;  // DYCS3
  LPC_IOCON->P2_24 = 1;  // CKEOUT0
  LPC_IOCON->P2_25 = 1;  // CKEOUT1
  LPC_IOCON->P2_26 = 1;  // CKEOUT2
  LPC_IOCON->P2_27 = 1;  // CKEOUT3
  LPC_IOCON->P2_28 = 1;  // DQMOUT0
  LPC_IOCON->P2_29 = 1;  // DQMOUT1
  LPC_IOCON->P2_30 = 1;  // DQMOUT2
  LPC_IOCON->P2_31 = 1;  // DQMOUT3
  //
  // Setup EMC config for SDRAM, timings for 60MHz bus
  //
  LPC_EMC->DynamicConfig0    = 0x00004480;  // 256MB, 8Mx32, 4 banks, 12 rows, 9 columns, buffers disabled
  LPC_EMC->DynamicRasCas0    = 0x00000202;  // 2 RAS, 2 CAS latency */
  LPC_EMC->DynamicReadConfig = 0x00000001;  // Command delayed strategy, using EMCCLKDELAY
  LPC_EMC->DynamicRP         = 0x00000001;  // n + 1 clock cycles
  LPC_EMC->DynamicRAS        = 0x00000003;  // n + 1 clock cycles
  LPC_EMC->DynamicSREX       = 0x00000005;  // n + 1 clock cycles
  LPC_EMC->DynamicAPR        = 0x00000002;  // n + 1 clock cycles
  LPC_EMC->DynamicDAL        = 0x00000003;  // n     clock cycles
  LPC_EMC->DynamicWR         = 0x00000001;  // n + 1 clock cycles
  LPC_EMC->DynamicRC         = 0x00000004;  // n + 1 clock cycles
  LPC_EMC->DynamicRFC        = 0x00000004;  // n + 1 clock cycles
  LPC_EMC->DynamicXSR        = 0x00000005;  // n + 1 clock cycles
  LPC_EMC->DynamicRRD        = 0x00000001;  // n + 1 clock cycles
  LPC_EMC->DynamicMRD        = 0x00000001;  // n + 1 clock cycles
  DelayMs(100);
  LPC_EMC->DynamicControl    = 0x00000183;  // Issue NOP command
  DelayMs(200);
  LPC_EMC->DynamicControl    = 0x00000103;  // Issue PALL command
  LPC_EMC->DynamicRefresh    = 0x00000002;  // n * 16 clock cycles
  for (i = 0; i < 0x80; i++);               // Wait 128 AHB clock cycles
  LPC_EMC->DynamicRefresh    = 0x0000003A;  // n * 16 clock cycles
  //
  // Init SDRAM
  //
  LPC_EMC->DynamicControl = 0x00000083;                             // Issue MODE command
  Dummy = *((volatile uint32_t*)(SDRAM_BASE_ADDR | (0x22 << (2+2+9))));  // 4 burst, 2 CAS latency
  LPC_EMC->DynamicControl = 0x00000000;                             // Issue NORMAL command
  LPC_EMC->DynamicConfig0 = 0x00084480;                             // 256MB, 8Mx32, 4 banks, 12 rows, 9 columns, buffers (re-)enabled
  //
  // Auto calibrate timings
  //
  CmdDly = _CalibrateOsc();
  //
  // Find best delay values
  //
  _FindDelay(0);  // EMCDLY
  _FindDelay(1);  // FBCLKDLY
  _AdjustEMCTiming(CmdDly);
}

/*********************************************************************
*
*       __low_level_init()
*
*       Initialize memory controller, clock generation and pll
*
*       Has to be modified, if another CPU clock frequency should be
*       used. This function is called during startup and
*       has to return 1 to perform segment initialization
*/

int __low_level_init(void) {
  SystemCoreClockUpdate();   // Ensure, the SystemCoreClock is set
  _EMC_Init();               // Init SDRAM, NAND- and NOR-flash
  return 1;
}

/*****  End of file  ************************************************/

void bsp_init(void)
{
}

void Serial_Init(const uint32_t BaudRate, const bool DoubleSpeed)
{
#ifdef __LPC177X_8X__
	UART_CFG_Type UARTConfigStruct;

	PINSEL_ConfigPin(UART_PORTNUM,UART_TX_PINNUM,UART_FUNCNUM);
	PINSEL_ConfigPin(UART_PORTNUM,UART_RX_PINNUM,UART_FUNCNUM);
	/* Initialize UART Configuration parameter structure to default state:
	 * Baudrate = 115200bps
	 * 8 data bit
	 * 1 Stop bit
	 * None parity
	 */
	UART_ConfigStructInit(&UARTConfigStruct);
	// Re-configure baudrate
	UARTConfigStruct.Baud_rate = BaudRate;

	// Initialize DEBUG_UART_PORT peripheral with given to corresponding parameter
	UART_Init(DEBUG_UART_PORT, &UARTConfigStruct);

	// Enable UART Transmit
	UART_TxCmd(DEBUG_UART_PORT, ENABLE);
#endif
}

void USB_Connect(void)
{
#ifdef __LPC177X_8X__

#endif
}


void Buttons_Init(void)
{
	GPIO_Init();
	GPIO_SetDir(BUTTONS_BUTTON1_GPIO_PORT_NUM,(1<<BUTTONS_BUTTON1_GPIO_BIT_NUM),0); // input
}

void Joystick_Init(void)
{
	GPIO_SetDir(JOYSTICK_UP_GPIO_PORT_NUM,(1<<JOYSTICK_UP_GPIO_BIT_NUM),0); 		// input
	GPIO_SetDir(JOYSTICK_DOWN_GPIO_PORT_NUM,(1<<JOYSTICK_DOWN_GPIO_BIT_NUM),0); 	// input
	GPIO_SetDir(JOYSTICK_LEFT_GPIO_PORT_NUM,(1<<JOYSTICK_LEFT_GPIO_BIT_NUM),0); 	// input
	GPIO_SetDir(JOYSTICK_RIGHT_GPIO_PORT_NUM,(1<<JOYSTICK_RIGHT_GPIO_BIT_NUM),0); 	// input
	GPIO_SetDir(JOYSTICK_PRESS_GPIO_PORT_NUM,(1<<JOYSTICK_PRESS_GPIO_BIT_NUM),0); 	// input
}

uint8_t Joystick_GetStatus(void)
{
	uint8_t ret = NO_BUTTON_PRESSED;

	if((GPIO_ReadValue(JOYSTICK_UP_GPIO_PORT_NUM) & (1<<JOYSTICK_UP_GPIO_BIT_NUM)) == 0x00)
	{
		ret |= JOY_UP;
	}
	else if((GPIO_ReadValue(JOYSTICK_DOWN_GPIO_PORT_NUM) & (1<<JOYSTICK_DOWN_GPIO_BIT_NUM)) == 0x00)
	{
		ret |= JOY_DOWN;
	}
	else if((GPIO_ReadValue(JOYSTICK_LEFT_GPIO_PORT_NUM) & (1<<JOYSTICK_LEFT_GPIO_BIT_NUM)) == 0x00)
	{
		ret |= JOY_LEFT;
	}
	else if((GPIO_ReadValue(JOYSTICK_RIGHT_GPIO_PORT_NUM) & (1<<JOYSTICK_RIGHT_GPIO_BIT_NUM)) == 0x00)
	{
		ret |= JOY_RIGHT;
	}
	else if((GPIO_ReadValue(JOYSTICK_PRESS_GPIO_PORT_NUM) & (1<<JOYSTICK_PRESS_GPIO_BIT_NUM)) == 0x00)
	{
		ret |= JOY_PRESS;
	}

	return ret;
}
#define AUDIO_MAX_PC	10
uint8_t audio_buffer[2048] __attribute__ ((aligned(4)))__DATA(USBRAM_SECTION);
uint32_t audio_buffer_size = 0;
uint32_t audio_buffer_rd_index = 0;
uint32_t audio_buffer_wr_index = 0;
uint32_t audio_buffer_count = 0;
void Audio_Init(uint32_t samplefreq)
{
	volatile uint32_t pclkdiv, pclk;
	PINSEL_ConfigPin(0, 26, 2);

	PINSEL_SetAnalogPinMode(0,26,ENABLE);
	PINSEL_DacEnable(0, 26, ENABLE);
	  LPC_SC->PCONP |= (1 << 12);
	  LPC_DAC->CR = 0x00008000;		/* DAC Output set to Middle Point */
	  pclkdiv = (LPC_SC->PCLKSEL >> 2) & 0x03;
	  switch ( pclkdiv )
	  {
		case 0x00:
		default:
		  pclk = SystemCoreClock/4;
		break;
		case 0x01:
		  pclk = SystemCoreClock;
		break;
		case 0x02:
		  pclk = SystemCoreClock/2;
		break;
		case 0x03:
		  pclk = SystemCoreClock/8;
		break;
	  }
	  LPC_TIM1->MR0 = pclk/samplefreq - 1;	/* TC0 Match Value 0 */
	  LPC_TIM1->MCR = 3;					/* TCO Interrupt and Reset on MR0 */
	  LPC_TIM1->TCR = 1;					/* TC0 Enable */
	  NVIC_EnableIRQ(TIMER1_IRQn);
	switch(samplefreq){
		case 11025:
		case 22050:
		case 44100:
			audio_buffer_size = 1764;
			break;
		case 8000:
		case 16000:
		case 32000:
		case 48000:
		default:
			audio_buffer_size = samplefreq * 4 * AUDIO_MAX_PC / 1000;
			break;
		}
}
void Audio_Reset_Data_Buffer(void)
{
	audio_buffer_count = 0;
	audio_buffer_wr_index = 0;
	audio_buffer_rd_index = 0;
}
uint32_t Audio_Get_ISO_Buffer_Address(uint32_t last_packet_size)
{
	audio_buffer_wr_index += last_packet_size;
	audio_buffer_count += last_packet_size;

	if(audio_buffer_count > audio_buffer_size)
	{
		Audio_Reset_Data_Buffer();
	}
	if(audio_buffer_wr_index >= audio_buffer_size)
		audio_buffer_wr_index -= audio_buffer_size;
	return (uint32_t)&audio_buffer[audio_buffer_wr_index];
}

void TIMER1_IRQHandler(void)
{
		uint32_t val;
		int32_t sample;
		if(audio_buffer_count >= 2) /*has enough data */
		{
			audio_buffer_count -= 2;
			sample = *(int16_t *)(audio_buffer + audio_buffer_rd_index);
			audio_buffer_rd_index+=2;
			if(audio_buffer_rd_index >= audio_buffer_size)
				audio_buffer_rd_index -= audio_buffer_size;
		}else{
			sample = 0;
		}
		val = sample>>6;
		val &= 0x3FF;
		val += (0x01 << 9);
		DAC_UpdateValue(0, val);
		if((audio_buffer_size != 0) && (audio_buffer_count >= (audio_buffer_size/2)))
		{
			audio_buffer_count-=2;
			audio_buffer_rd_index+=2;
			if(audio_buffer_rd_index >= audio_buffer_size)
				audio_buffer_rd_index -= audio_buffer_size;
		}
		LPC_TIM1->IR = 1;                         /* Clear Interrupt Flag */
}
void LEDs_Init(void){}
void LEDs_SetAllLEDs(uint32_t LEDMask){}
#endif
