//*****************************************************************************
//   +--+
//   | ++----+
//   +-++    |
//     |     |
//   +-+--+  |
//   | +--+--+
//   +----+    Copyright (c) 2011 Code Red Technologies Ltd.
//
// crt_flash_if.h : Flash driver interface header
//
//*****************************************************************************

#ifndef __CRT_FLASH_IF_H
#define __CRT_FLASH_IF_H

#ifndef __cplusplus
typedef unsigned long long	UINT64;
typedef unsigned int		UINT32;
typedef UINT32				UINT;
typedef unsigned short		UINT16;
typedef unsigned char		UINT8;

typedef unsigned int		bool;

#define false   0
#define true    1

#define FALSE   false
#define TRUE    true

#endif  // __cplusplus




#define VERS       1           // Interface Version 1.01

#define UNKNOWN    0           // Unknown
#define ONCHIP     1           // On-chip Flash Memory
#define EXT8BIT    2           // External Flash Device on 8-bit  Bus
#define EXT16BIT   3           // External Flash Device on 16-bit Bus
#define EXT32BIT   4           // External Flash Device on 32-bit Bus
#define EXTSPI     5           // External Flash Device on SPI

#define SECTOR_NUM 512         // Max Number of Sector Items
#define PAGE_MAX   65536       // Max Page Size for Programming

typedef unsigned int	U32;
typedef unsigned short	U16;
typedef unsigned char	U8;

struct FlashSectors  {
  unsigned long   szSector;    // Sector Size in Bytes
  unsigned long AddrSector;    // Address of Sector
};

#define SECTOR_END 0xFFFFFFFF, 0xFFFFFFFF

struct FlashDevice  {
   unsigned short     Vers;    // Version Number and Architecture
   char       DevName[128];    // Device Name and Description
   unsigned short  DevType;    // Device Type: ONCHIP, EXT8BIT, EXT16BIT, ...
   unsigned long    DevAdr;    // Default Device Start Address
   unsigned long     szDev;    // Total Size of Device
   unsigned long    szPage;    // Programming Page Size
   unsigned long       Res;    // Reserved for future Extension
   unsigned char  valEmpty;    // Content of Erased Memory

   unsigned long    toProg;    // Time Out of Program Page Function
   unsigned long   toErase;    // Time Out of Erase Sector Function

   struct FlashSectors sectors[SECTOR_NUM];
};

#define FLASH_DRV_VERS (0x0100+VERS)   // Driver Version, do not modify!


#ifdef __cplusplus
 extern "C" {
#endif

#define WEAK __attribute__ ((weak))

// ===========================
// Flash Programming Functions
// ===========================

 /*
  *  Initialize Flash Programming Functions
  *    Parameter:      adr:  Device Base Address
  *                    clk:  Clock Frequency (Hz)
  *                    fnc:  Function Code (1 - Erase, 2 - Program, 3 - Verify)
  *    Return Value:   0 - OK,  1 - Failed
  */
WEAK  int  Init        (unsigned long adr,   // Initialize Flash
                    unsigned long clk,
                    unsigned long fnc);


  /*
   *  De-Initialize Flash Programming Functions
   *    Parameter:      fnc:  Function Code (1 - Erase, 2 - Program, 3 - Verify)
   *    Return Value:   0 - OK,  1 - Failed
   */
WEAK  int  UnInit      (unsigned long fnc);  // De-initialize Flash


  /*
   *  Blank Check Checks if Memory is Blank
   *    Parameter:      adr:  Block Start Address
   *                    sz:   Block Size (in bytes)
   *                    pat:  Block Pattern
   *    Return Value:   0 - OK,  1 - Failed
   */
WEAK  int  BlankCheck  (unsigned long adr,   // Blank Check
                    unsigned long sz,
                    unsigned char pat);


  /*
   *  Erase complete Flash Memory
   *    Return Value:   0 - OK,  1 - Failed
   */
WEAK  int  EraseChip   (void);               // Erase complete Device

  /*
   *  Erase Sector in Flash Memory
   *    Parameter:      adr:  Sector Address
   *    Return Value:   0 - OK,  1 - Failed
   */
WEAK  int  EraseSector (unsigned long adr);  // Erase Sector Function

  /*
   *  Program Page in Flash Memory
   *    Parameter:      adr:  Page Start Address
   *                    sz:   Page Size
   *                    buf:  Page Data
   *    Return Value:   0 - OK,  1 - Failed
   */
WEAK  int  ProgramPage (unsigned long adr,   // Program Page Function
                    unsigned long sz,
                    unsigned char *buf);

/*
 *  Verify Flash Contents
 *    Parameter:      adr:  Start Address
 *                    sz:   Size (in bytes)
 *                    buf:  Data
 *    Return Value:   (adr+sz) - OK, Failed Address
 */
WEAK  unsigned long Verify	(unsigned long adr,   // Verify Function
                         unsigned long sz,
                         unsigned char *buf);


void __finish_loc(void);	  // Forward declare our API support

#ifdef __cplusplus
}
#endif


typedef struct MemoryDevice  {
  UINT16	version;		// Version of Code Red flash interface
  UINT8		magic_no[8];	// Magic number to identify flash driver file
  UINT32	load_base;		// Driver load address
  UINT32	image_size;		// Size of .text and .data
  UINT32	pcache;			// RAM buffer location
  UINT32	cache_size;		// RAM buffer size
  UINT32	pstack;			// Stack top
  UINT32	stack_size;		// Stack size
  UINT32	finish_loc;		// Operation finish location
  const struct FlashDevice *dev;
  	  	  	  	  	  	  	// Flash device structure
  int		(*Init)			(unsigned long adr,   // Initialize Flash
							 unsigned long clk,
							 unsigned long fnc);
  int		(*UnInit)		(unsigned long fnc);  // De-initialize Flash
  int		(*BlankCheck)	(unsigned long adr,   // Blank Check
							 unsigned long sz,
							 unsigned char pat);
  int		(*EraseChip)	(void);               // Erase complete Device
  int		(*EraseSector)	(unsigned long adr);  // Erase Sector Function
  int		(*ProgramPage)	(unsigned long adr,   // Program Page Function
							 unsigned long sz,
							 unsigned char *buf);
  unsigned long (*Verify)	(unsigned long adr,   // Verify Function
							 unsigned long sz,
							 unsigned char *buf);
} MemoryDevice_t;

#endif  // __CRT_FLASH_IF_H
