/****************************************************************************
 *   $Id:: i2c_slaveROM.c 5992 2010-12-22 21:10:51Z nxp28548                      $
 *   Project: NXP LPC11Axx I2C ROM Driver Slave Mode
 *
 *   Description:
 *     This example set up the LPC11Axx to emulate an SE95 
 *     The slave address is 0x48
 *
 ****************************************************************************
* Software that is described herein is for illustrative purposes only
* which provides customers with programming information regarding the
* products. This software is supplied "AS IS" without any warranties.
* NXP Semiconductors assumes no responsibility or liability for the
* use of the software, conveys no license or title under any patent,
* copyright, or mask work right to the product. NXP Semiconductors
* reserves the right to make changes in the software without
* notification. NXP Semiconductors also make no representation or
* warranty that such application will be suitable for the specified
* use without further testing or modification.

* Permission to use, copy, modify, and distribute this software and its 
* documentation is hereby granted, under NXP Semiconductors 
* relevant copyright in the software, without fee, provided that it 
* is used in conjunction with NXP Semiconductors microcontrollers.  This 
* copyright, permission, and disclaimer notice must appear in all copies of 
* this code.

****************************************************************************/

#include "LPC11Axx.h"                        /* LPC11Axx definitions */
#include "type.h"
#include "error.h"
#include "hw_i2cd_rom_api.h"
#include "hw_i2cd_i2cmco.h"
#define I2CROMD_PRESENT
#include "rom_drivers.h"

#define I2C_INTERRUPT_TEST		0

//*** Global Data ***
const I2CD_API_T*  pI2cApi ;  //define pointer to type API function addr table
I2C_PARAM*  ptop;   // define pointer to param of type 1 structure
I2C_PARAM   s1;     // s1 is a structure of type I2C_PARAM
I2C_RESULT* ptor;   // define pointer to return values of structure
I2C_RESULT  s2;	    // s2 is a structure of type I2C_result

I2C_HANDLE_T *i2c_handle;
ErrorCode_t error_code;
   
volatile uint32_t size_in_bytes;
volatile uint32_t ver_num;
volatile uint32_t i2c_counter = 0;
volatile uint32_t i2c_rx_callback_cnt = 0;
volatile uint32_t i2c_tx_callback_cnt = 0;
volatile uint32_t i2c_done = FALSE;
volatile uint32_t i2c_state = SLAVE_STATE_IDLE;
volatile uint8_t I2C_Handle[100];
volatile uint8_t I2C_TxBuffer[MAX_BUF_SIZE];
volatile uint8_t I2C_RxBuffer[MAX_BUF_SIZE];

void slave_rx_callback ( void );
void slave_tx_callback ( void );

//*******************************************************************
void clear_buffer( void) {   // Clear buffer
  uint8_t i;
  
  for ( i = 0; i < MAX_BUF_SIZE; i++ )
  {
	I2C_TxBuffer[i] = 0;
	I2C_RxBuffer[i] = 0;
  } 
}

//*******************************************************************
void init_i2c( void) {   // enable P_clock and pins for I2C

  LPC_SYSCON->PRESETCTRL |= (0x1<<1);
  LPC_SYSCON->SYSAHBCLKCTRL |= ((1 << 16) | (1 << 6) | (1<<5));
  // Turn on the P_clk to the I2C blcok, 16 = IOCON, 6=GPIO, 5=I2C 
						  
  //enable two pins for I2C use as SCL and SDA
  LPC_IOCON->PIO0_2 &= ~0x3F;	/*  I2C I/O config */
  LPC_IOCON->PIO0_2 |= 0x01;		/* I2C SCL */
  LPC_IOCON->PIO0_3 &= ~0x3F;	
  LPC_IOCON->PIO0_3 |= 0x01;		/* I2C SDA */

  clear_buffer();
  ptop = &s1;    // addr of PARAM struct, s1 is assigned to pointer ptop
  ptor = &s2;    // addr of RESULT struct, s2 is assigned to pointer ptor
}

//*******************************************************************
void I2C_IRQHandler( void){	   // Application Program enables interrupts and calls ISR

  i2c_counter++; 	 // bump counter for debug
  pI2cApi->i2c_isr_handler( (I2C_HANDLE_T*) i2c_handle );  
}	  

//*******************************************************************
void I2c_slave_polling_test( void )
{
  clear_buffer();
  i2c_state = SLAVE_STATE_IDLE;
  ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
  ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
  error_code = pI2cApi->i2c_slave_receive_poll((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);

  if ( (I2C_RxBuffer[0] == SE95_ADDR) && (I2C_RxBuffer[1] == SE95_ID) )
  {
	clear_buffer();
	i2c_state = SLAVE_STATE_ID_READ;
	ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
	ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
	error_code = pI2cApi->i2c_slave_receive_poll((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
  }
    
  if ( I2C_RxBuffer[0] == SE95_ADDR+1 )
  {
	clear_buffer();
	I2C_TxBuffer[0] = SE95_ID_VALUE;
	ptop->num_bytes_send =1;
	ptop->num_bytes_rec = 0;
	ptop->buffer_ptr_send	= (uint8_t *)&I2C_TxBuffer[0];
	ptop->buffer_ptr_rec = NULL;
	error_code = pI2cApi->i2c_slave_transmit_poll((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
	i2c_state = SLAVE_STATE_ID_WRITTEN;
  }

  clear_buffer();
  ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
  ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
  error_code = pI2cApi->i2c_slave_receive_poll((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
  if ( (I2C_RxBuffer[0] == SE95_ADDR) && (I2C_RxBuffer[1] == SE95_CONFIG) )
  {
 	i2c_state = SLAVE_STATE_CONFIGURED;
  }

  clear_buffer();  		  
  ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
  ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
  error_code = pI2cApi->i2c_slave_receive_poll((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
  if ( (I2C_RxBuffer[0] == SE95_ADDR) && (I2C_RxBuffer[1] == SE95_TEMP) )
  {
	clear_buffer();
	i2c_state = SLAVE_STATE_TEMP_READ;
	ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
	ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
	error_code = pI2cApi->i2c_slave_receive_poll((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
  }
    
  if ( I2C_RxBuffer[0] == SE95_ADDR+1 )
  {
	clear_buffer();
	I2C_TxBuffer[0] = (SE95_TEMP_VALUE & 0xFF00) >> 8;
	I2C_TxBuffer[1] = (SE95_TEMP_VALUE & 0x00FF);
	ptop->num_bytes_send =2;
	ptop->num_bytes_rec = 0;
	ptop->buffer_ptr_send	= (uint8_t *)&I2C_TxBuffer[0];
	ptop->buffer_ptr_rec = NULL;
	error_code = pI2cApi->i2c_slave_transmit_poll((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
	i2c_state = SLAVE_STATE_TEMP_WRITTEN;
  }
  return;
}

//*******************************************************************
void slave_tx_callback( void )
{
  i2c_tx_callback_cnt++;
  if ( i2c_state == SLAVE_STATE_ID_READ )
  {
	/* slave transmit is done, ID has been sent to the master. Wait for
	following command from the master, it should be the CFG command. */
	clear_buffer();
	if ( pI2cApi->i2c_get_status((I2C_HANDLE_T*)i2c_handle) == 0 )
	{
	  i2c_state = SLAVE_STATE_ID_WRITTEN;
	}
	ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
	ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
	ptop->func_pt = (I2C_CALLBK_T)slave_rx_callback;  
	error_code = pI2cApi->i2c_slave_receive_intr((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
  }
  if ( i2c_state == SLAVE_STATE_TEMP_READ )
  {
	/* Slave transmit is done, TEMP value should be sent to the master already.
	Slave should be back to the IDLE state already. */
	clear_buffer();
	if ( pI2cApi->i2c_get_status((I2C_HANDLE_T*)i2c_handle) == 0 )
	{
	  i2c_state = SLAVE_STATE_TEMP_WRITTEN;
	  i2c_done = TRUE;
	}
	else
	{
	  /* If it comes here, something is wrong. */
	  ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
	  ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
	  ptop->func_pt = (I2C_CALLBK_T)slave_rx_callback;  
	  error_code = pI2cApi->i2c_slave_receive_intr((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
	}
  }
}

//*******************************************************************
void slave_rx_callback( void )
{
  i2c_rx_callback_cnt++;
  if ( (i2c_state == SLAVE_STATE_IDLE) 
	&& (I2C_RxBuffer[0] == SE95_ADDR) && (I2C_RxBuffer[1] == SE95_ID) )
  {
	/* Wait for ID command from the master. */
	clear_buffer();
	ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
	ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
	ptop->func_pt = (I2C_CALLBK_T)slave_rx_callback;  
	error_code = pI2cApi->i2c_slave_receive_intr((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
	i2c_state = SLAVE_STATE_ID_READ;
  }
  
  if ( (i2c_state == SLAVE_STATE_ID_READ) && (I2C_RxBuffer[0] == (SE95_ADDR+1)) )
  {
	/* Slave address with RD bit set arrives, read to do slave transmit and 
	send ID back to the master. */	
	clear_buffer();
	I2C_TxBuffer[0] = SE95_ID_VALUE;
	ptop->num_bytes_send =1;
	ptop->buffer_ptr_send	= (uint8_t *)&I2C_TxBuffer[0];
	ptop->func_pt = (I2C_CALLBK_T)slave_tx_callback;
	error_code = pI2cApi->i2c_slave_transmit_intr((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
  }
  
  if (i2c_state == SLAVE_STATE_ID_WRITTEN)
  {
	/* Wait for CFG command from master */
	clear_buffer();
	ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
	ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
	ptop->func_pt = (I2C_CALLBK_T)slave_rx_callback;  
	error_code = pI2cApi->i2c_slave_receive_intr((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
	i2c_state = SLAVE_STATE_CONFIGURED;
  }
  
  if ( (i2c_state == SLAVE_STATE_CONFIGURED ) 
	&& (I2C_RxBuffer[0] == SE95_ADDR) && (I2C_RxBuffer[1] == SE95_TEMP) )
  {
	/* The TEMP reading command has arrived, wait for repeated start and slave
	address with RD bit set. */
	clear_buffer();
	ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
	ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
	ptop->func_pt = (I2C_CALLBK_T)slave_rx_callback;  
	error_code = pI2cApi->i2c_slave_receive_intr((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
	i2c_state = SLAVE_STATE_TEMP_READ;
  }
  if ( (i2c_state == SLAVE_STATE_TEMP_READ ) 
	&& (I2C_RxBuffer[0] == (SE95_ADDR+1)) )
  {
	/* Slave address with RD bit set arrives, ready to do slave transmit. */	  
	clear_buffer();
	I2C_TxBuffer[0] = (SE95_TEMP_VALUE & 0xFF00) >> 8;
	I2C_TxBuffer[1] = (SE95_TEMP_VALUE & 0x00FF);
	ptop->num_bytes_send =2;
	ptop->buffer_ptr_send	= (uint8_t *)&I2C_TxBuffer[0];
	ptop->func_pt = (I2C_CALLBK_T)slave_tx_callback;
	error_code = pI2cApi->i2c_slave_transmit_intr((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
	if ( pI2cApi->i2c_get_status((I2C_HANDLE_T*)i2c_handle) == 0 )
	{
	  i2c_state = SLAVE_STATE_TEMP_WRITTEN;
	  i2c_done = TRUE;
	}
  }  
  return;
}

//*******************************************************************
void I2c_slave_interrupt_test( void )
{
  ptop->num_bytes_rec = MAX_BUF_SIZE;		/* max buffer size */
  ptop->buffer_ptr_rec = (uint8_t *)&I2C_RxBuffer[0];
  ptop->func_pt = (I2C_CALLBK_T)slave_rx_callback;  
  i2c_done = FALSE;
  error_code = pI2cApi->i2c_slave_receive_intr((I2C_HANDLE_T*)i2c_handle,  
		(I2C_PARAM*)ptop, (I2C_RESULT*)ptor);
  while ( i2c_done == FALSE );
  
  return;
}

/* In slave mode, the slave acts exactly as a NXP Temp SE95 sensor */
/* Main Program */

int main (void) 
{
  SystemCoreClockUpdate();

  pI2cApi = ROM_DRIVERS_PTR->pI2CD; 
	    	  
  size_in_bytes =  pI2cApi->i2c_get_mem_size(); 

#if 0
  if ( RAMBLOCK_H < (size_in_bytes /4) ) {
     while (1) ;   // here forever if not enough ram for handle
  }
#endif
 				   	    
  init_i2c() ;   //   assign pins for I2C, set Pclock to I2C 
				    
  i2c_handle = pI2cApi->i2c_setup( LPC_I2C_BASE, (uint32_t *)&I2C_Handle[0] );
  // enable SLAVE ADR, Mask is not set
  pI2cApi->i2c_set_slave_addr( (I2C_HANDLE_T*)i2c_handle, SE95_ADDR, 0x00000000) ;

  ver_num = pI2cApi->i2c_get_firmware_version();
  
  /* Interrupt or polling, this example emulate the NXP semi SE95 sensor.
  The sequence from the master is:
  (1) Get ID and Revision number, 
  Start, 0x90, 0x05(ID), Repeated Start, 0x91(R), 0xA1, Stop
  (2) Configure temp sensor to default value, 
  Start, 0x90, 0x01(CFG), 0x00(Default), Stop
  (3) Get TEMP Reading, 
  Start, 0x90, 0x00(TEMP), Repeated Start, 0x91(R), 0x15, 0x5C, Stop
  */
#if I2C_INTERRUPT_TEST
  NVIC_EnableIRQ(I2C_IRQn);
  I2c_slave_interrupt_test();
#else
  I2c_slave_polling_test();
#endif

  while ( 1 );			/* Never exit from main for easy debugging. */

}
