
#include "LPC43xx.h"
#include "i2c.h"

/********************************************************************//**
 * @brief		Initializes the I2Cx peripheral with specified parameter.
 * @param[in]	I2Cx	I2C peripheral selected, should be
 * 					- LPC_I2C0	:I2C0 peripheral
 * 					- LPC_I2C1	:I2C1 peripheral
 * @param[in]	clockrate Target clock rate value to initialized I2C
 * 				peripheral (Hz)
 * @return 		None
 *********************************************************************/
void I2C_Init(LPC_I2Cn_Type *I2Cx, uint32_t clockrate)
{
	uint32_t tem;
	
	// assume clock rate for APB1 and APB3 are set elsewhere to 12 MHz of either IRC or Crystal oscillator
	
	LPC_SCU->SFSI2C0 = SFSI2C0_CONFIGURE_STANDARD_FAST_MODE;

    /* Set clock rate */
	if(clockrate<1000)	//make sure SCLH,SCLL not exceed its 16bit value
		return;
	tem = 12000000 / clockrate;
	I2Cx->SCLH = (uint32_t)(tem / 2);
	I2Cx->SCLL = (uint32_t)(tem - I2Cx->SCLH);
	
    /* Set I2C operation to default */
    I2Cx->CONCLR = (I2C_I2CON_AA |I2C_I2CON_SI | I2C_I2CON_STA | I2C_I2CON_I2EN);
    
	/* Enable Slave I2C operation */
	I2Cx->CONSET = I2C_I2CON_I2EN;
}



// if regDATA == -1 then we are just setting up the address for a read

void I2C_STOP(LPC_I2Cn_Type *I2Cx) {
	I2Cx->CONSET = I2C_I2CON_STO;                                       //set master's STO bit
    I2Cx->CONCLR = I2C_I2CON_SI;                                        //clear master's SI bit
    while( (I2Cx->CONSET & I2C_I2CON_STO)  == 1);			         		//wait until master's STO is set
}

void I2C_WAIT4CHANGE(LPC_I2Cn_Type *I2Cx) {
	int timeout;
	
	timeout=40000;
    while( (I2Cx->CONSET & I2C_I2CON_SI)  == 0){                   		//wait until master's SI bit is set     
    	if (timeout-- < 0) 
				while (1);		// either crash and burn here for debugging
//				break;			// or break to continue
	}
	
}

int I2C_Write(LPC_I2Cn_Type *I2Cx, int slaveADDR, int regADDR, int regDATA) {

     // JD, clear STA, SI and STO for master side
    I2Cx->CONCLR = I2C_I2CON_SI ;

       /*--set master's AA bit --------------------------------------*/
    I2Cx->CONSET = I2C_I2CON_AA;

       /*--master I2C set START condition bit to be transmitted-----*/
    I2Cx->CONSET = I2C_I2CON_STA;  //set master's STA bit // JD, it works. slcon is 0x60 now

	I2C_WAIT4CHANGE(I2Cx);                   //wait until master's SI is set

       /*--I2C master load SLA+W--------------------------------------*/
    I2Cx->DAT = slaveADDR & ~1 ;	
    I2Cx->CONCLR = I2C_I2CON_STA;                      //clear master's STA bit
    I2Cx->CONCLR = I2C_I2CON_SI;                       //clear master's SI bit

	I2C_WAIT4CHANGE(I2Cx);                   //wait until master's SI is set

	if ((I2Cx->STAT & I2C_STAT_CODE_BITMASK) == I2C_I2STAT_M_TX_SLAW_NACK) {
		I2C_STOP(I2Cx);
		return -1;
	}

	/*--I2C slave transmitter load one data byte-------------------*/
  	I2Cx->DAT = regADDR;	
  	I2Cx->CONCLR = I2C_I2CON_SI;				//clear master's SI bit
  	
	I2C_WAIT4CHANGE(I2Cx);                   //wait until master's SI is set
  	
	if ((I2Cx->STAT & I2C_STAT_CODE_BITMASK) == I2C_I2STAT_M_TX_DAT_NACK) {
		I2C_STOP(I2Cx);
		return -1;
	}

	if (regDATA == -1) {		// we're done
//		I2C_STOP(I2Cx);
	    return 0;
	}

	/*--I2C slave transmitter load one data byte-------------------*/
  	I2Cx->DAT = regDATA;	
  	I2Cx->CONCLR = I2C_I2CON_SI;				//clear master's SI bit
  	
	I2C_WAIT4CHANGE(I2Cx);                   //wait until master's SI is set
  	
	if ((I2Cx->STAT & I2C_STAT_CODE_BITMASK) == I2C_I2STAT_M_TX_DAT_NACK) {
		I2C_STOP(I2Cx);
		return -1;
	}
       /*--I2C master load STOP condition to be transmitted-----------*/
	I2C_STOP(I2Cx);
		
    return 0;	
}


int I2C_Read(LPC_I2Cn_Type *I2Cx, int slaveADDR, int regADDR)  {
	
	if (I2C_Write(I2Cx, slaveADDR, regADDR, -1)) return -1;
	

     // JD, clear STA, SI and STO for master side
    I2Cx->CONCLR = I2C_I2CON_SI ;

       /*--set master's AA bit --------------------------------------*/
    I2Cx->CONSET = I2C_I2CON_AA;

       /*--master I2C set START condition bit to be transmitted-----*/
    I2Cx->CONSET = I2C_I2CON_STA;  		//set master's STA bit 

	I2C_WAIT4CHANGE(I2Cx);                  //wait until master's SI is set

       /*--I2C master load SLA+R-------------------------------------*/
    I2Cx->DAT = slaveADDR | 1;	
        
    I2Cx->CONCLR = I2C_I2CON_STA;        //clear master's STA bit
    I2Cx->CONCLR = I2C_I2CON_SI;         //clear master's SI bit

	I2C_WAIT4CHANGE(I2Cx);                  //wait until master's SI is set
	if ((I2Cx->STAT & I2C_STAT_CODE_BITMASK) == I2C_I2STAT_M_RX_SLAR_NACK) {
		I2C_STOP(I2Cx);
		return -1;
	}

		/*--I2C slave transmitter load one data byte-------------------*/
	I2Cx->CONCLR = I2C_I2CON_SI;				//clear master's SI bit

	I2C_WAIT4CHANGE(I2Cx);                   //wait until master's SI is set
		
	I2C_STOP(I2Cx);
	
	return I2Cx->DAT;

}
