#include "LPC43xx.h"

#define USE_SPIFI_LIB          // needs to be in C command line

#include "spifi_rom_api.h"
#include "system_LPC43xx.h"


/**********************************************************************
 ** SPIFI Flash comes here, copies itself to RAM and jumps there
 **   someday it will copy all RAM so multiple CPUs supported
 **
 ** the idea is to be able to take a snapshot of RAM -> SPIFI 
 **********************************************************************/

__asm int gotoR0(int x) {	
 	bx	r0
} 	

__asm int whereAMi() {
	mov		r0,pc
	bx 		lr
}
 
#if 0  // LPC4357
#define RAM1start	0x10000000
#define RAM1size	0x8000
#define RAM2start	0x10080000
#define RAM2size	0x9000
#define RAM3start	0x20000000
#define RAM3size	0x8000
#define RAM4start	0
#define RAM4size	0



#else	// LPC4300
#define RAM1start	0x10000000
#define RAM1size	0x20000
#define RAM2start	0x10080000
#define RAM2size	0x12000
#define RAM3start	0x20000000
#define RAM3size	0x10000
#define RAM4start	0
#define RAM4size	0
#endif

// use 384K for snapshots of RAM, eventually allow lowspeed shared code to reside in the rest of the space

#define SPIFIsaveCODE	(0x80000000)			// upto 688K space
#define SPIFIsaveRAM1	(0x80100000 - 0x58000)	// 128K space
#define SPIFIsaveRAM2	(0x80100000 - 0x38000)	// 128K space
#define SPIFIsaveRAM3	(0x80100000 - 0x18000)	// 64K space
#define SPIFIsaveRAM4	(0x80100000 - 0x8000)	// 32K space

#define codeSTART	RAM1start		// sutomate this someday	

#define SPIFIstart	0x80000000	   		// grab this from SPIFI table obj someday


/////////////////////////////////////////////////////////////////////////////////
//
//	checkSPIFIstart called at startup -- if its called from other than SPIFI memory it just returns
//
//	if it is SPIFI, then copy the contents of SPIFI back into RAM and branch to it
//
//	DOH -- as this routine overwrites the stack it has to be inline code
//
/////////////////////////////////////////////////////////////////////////////////
 
void checkSPIFIstart(void) {

#ifndef RUN_IN_SPIFI
	return;
#else
	unsigned int *ram, *spifi;

	if ((whereAMi() & 0xff000000) != 0x80000000) return;	// if not running in SPIFI then just return

// from here on inline code only and only register access

	ram = (unsigned int *)RAM1start;
	spifi = (unsigned int *)SPIFIsaveRAM1;

	do {
		*ram++ = *spifi++;
	} while (ram < (unsigned int *)(RAM1start + RAM1size));
		
	ram = (unsigned int *)RAM2start;
	spifi = (unsigned int *)SPIFIsaveRAM2;

	do {
		*ram++ = *spifi++;
	} while (ram < (unsigned int *)(RAM2start + RAM2size));
	
	ram = (unsigned int *)RAM3start;
	spifi = (unsigned int *)SPIFIsaveRAM3;

	do {
		*ram++ = *spifi++;
	} while (ram < (unsigned int *)(RAM3start + RAM3size));

	// yes the next routine writes to the stack, but it never returns

	gotoR0 (((*(int *)0x80000004) & 0x0000FFFF) | codeSTART );	// this only works if the spifi image and ram image are the same for low memory which should be true.
#endif	
}
/**********************************************************************
 ** take a snapshot of RAM and move to Flash
 **
 ** use Keil to set PC to takeSnapshot routine and run
 **********************************************************************/

SPIFIobj obj;
SPIFI_RTNS * pSpifi;
SPIFIopers opers; 

#ifdef USE_SPIFI_LIB
extern SPIFI_RTNS spifi_table;
#endif	/* USE_SPIFI_LIB */

void copyRAMblock(unsigned int start, unsigned int SPIFIaddr, unsigned int size) {
	int i;
	
	opers.length = PROG_SIZE;
	opers.scratch = NULL;
	opers.protect = 0;
	opers.options = S_CALLER_ERASE;
	for ( i = 0 ; i < size / PROG_SIZE; i++ )
	{
        /* Write */
	    opers.dest = (char *)( SPIFIaddr + (i*PROG_SIZE) );
	    if (pSpifi->spifi_program (&obj, (char *)(start + (i*PROG_SIZE)), &opers)) 
			while (1);	 	// crash and burn on fail
	    /* Verify */
//	    if (memcmp((void *)SPIFIaddr,(void *)(start + (i*PROG_SIZE)),PROG_SIZE)) printf("a few expected miscompare %x\n",start + (i*PROG_SIZE));
	}	
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/**********************************************************************
 ** take a snapshot of RAM and move to Flash
 **
 ** use Keil to set PC to takeSnapshot routine and run from there, which copies the RAM contents to SPIFI
 **********************************************************************/
//  currently Keil shows a couple illegal accesses
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

void takeSnapshot(void) {
	int i;
	
	SystemInit(); 	// to take a SPIFI snapshot use Keil to set the PC here -- eventually add boot RAM selection

#ifdef USE_SPIFI_LIB
    pSpifi = &spifi_table;
#else
    pSpifi = (SPIFI_RTNS *)(SPIFI_ROM_TABLE);
#endif

	/* Initialize SPIFI driver */
	if (pSpifi->spifi_init(&obj, 3, S_RCVCLK | S_FULLCLK, 12)) while (1);
		
	/* set SPIFI clock */
	LPC_CGU->BASE_SPIFI0_CLK = 1<<24 | 1<<11; /* IRC 12 MHz is good enough for us */

	/* set up SPIFI I/O (undocumented bit 7 set as 1, Aug 2 2011) */
	LPC_SCU->SFSP3_3 = 0xF3; /* high drive for SCLK */
	/* IO pins */
	LPC_SCU->SFSP3_4=LPC_SCU->SFSP3_5=LPC_SCU->SFSP3_6=LPC_SCU->SFSP3_7 = 0xD3;
	LPC_SCU->SFSP3_8 = 0x13; /* CS doesn't need feedback */

		/* Erase Entire SPIFI Device if required */
	for ( i = 0 ; i < obj.memSize / 4; i+=4 )
	{
	    if ( *((uint32_t *)(obj.base+i)) != 0xFFFFFFFF )
		{
		    opers.dest = (char *)(obj.base);
		    opers.length = obj.memSize;
      	    opers.scratch = NULL;
			opers.options = S_VERIFY_ERASE;
		    /* Erase Device */
			if (pSpifi->spifi_erase(&obj, &opers)) while (1);
			break;
		}
	}


	copyRAMblock(RAM1start, SPIFIsaveRAM1, RAM1size);
	copyRAMblock(RAM2start, SPIFIsaveRAM2, RAM2size);
	copyRAMblock(RAM3start, SPIFIsaveRAM3, RAM3size);
#if RAM4start != 0	
	copyRAMblock(RAM4start, SPIFIsaveRAM4, RAM4size);		// M0 memory
#endif	


	// copy code block vectors but change start address to SPIFI
	i = *(int *) (codeSTART + 4);
	*(int *) (codeSTART + 4) = (i & 0x0000ffff) | 0x80000000;	// reset vector in SPIFI

#if (codeSTART & 0xFFFF0000) == RAM1start
	copyRAMblock(RAM1start, SPIFIsaveCODE, RAM1size);			
#elif (codeSTART & 0xFFFF0000) == RAM2start
	copyRAMblock(RAM2start, SPIFIsaveCODE, RAM2size);		
#elif (codeSTART & 0xFFFF0000) == RAM3start
	copyRAMblock(RAM3start, SPIFIsaveCODE, RAM3size);	
#else
  #error  --  boot code not supported in this space		
#endif

	
	while(1) {
		// hang up here		
	}		
	
}

