/*****************************************************************************
 *   eeprom.c:  Driver for the 24LC08 EEPROM
 *
 *   Copyright(C) 2009, Embedded Artists AB
 *   All rights reserved.
 *
 ******************************************************************************/

/*
 * NOTE: I2C must have been initialized before calling any functions in this
 * file.
 */

/******************************************************************************
 * Includes
 *****************************************************************************/
#include "application.h"
#if MC_BOARD_ENABLE_EEPROM == 1
/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/

#ifndef MIN
#define MIN(x, y) ((x) < (y) ? (x) : (y))
#endif

#define EEPROM_I2C_ADDR    (0x50 << 1)


#define EEPROM_TOTAL_SIZE 4096
#define EEPROM_PAGE_SIZE    8

#define E_WRITE_LEN 100

/******************************************************************************
 * External global variables
 *****************************************************************************/
extern volatile uint32_t I2CCount;
extern volatile uint8_t I2CMasterBuffer[I2C_BUFSIZE];
extern volatile uint8_t I2CSlaveBuffer[I2C_BUFSIZE];
extern volatile uint32_t I2CMasterState;
extern volatile uint32_t I2CReadLength, I2CWriteLength;

extern volatile uint32_t SysTick_cntr;

/******************************************************************************
 * Local variables
 *****************************************************************************/


/******************************************************************************
 * Local Functions
 *****************************************************************************/

static void eepromDelay(void)
{
	//wait for 20 ms
	SysTick_cntr = 0;
	while ( SysTick_cntr < 20 );
	SysTick_cntr = 0;
}

/******************************************************************************
 * Public Functions
 *****************************************************************************/

/******************************************************************************
 *
 * Description:
 *    Test the EEPROM, return 1 if OK
 *
 *****************************************************************************/
uint8_t eeprom_test (void)
{
	uint16_t i = 0;
    uint16_t offset = 240;
    static uint8_t b[E_WRITE_LEN];
    int16_t len = 0;

    //Initialize test vector
    for (i = 0; i < E_WRITE_LEN; i++) {
        b[i] = (uint8_t)(i+3);
    }

    //Write data
    len = eeprom_write(b, offset, E_WRITE_LEN);

    if (len != E_WRITE_LEN)
        return 0;

    //delay 2 sec
    SysTick_cntr = 0;
   	while ( SysTick_cntr < 2000 );
   	SysTick_cntr = 0;

    memset(b, 0, E_WRITE_LEN);

    //Read data back
    len = eeprom_read(b, offset, E_WRITE_LEN);

    if (len != E_WRITE_LEN)
        return 0;

    //Verify data
    for (i = 0; i < E_WRITE_LEN; i++) {
        if (b[i] != (uint8_t)(i+3))
            return 0;
    }

    return 1;
}

/******************************************************************************
 *
 * Description:
 *    Initialize the EEPROM Driver
 *
 *****************************************************************************/
void eeprom_init (void)
{

}

/******************************************************************************
 *
 * Description:
 *    Read from the EEPROM
 *
 * Params:
 *   [in] buf - read buffer
 *   [in] offset - offset to start to read from
 *   [in] len - number of bytes to read
 *
 * Returns:
 *   number of read bytes or -1 in case of an error
 *
 *****************************************************************************/
int16_t eeprom_read(uint8_t* buf, uint16_t offset, uint16_t len)
{
    volatile uint16_t i = 0;

    if (len > EEPROM_TOTAL_SIZE || offset+len > EEPROM_TOTAL_SIZE) {
        return -1;
    }

    /* Write SLA(W), address and two data bytes */
    I2CWriteLength = 3;
    I2CReadLength = len;
    I2CMasterBuffer[0] = EEPROM_I2C_ADDR;
    I2CMasterBuffer[1] = ((offset >> 8) & 0xff);
    I2CMasterBuffer[2] = (offset & 0xff);
    I2CMasterBuffer[3] = EEPROM_I2C_ADDR  | RD_BIT;

    I2CEngine();

//    for ( i = 0; i < 0x2000; i++);

//    I2CRead((addr << 1), buf, len);

    /* Write SLA(W), address, SLA(R), and read two bytes back. */
//    I2CWriteLength = 0;
//    I2CReadLength = len;
//    I2CMasterBuffer[0] = EEPROM_I2C_ADDR  | RD_BIT;
//    I2CMasterBuffer[1] = ((offset >> 8) & 0xff);
//    I2CMasterBuffer[2] = (offset & 0xff);

//    I2CEngine();

    memcpy((void*)buf, (void*)I2CSlaveBuffer, len);

    return len;

}

/******************************************************************************
 *
 * Description:
 *    Write to the EEPROM
 *
 * Params:
 *   [in] buf - data to write
 *   [in] offset - offset to start to write to
 *   [in] len - number of bytes to write
 *
 * Returns:
 *   number of written bytes or -1 in case of an error
 *
 *****************************************************************************/
int16_t eeprom_write(uint8_t* buf, uint16_t offset, uint16_t len)
{
    int16_t written = 0;
    uint16_t wLen = 0;
    uint16_t off = offset;

    if (len > EEPROM_TOTAL_SIZE || offset+len > EEPROM_TOTAL_SIZE) {
        return -1;
    }

    wLen = EEPROM_PAGE_SIZE - (off % EEPROM_PAGE_SIZE);
    wLen = MIN(wLen, len);

    while (len) {
        /* Write SLA(W), address and led+2 data bytes */
        I2CWriteLength = 3 + wLen;
        I2CReadLength = 0;
        I2CMasterBuffer[0] = EEPROM_I2C_ADDR;
        I2CMasterBuffer[1] = ((off >> 8) & 0xff);
        I2CMasterBuffer[2] = (off & 0xff);
        memcpy(&I2CMasterBuffer[3], (void*)&buf[written], wLen);

        I2CEngine();

        /* delay to wait for a write cycle */
        eepromDelay();

        len     -= wLen;
        written += wLen;
        off     += wLen;

        wLen = MIN(EEPROM_PAGE_SIZE, len);
    }

    return written;
}
#endif //MC_BOARD_ENABLE_EEPROM
