/**
 * @file		uda1380.c
 * @purpose		Driver code for UDA1380 audio codec used on RDB1768v2
 * @version		0.3
 * @date		14. 02. 2012
 * @author		CT-PIM/NXP
 *----------------------------------------------------------------------------
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 **********************************************************************/

/**********************************************************************/
/* Include files                                                      */
/**********************************************************************/
#include "LPC17xx.h"
#include "lpc_types.h"
#include "lpc17xx_pinsel.h"
#include "lpc17xx_i2c.h"
#include "lpc17xx_i2s.h"

#include "uda1380.h"
#include <stdlib.h>

/** Used I2C device as slave definition */
#define USEDI2CDEV_M		0

/** Max buffer length */
#define BUFFER_SIZE			0x10
#define AUDIO_INIT_DELAY    2000

#define ECHO_RECORDING
//#define REC_FROM_LINE_IN

#if (USEDI2CDEV_M == 0)
#define I2CDEV_M LPC_I2C0
#elif (USEDI2CDEV_M == 2)
#define I2CDEV_M LPC_I2C2
#else
#error "Master I2C device not defined!"
#endif

I2C_M_SETUP_Type transferMCfg;
uint8_t I2C_data_transfer[3];

extern long int gQ31_Peak_L, gQ31_Peak_R;
extern long int gQ31_Peak[2];

long int Q31_I2S_Left, Q31_I2S_Right;

struct uda1380_setup
{
	UDA1380_Registers reg;
	uint16_t val;
};

/* Codec Presetings */
static const struct uda1380_setup uda1380_init[] =
{
  {UDA1380_REG7F_SW_RESET, 0xFFFF},
  {UDA1380_REG01_I2S, 0x0000},
  {UDA1380_REG03_ANAMIX, 0x3F3F},
  {UDA1380_REG13_MSTRMUTE, 0x0000},
#ifdef REC_FROM_LINE_IN
  {UDA1380_REG22_ADC, 0x00},
#else
  {UDA1380_REG22_ADC, REG22_SEL_LNA_ENABLE | REG22_SEL_MIC_ENABLE | REG22_SKIP_DCFIL},
#endif
  {UDA1380_REG21_PGA, 0x00},
};

static const struct uda1380_setup playback_enable[] =
{
  {UDA1380_REG00_EVALCLK, (1<<9) | (1<<8) | (1<<4)},
  {UDA1380_REG02_PWRCTRL, REG02_PON_BIAS | REG02_PON_DAC | REG02_PON_HP | (1<<15)},
};

static const struct uda1380_setup playback_disable[] =
{
  {UDA1380_REG00_EVALCLK, (1<<9) | (1<<8) | (1<<4)},
  {UDA1380_REG02_PWRCTRL, REG02_PON_BIAS | REG02_PON_DAC | REG02_PON_HP | (1<<15)},
};

static const struct uda1380_setup recording_enable[] =
{
#ifdef ECHO_RECORDING
  {UDA1380_REG02_PWRCTRL, (1<<15) | (1 << 13) | (1 << 10) | ( 1 << 8) | (1 << 7) | (1 << 6) | (1 << 4) | (1 << 3) | (1<<2) | (1 << 1) | (1<<0)},
  {UDA1380_REG00_EVALCLK, (1<<11) | (1<<10) | (1<<9) | (1<<8) | (1<<4) | (1<<5)},
#else
  {UDA1380_REG02_PWRCTRL, (1<<15) | ( 1 << 8) | (1 << 7) | (1 << 6) | (1 << 4) | (1 << 3) | (1<<2) | (1 << 1) | (1<<0)},
  {UDA1380_REG00_EVALCLK, (1<<11) | (1<<10)| (1<<5)},
#endif
};

static const struct uda1380_setup recording_disable[] =
{
  {UDA1380_REG02_PWRCTRL, 0x00},
  {UDA1380_REG00_EVALCLK, 0x00},
};

/**********************************************************************/
/* Local Functions prototypes                                         */
/**********************************************************************/
static int32_t UDA1380_SetRegister(UDA1380_Registers reg, uint32_t value);
static int uda1380_SetupData(const struct uda1380_setup *preset, int cnt);

/**********************************************************************/
/* Public Functions definitions                                       */
/**********************************************************************/
int UDA1380_EnableAudioIn(int32_t enable)
{
  const struct uda1380_setup *preset = enable ? recording_enable : recording_disable;
  int cnt = enable ? sizeof(recording_enable) : sizeof(recording_disable);
  cnt /= sizeof(*preset);
  return uda1380_SetupData(preset, cnt);
}

int UDA1380_EnableAudioOut(int32_t enable)
{
  const struct uda1380_setup *preset = enable ? playback_enable : playback_disable;
  int cnt = enable ? sizeof(playback_enable) : sizeof(playback_disable);
  cnt /= sizeof(*preset);
  return uda1380_SetupData(preset, cnt);
}

void UDA1380_CodecInit(void)
{
	PINSEL_CFG_Type PinCfg;

	/* I2C block ------------------------------------------------------------------- */
	/*
	 * Init I2C pin connect
	 */
	PinCfg.OpenDrain = 0;
	PinCfg.Pinmode = 0;
	PinCfg.Funcnum = 1;
	PinCfg.Pinnum = 27;
	PinCfg.Portnum = 0;
	PINSEL_ConfigPin(&PinCfg);
	PinCfg.Pinnum = 28;
	PINSEL_ConfigPin(&PinCfg);

	/* Initialize Master I2C peripheral */
	I2C_Init(I2CDEV_M, 400000);

	/* Enable Master I2C operation */
	I2C_Cmd(I2CDEV_M, ENABLE);

	/* Start I2C slave device first */
	transferMCfg.sl_addr7bit = UDA1380_ADDR;
	transferMCfg.tx_data = I2C_data_transfer ;
	transferMCfg.tx_length = sizeof(I2C_data_transfer);
	transferMCfg.rx_data = NULL;
	transferMCfg.rx_length = 0;
	transferMCfg.retransmissions_max = 3;

	uda1380_SetupData(uda1380_init, sizeof(uda1380_init)/sizeof(*uda1380_init));
}

/**********************************************************************/
/* Local Functions definitions                                        */
/**********************************************************************/
int32_t UDA1380_SetRegister(UDA1380_Registers reg, uint32_t value)
{
	I2C_data_transfer[0] = reg;
	I2C_data_transfer[1] = (uint8_t)((value >> 8) & 0xFF);
	I2C_data_transfer[2] = (uint8_t)(value & 0xFF);
	if (I2C_MasterTransferData(I2CDEV_M, &transferMCfg, I2C_TRANSFER_POLLING)== SUCCESS)
	{
		return(value);
	}
	else
	{
		return(~value);
	}
}

static int uda1380_SetupData(const struct uda1380_setup *preset, int cnt)
{
  int error = 0, i;
  for (i = 0; i < cnt; i ++)
  {
    error += UDA1380_SetRegister(preset->reg, (uint32_t)preset->val) != (uint32_t)preset->val;
    preset++;
  }
  return error;
}
