/**
 * @file		main.c
 * @purpose		Example application of vocoder using Speex CODEC.
 * @version		0.3
 * @date		14. 02. 2012
 * @author		CT-PIM/NXP
 *----------------------------------------------------------------------------
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 **********************************************************************/

/**********************************************************************/
/* Include files                                                      */
/**********************************************************************/
#include "lpc17xx_libcfg.h"
#include "lpc17xx_gpio.h"
#include "lpc17xx_timer.h"
#include "lpc17xx_clkpwr.h"

#include "lpc17xx_vocoder.h"
#include "speex128x57.h"
#include "lcd.h"


uint8_t app_loop, app_action;
uint8_t cursor_position;

#define ACTION_NONE 0
#define ACTION_REC  1
#define ACTION_PLAY 2

/************************** PRIVATE FUNCTIONS *************************/
void EINT3_IRQHandler(void);
void delay (void);

void UpdateMenu(void)
{
   if (app_action == ACTION_NONE)
   {
	   LCD_PrintString(50,90," Record     ", COLOR_WHITE, COLOR_BLACK);
	   LCD_PrintString(50,105," Play       ", COLOR_WHITE, COLOR_BLACK);
	   LCD_PrintString(34, 90 + 15*cursor_position,">>", COLOR_WHITE, COLOR_BLACK);
	   LCD_PrintString(34, 90 + 15*(cursor_position^1),"  ", COLOR_WHITE, COLOR_BLACK);
   }
   else if (app_action == ACTION_REC)
   {
	   LCD_PrintString(34,90,"Recording...", COLOR_RED, COLOR_BLACK);
	   LCD_PrintString(34,105,"             ", COLOR_WHITE, COLOR_BLACK);
   }
   else if (app_action == ACTION_PLAY)
   {
	   LCD_PrintString(34,90,"            ", COLOR_WHITE, COLOR_BLACK);
	   LCD_PrintString(34,105,"Playing...   ", COLOR_RED, COLOR_BLACK);
   }
}

void EINT3_IRQHandler(void)
{
	if (GPIO_GetIntStatus(0, 5, 1))
	{
		cursor_position = cursor_position^1;
		UpdateMenu();
		delay();
		GPIO_ClearInt(0,(1<<5));
	}
	else if (GPIO_GetIntStatus(0, 6, 1))
	{
		if (cursor_position == 0)
		{
			app_action = ACTION_REC;
		}
		else
		{
			app_action = ACTION_PLAY;
		}
		delay();
		GPIO_ClearInt(0,(1<<6));
	}
}

/*********************************************************************//**
 * @brief		Delay function
 * @param[in]	None
 * @return 		None
 **********************************************************************/
void delay (void) {
  unsigned int i;

  for (i = 0; i < 0x200000; i++) {
  }
}

/**
 * @brief Main program body
 */
int c_entry(void)
{
   SPX_CODEC_CFG_Type spx_config;

   SystemCoreClockUpdate();

   /* Speex CODEC configuration */
   spx_config.quality    = 4;                    /* Encoder parameter */
   spx_config.complexity = 1;                    /* Encoder parameter */
   spx_config.vbr        = SPX_VBR_DISABLED;     /* Encoder parameter */
   spx_config.enh        = SPX_ENHANCER_ENABLED; /* Decoder parameter */
   spx_Init(&spx_config);

   /* Audio input/output initialization */
   AudioOutput_Init();
   AudioInput_Init();

   /* Initialize the ST7637 LCD Controller/driver for use */
   LCDdriver_initialisation();

   /* Plot Speex logo at top of screen */
   LCD_PlotBitmap (0, 0, speex128x57_pixel_data ,speex128x57_width, speex128x57_height);

   LCD_PrintString(10,57,"LPC17xx Speex", COLOR_WHITE, COLOR_BLACK);
   LCD_PrintString(38,70,"Vocoder", COLOR_WHITE, COLOR_BLACK);

   GPIO_IntCmd(0,(1<<5) | (1<<6),1);
   NVIC_EnableIRQ(EINT3_IRQn);

   GPIO_SetDir(1, (1<<25), 1);
   app_loop = 1;
   cursor_position = 0;
   app_action = ACTION_NONE;

   while(1)
   {
	   UpdateMenu();
	   while(app_action == ACTION_NONE);

	   UpdateMenu();
	   if (app_action == ACTION_REC)
	   {
		   GPIO_SetValue(1,(1<<25));
		   spx_StartEncoder();
		   GPIO_ClearValue(1, (1<<25));
		   app_action = ACTION_NONE;
	   }
	   if (app_action == ACTION_PLAY)
	   {
		   GPIO_SetValue(1,(1<<25));
		   spx_StartDecoder();
		   GPIO_ClearValue(1, (1<<25));
		   app_action = ACTION_NONE;
	   }
   }
   return 1;
}

/* Support required entry point for other toolchain */
int main (void)
{
   return c_entry();
}

#ifdef  DEBUG
/*******************************************************************************
* @brief	Reports the name of the source file and the source line number
* 		where the CHECK_PARAM error has occurred.
* @param[in]	file Pointer to the source file name
* @param[in]    line assert_param error line source number
* @return		None
*******************************************************************************/
void check_failed(uint8_t *file, uint32_t line)
{
   /* User can add his own implementation to report the file name and line number,
   ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

   /* Infinite loop */
   while(1);
}
#endif

