/****************************************************************************
*   $Id:: main.c 6749 2011-03-08 18:38:40Z nxp28548                $        *
*   Project: Division API Example					    *
*									    *
*   Description:							    *
*			This project demonstrates LPC122x ROM based Division*
*
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/
#include "LPC122x.h"			/* LPC12xx Peripheral Registers */
#include "LPC_ROM_DIV.h"
#include <stdio.h>
#include <string.h>

#define BUF_SIZE 40

/*******************************************************************************
Function Declarations
*******************************************************************************/

/* Use Timer to measure the cycles needed to perform division */
static void PerformanceTest(void);

/* Numeric computation tests, check validity of results */
static void uDivisionTest(void);
static void iDivisionTest(void);

/*******************************************************************************
Global Variable Declarations
*******************************************************************************/

/* Notice all Values are odd, to prevent shifting */
const uint32_t unsignedOperandTable[] = {
  0x00000001,
  0x00000201,
  0x00002001,
  0x0000FFFF,
  0x00020001,
  0x00040001,
  0x00ABCDEF,
  0x00F0A597,
  0x00FFFFFE,
  0x7FFFFFFF,
};

const int32_t signedOperandTable[] ={
  1023,
  513,
  127,
  1,
  -1,
  -255,
  -2047,
  -65535,
};

const uint32_t expectedResultsUnsigned[]={
  0x7FFFFFFF,
  0x003FE00F,
  0x0003FFE0,
  0x00008000,
  0x00003FFF,
  0x00001FFF,
  0x000000BE,
  0x00000088,
  0x00000080,
  0x00000001,
};

const int32_t expectedResultsSigned[]={
  -64,-127,-516,-65535,65535,257,32,1};

/* Run the tests */
int main (void)
{  
  /* Setup Timer to performance metrics */
  LPC_SYSCON->SYSAHBCLKCTRL |= (1<<9);
  
  printf("Device ID:0x%08X\r\n", LPC_SYSCON->DEVICE_ID);
  
  printf("Starting Performance Test.\r\n");  
  PerformanceTest();
  printf("Performance Test Complete.\r\n");
  
  printf("\r\nStarting Unsigned Test.\r\n");
  uDivisionTest();
  printf("Unsigned Test Complete.\r\n");
  
  printf("\r\nStarting Signed Test.\r\n");
  iDivisionTest();
  printf("Signed Test Complete.\r\n");
  
  return 0;
}

/* Gather performance metrics using Timer Counter */
static void PerformanceTest(void)
{
  /*
  Results are not actually used, and must be declared volatile
  to ensure compiler doesn't optimize away calls to eabi division
  */
  volatile uint32_t result;
  char numBuff[3];
  char lineBuff[BUF_SIZE];
  uint32_t num,div,x,y;

  printf("Numerators:\r\n");
  for (x=0; x<(sizeof(unsignedOperandTable)/sizeof(uint32_t)); x++)
  {
    num = unsignedOperandTable[x];
    sprintf(lineBuff,"0x%08X:" , num);
    
    for (y=0; y<(sizeof(unsignedOperandTable)/sizeof(uint32_t)); y++)
    {
      div = unsignedOperandTable[y];
      
      LPC_CT32B0->TCR = 2;  /* Reset timer */
      LPC_CT32B0->TCR = 1;  /* Run timer */
      result = num / div;   /* Perform Division */
      LPC_CT32B0->TCR = 0;  /* stop timer */
      
      sprintf(numBuff, "%03d", LPC_CT32B0->TC);
      strcat(lineBuff, " ");
      strcat(lineBuff, numBuff);
    }
    strcat(lineBuff,"\r\n");
    /* Print string has fixed buffer size of 40 characters, so use UARTSend */
    printf(lineBuff);
  }
}

/* Unsigned division - perform operations, 
   and compare with precomputed results */
void uDivisionTest(void)
{
  uint32_t num, div, i;
  uint32_t result;
  char lineBuff[BUF_SIZE]="";
  
  num = unsignedOperandTable[9];
  for (i=0;i<(sizeof(unsignedOperandTable)/sizeof(uint32_t));i++)
  {
    div = unsignedOperandTable[i];
    result = num/div;
    /* Break up statements to not overflow buffer */
    sprintf(lineBuff,"%09u / %09u", num, div );
    printf(lineBuff);
    sprintf(lineBuff," = %05u %s\r\n",result, result==expectedResultsUnsigned[i] ? "PASS":"FAIL");
    printf(lineBuff);
  }
}

/* Signed division - perform operations, 
   and compare with precomputed results */
void iDivisionTest(void)
{
  int32_t num, div, i;
  int32_t result;
  char lineBuff[BUF_SIZE]="";
  
  num = signedOperandTable[7];
  for (i=0;i<(sizeof(signedOperandTable)/sizeof(int32_t));i++)
  {
    div = signedOperandTable[i];
    result = num/div;
    /* Break up statements to not overflow buffer */
    sprintf(lineBuff,"%09i / %09i", num, div );
    printf(lineBuff);
    sprintf(lineBuff," = %05i %s\r\n",result, result==expectedResultsSigned[i] ? "PASS":"FAIL");
    printf(lineBuff);
  }
}