/****************************************************************************
 *   $Id:: clkconfig.c 6782 2011-03-11 21:53:54Z nxp28548                   $
 *   Project: NXP LPC122x Clock Configuration example
 *
 *   Description:
 *     This file contains clock configuration code example which include 
 *     watchdog setup and debug clock out setup.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/
#include "LPC122x.h"
#include "clkconfig.h"

/*****************************************************************************
** Function name:		WDT_CLK_Setup
**
** Descriptions:		Configure WDT clock.
** parameters:			clock source: irc_osc(0), wdt_osc(1).			 
** 						
** Returned value:		None
** 
*****************************************************************************/
void Setup_WDOsc ( const uint32_t clksrc, const uint32_t oscdiv, const uint32_t freqsel )
{
	/* Watchdog configuration. */
	LPC_SYSCON->WDTOSCCTRL = ( (freqsel&0xF)<<5 ) | (oscdiv&0x1F);
	LPC_SYSCON->PDRUNCFG &= ~(PWR_MASK_WDO);    /* Let WDT clock run */

	/* Enables peripheral clock for WDT */
	LPC_SYSCON->SYSAHBCLKCTRL |= (1<<15);
	/* Select clock source */
	LPC_WWDT->CLKSEL = (clksrc==WDT_IRC) ? WDT_IRC : WDT_WDO;
	return;
}

/*****************************************************************************
** Function name:		CLKOUT_Setup
**
** Descriptions:		Configure CLKOUT for reference clock check.
** parameters:			clock source: irc_osc(0), sys_osc(1), wdt_osc(2),
**						main_clk(3).			 
** 						
** Returned value:		None
** 
*****************************************************************************/
void Setup_CLKOUT ( const uint32_t clksrc, const uint32_t divider )
{
	LPC_SYSCON->CLKOUTCLKSEL = clksrc;			/* Select Main clock */
	LPC_SYSCON->CLKOUTUEN = 0x01;					/* Update clock */
	LPC_SYSCON->CLKOUTUEN = 0x00;					/* Toggle update register once */
	LPC_SYSCON->CLKOUTUEN = 0x01;
	while ( !(LPC_SYSCON->CLKOUTUEN & 0x01) );	/* Wait until updated */
	LPC_SYSCON->CLKOUTDIV = (divider&0xFF);		/* Set Divider */

	LPC_SYSCON->SYSAHBCLKCTRL |= 0x10000;
	LPC_IOCON->PIO0_12 |= 0x2;
	return;
}

/*****************************************************************************
** Function name:		SetupClock
**
** Descriptions:		Will configure the main clock based on 5 configrations.
** parameters:			sel - see definition of CLK_SOURCE_MAINCLK enum.
**
** Returned value:		None
**
*****************************************************************************/
void Setup_SystemCoreClock(const CLK_SOURCE_MAINCLK sel)
{
  uint32_t pll_enable = 0;			/* Config uses PLL output */
  uint32_t pll_input_enable = 0;	/* Config uses PLL input */
  uint32_t pll_input_source = 0;	/* Select PLL input source */
  uint32_t main_input_source = 0;	/* Select main clock input source */
  uint32_t runcfg_selection = 0;	/* Minimal power configuration */
  volatile uint32_t i;
  
  switch(sel)
  {
  case MAIN_IRC:	/* IRC Only */
    pll_enable = 0;
    pll_input_enable = 0;
    pll_input_source = 0;
    main_input_source = 0x0;
    /* IRC - Turn Turn off SYSOSC,WDOSC and PLL */
    runcfg_selection = (	(PWR_MASK_PLL) |
							(PWR_MASK_WDO) |
							(PWR_MASK_SYSOSC) );
    break;
    
  case MAIN_IRC_PLL:	/* PLL w IRC */
    pll_enable = 1;
    pll_input_enable = 1;
    pll_input_source = 0;
    main_input_source = 0x3;
    /* IRC and PLL - Turn off SYSOSC, WDOSC */
    runcfg_selection = (	(PWR_MASK_WDO) |
							(PWR_MASK_SYSOSC) );
    break;		
    
  case MAIN_SYSOSC:	/* SYSOSC Only */
    pll_enable = 0;
    pll_input_enable = 1;
    pll_input_source = 1;
    main_input_source = 0x1;
    /* SYSOSC - Turn off PLL,WDOSC and IRC */
    runcfg_selection = (	(PWR_MASK_PLL) |
							(PWR_MASK_WDO) |
							(PWR_MASK_IRC) |
							(PWR_MASK_IRCOUT) );
    break;
    
  case MAIN_SYSOSC_PLL:	/* PLL w SYSOSC */
    pll_enable = 1;
    pll_input_enable = 1;
    pll_input_source = 1;
    main_input_source = 0x3;
    /* SYSOSC and PLL - Turn off IRC and WDOSC */
    runcfg_selection = (	(PWR_MASK_WDO) |
							(PWR_MASK_IRC) |
							(PWR_MASK_IRCOUT) );
    break;
    
  case MAIN_WDO:
    pll_enable = 0;
    pll_input_enable = 0;
    pll_input_source = 0;
    main_input_source = 0x2;
    /* WDOSC- Turn Turn off SYSOSC,IRC and PLL */
    runcfg_selection = (	(PWR_MASK_PLL) |
							(PWR_MASK_SYSOSC) |
							(PWR_MASK_IRC) |
							(PWR_MASK_IRCOUT) );
    break;
  }
  
  /* Momentarily enable all bits */ 
  LPC_SYSCON->PDRUNCFG &= ~(	(PWR_MASK_PLL) |
								(PWR_MASK_WDO) |
								(PWR_MASK_SYSOSC) |
								(PWR_MASK_IRC) |
								(PWR_MASK_IRCOUT));

  /* Switch to IRC initially */
  LPC_SYSCON->MAINCLKSEL = 0x00;
  LPC_SYSCON->MAINCLKUEN = 0x01;		
  LPC_SYSCON->MAINCLKUEN = 0x00;		
  LPC_SYSCON->MAINCLKUEN = 0x01;
  while ( !(LPC_SYSCON->MAINCLKUEN & 0x01) );
  
  /* 	Wait 200us for OSC to become stable, no status indication, dummy wait. */
  for (i = 0; i<200; i++) __NOP();

  if (pll_input_enable)
  {
    LPC_SYSCON->SYSPLLCLKSEL = pll_input_source;
    LPC_SYSCON->SYSPLLCLKUEN = 0x01;
    LPC_SYSCON->SYSPLLCLKUEN = 0x00;
    LPC_SYSCON->SYSPLLCLKUEN = 0x01;
    while ( !(LPC_SYSCON->SYSPLLCLKUEN & 0x01) );
  }
  
  /* Fixed PLL Settings (2x) */
  if (pll_enable)
  {
    LPC_SYSCON->SYSPLLCTRL = (1<<5) | 1;
    while ( !(LPC_SYSCON->SYSPLLSTAT & 0x01) );	/* Wait until it's locked */
  }	
  
  LPC_SYSCON->MAINCLKSEL = main_input_source;
  LPC_SYSCON->MAINCLKUEN = 0x01;		
  LPC_SYSCON->MAINCLKUEN = 0x00;		
  LPC_SYSCON->MAINCLKUEN = 0x01;
  while ( !(LPC_SYSCON->MAINCLKUEN & 0x01) );	/* Wait until updated */

  /* Shut power down to the peripherals which weren't needed */
  LPC_SYSCON->PDRUNCFG |= runcfg_selection;

  /* Other code may depend on SystemCoreClock... */
  SystemCoreClockUpdate();
  return;
}
