/****************************************************************************
 *   $Id:: timer32.c 3913 2010-07-21 23:58:20Z usb00423                     $
 *   Project: NXP LPC122x 32-bit timer example
 *
 *   Description:
 *     This file contains 32-bit timer code example which include timer 
 *     initialization, timer interrupt handler, and related APIs for 
 *     timer setup.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/
#include "LPC122x.h"
#include "timer32.h"

volatile uint32_t timer32_0_counter[4] = {0,0,0,0};
volatile uint32_t timer32_1_counter[4] = {0,0,0,0};
volatile uint32_t timer32_0_capture[4] = {0,0,0,0};
volatile uint32_t timer32_1_capture[4] = {0,0,0,0};
volatile uint32_t timer32_0_period = 0;
volatile uint32_t timer32_1_period = 0;

/*****************************************************************************
** Function name:		delay32Ms
**
** Descriptions:		Start the timer delay in milo seconds
**						until elapsed
**
** parameters:			timer number, Delay value in milo second			 
** 						
** Returned value:		None
** 
*****************************************************************************/
void delay32Ms(uint8_t timer_num, uint32_t delayInMs)
{
  if (timer_num == 0)
  {
    /* setup timer #0 for delay */
    LPC_TMR32B0->TCR = 0x02;		/* reset timer */
    LPC_TMR32B0->PR  = 0x00;		/* set prescaler to zero */
    LPC_TMR32B0->MR0 = delayInMs * (SystemAHBFrequency / 1000);
    LPC_TMR32B0->IR  = 0xff;		/* reset all interrrupts */
    LPC_TMR32B0->MCR = 0x04;		/* stop timer on match */
    LPC_TMR32B0->TCR = 0x01;		/* start timer */
  
    /* wait until delay time has elapsed */
    while (LPC_TMR32B0->TCR & 0x01);
  }
  else if (timer_num == 1)
  {
    /* setup timer #1 for delay */
    LPC_TMR32B1->TCR = 0x02;		/* reset timer */
    LPC_TMR32B1->PR  = 0x00;		/* set prescaler to zero */
    LPC_TMR32B1->MR0 = delayInMs * (SystemAHBFrequency / 1000);
    LPC_TMR32B1->IR  = 0xff;		/* reset all interrrupts */
    LPC_TMR32B1->MCR = 0x04;		/* stop timer on match */
    LPC_TMR32B1->TCR = 0x01;		/* start timer */
  
    /* wait until delay time has elapsed */
    while (LPC_TMR32B1->TCR & 0x01);
  }
  return;
}

/******************************************************************************
** Function name:		TIMER32_0_IRQHandler
**
** Descriptions:		Timer/CounterX and captureX interrupt handler
**
** parameters:			None
** Returned value:		None
** 
******************************************************************************/
void TIMER32_0_IRQHandler(void)
{
  if ( LPC_TMR32B0->IR & (0x01<<0) )
  {  
	LPC_TMR32B0->IR = 0x1<<0;			/* clear interrupt flag */
	timer32_0_counter[0]++;
  }
  if ( LPC_TMR32B0->IR & (0x01<<1) )
  {  
	LPC_TMR32B0->IR = 0x1<<1;			/* clear interrupt flag */
	timer32_0_counter[1]++;
  }
  if ( LPC_TMR32B0->IR & (0x01<<2) )
  {  
	LPC_TMR32B0->IR = 0x1<<2;			/* clear interrupt flag */
	timer32_0_counter[2]++;
  }
  if ( LPC_TMR32B0->IR & (0x01<<3) )
  {  
	LPC_TMR32B0->IR = 0x1<<3;			/* clear interrupt flag */
	timer32_0_counter[3]++;
  }
  if ( LPC_TMR32B0->IR & (0x1<<4) )
  {  
	LPC_TMR32B0->IR = 0x1<<4;			/* clear interrupt flag */
	timer32_0_capture[0]++;
  }
  if ( LPC_TMR32B0->IR & (0x1<<5) )
  {  
	LPC_TMR32B0->IR = 0x1<<5;			/* clear interrupt flag */
	timer32_0_capture[1]++;
  }
  if ( LPC_TMR32B0->IR & (0x1<<6) )
  {  
	LPC_TMR32B0->IR = 0x1<<6;			/* clear interrupt flag */
	timer32_0_capture[2]++;
  }
  if ( LPC_TMR32B0->IR & (0x1<<7) )
  {  
	LPC_TMR32B0->IR = 0x1<<7;			/* clear interrupt flag */
	timer32_0_capture[3]++;
  }
  return;
}

/******************************************************************************
** Function name:		TIMER32_1_IRQHandler
**
** Descriptions:		Timer/CounterX and captureX interrupt handler
**
** parameters:			None
** Returned value:		None
** 
******************************************************************************/
void TIMER32_1_IRQHandler(void)
{
  if ( LPC_TMR32B1->IR & (0x01<<0) )
  {  
	LPC_TMR32B1->IR = 0x1<<0;			/* clear interrupt flag */
	timer32_1_counter[0]++;
  }
  if ( LPC_TMR32B1->IR & (0x01<<1) )
  {  
	LPC_TMR32B1->IR = 0x1<<1;			/* clear interrupt flag */
	timer32_1_counter[1]++;
  }
  if ( LPC_TMR32B1->IR & (0x01<<2) )
  {  
	LPC_TMR32B1->IR = 0x1<<2;			/* clear interrupt flag */
	timer32_1_counter[2]++;
  }
  if ( LPC_TMR32B1->IR & (0x01<<3) )
  {  
	LPC_TMR32B1->IR = 0x1<<3;			/* clear interrupt flag */
	timer32_1_counter[3]++;
  }
  if ( LPC_TMR32B1->IR & (0x1<<4) )
  {  
	LPC_TMR32B1->IR = 0x1<<4;			/* clear interrupt flag */
	timer32_1_capture[0]++;
  }
  if ( LPC_TMR32B1->IR & (0x1<<5) )
  {  
	LPC_TMR32B1->IR = 0x1<<5;			/* clear interrupt flag */
	timer32_1_capture[1]++;
  }
  if ( LPC_TMR32B1->IR & (0x1<<6) )
  {  
	LPC_TMR32B1->IR = 0x1<<6;			/* clear interrupt flag */
	timer32_1_capture[2]++;
  }
  if ( LPC_TMR32B1->IR & (0x1<<7) )
  {  
	LPC_TMR32B1->IR = 0x1<<7;			/* clear interrupt flag */
	timer32_1_capture[3]++;
  }
  return;
}

/******************************************************************************
** Function name:		enable_timer
**
** Descriptions:		Enable timer
**
** parameters:			timer number: 0 or 1
** Returned value:		None
** 
******************************************************************************/
void enable_timer32(uint8_t timer_num)
{
  if ( timer_num == 0 )
  {
    LPC_TMR32B0->TCR = 1;
  }
  else
  {
    LPC_TMR32B1->TCR = 1;
  }
  return;
}

/******************************************************************************
** Function name:		disable_timer
**
** Descriptions:		Disable timer
**
** parameters:			timer number: 0 or 1
** Returned value:		None
** 
******************************************************************************/
void disable_timer32(uint8_t timer_num)
{
  if ( timer_num == 0 )
  {
    LPC_TMR32B0->TCR = 0;
  }
  else
  {
    LPC_TMR32B1->TCR = 0;
  }
  return;
}

/******************************************************************************
** Function name:		reset_timer
**
** Descriptions:		Reset timer
**
** parameters:			timer number: 0 or 1
** Returned value:		None
** 
******************************************************************************/
void reset_timer32(uint8_t timer_num)
{
  uint32_t regVal;

  if ( timer_num == 0 )
  {
    regVal = LPC_TMR32B0->TCR;
    regVal |= 0x02;
    LPC_TMR32B0->TCR = regVal;
  }
  else
  {
    regVal = LPC_TMR32B1->TCR;
    regVal |= 0x02;
    LPC_TMR32B1->TCR = regVal;
  }
  return;
}

/******************************************************************************
** Function name:		set_timer_capture
**
** Descriptions:		Set timer capture based on location
**
** parameters:			timer number: 0~1, location 0~2
** Returned value:		None
** 
******************************************************************************/
void set_timer32_capture( uint8_t timer_num, uint8_t location )
{
  /* Some of the I/O pins need to be clearfully planned if
  you use below module because JTAG and TIMER CAP/MAT pins are muxed. */
  if ( timer_num == 0 )
  {
	/*  Timer0_32 I/O config */
	if ( location == 0 )
	{
	  LPC_IOCON->PIO0_1           &= ~0x07;
	  LPC_IOCON->PIO0_1           |= 0x03;		/* Timer0_32 CAP0 */
	  LPC_IOCON->PIO0_2           &= ~0x07;
	  LPC_IOCON->PIO0_2           |= 0x03;		/* Timer0_32 CAP1 */
	  LPC_IOCON->PIO0_3           &= ~0x07;
	  LPC_IOCON->PIO0_3           |= 0x03;		/* Timer0_32 CAP2 */
	  LPC_IOCON->PIO0_4           &= ~0x07;
	  LPC_IOCON->PIO0_4           |= 0x03;		/* Timer0_32 CAP3 */
	}
	else if ( location == 1 )
	{
	  LPC_IOCON->PIO0_18          &= ~0x07;
	  LPC_IOCON->PIO0_18          |= 0x03;		/* Timer0_32 CAP0 */
	  LPC_IOCON->PIO0_19          &= ~0x07;
	  LPC_IOCON->PIO0_19          |= 0x03;		/* Timer0_32 CAP1 */
	  LPC_IOCON->PIO0_20          &= ~0x07;
	  LPC_IOCON->PIO0_20          |= 0x03;		/* Timer0_32 CAP2 */
	  LPC_IOCON->PIO0_21          &= ~0x07;
	  LPC_IOCON->PIO0_21          |= 0x03;		/* Timer0_32 CAP3 */
	}
	else if ( location == 2 )
	{
	  LPC_IOCON->PIO2_4           &= ~0x07;
	  LPC_IOCON->PIO2_4           |= 0x02;		/* Timer0_32 CAP0 */
	  LPC_IOCON->PIO2_5           &= ~0x07;
	  LPC_IOCON->PIO2_5           |= 0x02;		/* Timer0_32 CAP1 */
	  LPC_IOCON->PIO2_6           &= ~0x07;
	  LPC_IOCON->PIO2_6           |= 0x02;		/* Timer0_32 CAP2 */
	  LPC_IOCON->PIO2_7           &= ~0x07;
	  LPC_IOCON->PIO2_7           |= 0x02;		/* Timer0_32 CAP3 */
	}
	else
	{
	  while ( 1 );				/* Fatal location number error */
	}
  }
  else
  {
	/*  Timer1_32 I/O config */
	if ( location == 0 )
	{
	  LPC_IOCON->PIO0_6           &= ~0x07;
	  LPC_IOCON->PIO0_6           |= 0x03;		/* Timer1_32 CAP0 */
	  LPC_IOCON->PIO0_7           &= ~0x07;
	  LPC_IOCON->PIO0_7           |= 0x03;		/* Timer1_32 CAP1 */
	  LPC_IOCON->PIO0_8           &= ~0x07;
	  LPC_IOCON->PIO0_8           |= 0x03;		/* Timer1_32 CAP2 */
	  LPC_IOCON->PIO0_9           &= ~0x07;
	  LPC_IOCON->PIO0_9           |= 0x03;		/* Timer1_32 CAP3 */
	}
	else if ( location == 1 )
	{
	  LPC_IOCON->PIO0_23          &= ~0x07;
	  LPC_IOCON->PIO0_23          |= 0x03;		/* Timer1_32 CAP0 */
	  LPC_IOCON->PIO0_24          &= ~0x07;
	  LPC_IOCON->PIO0_24          |= 0x03;		/* Timer1_32 CAP1 */
#ifdef __SWD_DISABLED
	  LPC_IOCON->SWDIO_PIO0_25    &= ~0x07;
	  LPC_IOCON->SWDIO_PIO0_25    |= 0x03;		/* Timer1_32 CAP2 */
	  LPC_IOCON->SWCLK_PIO0_26    &= ~0x07;
	  LPC_IOCON->SWCLK_PIO0_26    |= 0x03;		/* Timer1_32 CAP3 */
#endif
	}
	else if ( location == 2 )
	{
	  LPC_IOCON->PIO2_8           &= ~0x07;
	  LPC_IOCON->PIO2_8           |= 0x02;		/* Timer1_32 CAP0 */
	  LPC_IOCON->PIO2_9           &= ~0x07;
	  LPC_IOCON->PIO2_9           |= 0x02;		/* Timer1_32 CAP1 */
	  LPC_IOCON->PIO2_10          &= ~0x07;
	  LPC_IOCON->PIO2_10          |= 0x02;		/* Timer1_32 CAP2 */
	  LPC_IOCON->PIO2_11          &= ~0x07;
	  LPC_IOCON->PIO2_11          |= 0x02;		/* Timer1_32 CAP3 */
	}
	else
	{
	  while ( 1 );				/* Fatal location number error */
	}
  }	
  return;
}

/******************************************************************************
** Function name:		set_timer_match
**
** Descriptions:		Set timer match based on location
**
** parameters:			timer number: 0~1, location 0~2
** Returned value:		None
** 
******************************************************************************/
void set_timer32_match( uint8_t timer_num, uint8_t match_enable, uint8_t location )
{
  /* Some of the I/O pins need to be clearfully planned if
  you use below module because JTAG and TIMER CAP/MAT pins are muxed. */
  if ( timer_num == 0 )
  {
	/*  Timer0_32 I/O config */
	if ( match_enable & (0x1<<0) )
	{
	  if ( location == 0 )
	  {
		LPC_IOCON->PIO0_1         &= ~0x07;
		LPC_IOCON->PIO0_1         |= 0x04;		/* Timer0_32 MAT0 */
	  }
	  else if ( location == 1 )
	  {
		LPC_IOCON->PIO0_18        &= ~0x07;
		LPC_IOCON->PIO0_18        |= 0x04;		/* Timer0_32 MAT0 */
	  }
	  else if ( location == 2 )
	  {
		LPC_IOCON->PIO2_4         &= ~0x07;
		LPC_IOCON->PIO2_4         |= 0x03;		/* Timer0_32 MAT0 */
	  }
	}
	if ( match_enable & (0x1<<1) )
	{
	  if ( location == 0 )
	  {
		LPC_IOCON->PIO0_2         &= ~0x07;
		LPC_IOCON->PIO0_2         |= 0x04;		/* Timer0_32 MAT1 */
	  }
	  else if ( location == 1 )
	  {
		LPC_IOCON->PIO0_19        &= ~0x07;
		LPC_IOCON->PIO0_19        |= 0x04;		/* Timer0_32 MAT1 */	 
	  }
	  else if ( location == 2 )
	  {
		LPC_IOCON->PIO2_5         &= ~0x07;	
		LPC_IOCON->PIO2_5         |= 0x03;		/* Timer0_32 MAT1 */
	  }
	}
	if ( match_enable & (0x1<<2) )
	{
	  if ( location == 0 )
	  {
		LPC_IOCON->PIO0_3         &= ~0x07;
		LPC_IOCON->PIO0_3         |= 0x04;		/* Timer0_32 MAT2 */
	  }
	  else if ( location == 1 )
	  {
		LPC_IOCON->PIO0_20        &= ~0x07;
		LPC_IOCON->PIO0_20        |= 0x04;		/* Timer0_32 MAT2 */
	  }
	  else if ( location == 2 )
	  {
		LPC_IOCON->PIO2_6         &= ~0x07;	
		LPC_IOCON->PIO2_6         |= 0x03;		/* Timer0_32 MAT2 */
	  }
	}
	if ( match_enable & (0x1<<3) )
	{
	  if ( location == 0 )
	  {
		LPC_IOCON->PIO0_4         &= ~0x07;
		LPC_IOCON->PIO0_4         |= 0x04;		/* Timer0_32 MAT3 */
	  }
	  else if ( location == 1 )
	  {
		LPC_IOCON->PIO0_21        &= ~0x07;
		LPC_IOCON->PIO0_21        |= 0x04;		/* Timer0_32 MAT3 */
	  }
	  else if ( location == 2 )
	  {
		LPC_IOCON->PIO2_7         &= ~0x07;	
		LPC_IOCON->PIO2_7         |= 0x03;		/* Timer0_32 MAT3 */
	  }
	}
  }
  else
  {
	/*  Timer1_32 I/O config */
	if ( match_enable & (0x1<<0) )
	{
	  if ( location == 0 )
	  {
		LPC_IOCON->PIO0_6         &= ~0x07;
		LPC_IOCON->PIO0_6         |= 0x04;		/* Timer1_32 MAT0 */
	  }
	  else if ( location == 1 )
	  {
		LPC_IOCON->PIO0_23        &= ~0x07;
		LPC_IOCON->PIO0_23        |= 0x04;		/* Timer1_32 MAT0 */
	  }
	  else if ( location == 2 )
	  {
		LPC_IOCON->PIO2_8         &= ~0x07;
		LPC_IOCON->PIO2_8         |= 0x03;		/* Timer1_32 MAT0 */
	  }
	}
	/*  Timer1_32 I/O config */
	if ( match_enable & (0x1<<1) )
	{
	  if ( location == 0 )
	  {
		LPC_IOCON->PIO0_7         &= ~0x07;
		LPC_IOCON->PIO0_7         |= 0x04;		/* Timer1_32 MAT1 */
	  }
	  else if ( location == 1 )
	  {
		LPC_IOCON->PIO0_24        &= ~0x07;
		LPC_IOCON->PIO0_24        |= 0x04;		/* Timer1_32 MAT1 */
	  }
	  else if ( location == 2 )
	  {
		LPC_IOCON->PIO2_9         &= ~0x07;	
		LPC_IOCON->PIO2_9         |= 0x03;		/* Timer1_32 MAT1 */
	  }
	}
		/*  Timer1_32 I/O config */
	if ( match_enable & (0x1<<2) )
	{
	  if ( location == 0 )
	  {
		LPC_IOCON->PIO0_8         &= ~0x07;
		LPC_IOCON->PIO0_8         |= 0x04;		/* Timer1_32 MAT2 */
	  }
	  else if ( location == 1 )
	  {
#ifdef __SWD_DISABLED
		LPC_IOCON->SWDIO_PIO0_25  &= ~0x07;
		LPC_IOCON->SWDIO_PIO0_25  |= 0x04;		/* Timer1_32 MAT2 */
#endif
	  }
	  else if ( location == 2 )
	  {
		LPC_IOCON->PIO2_10        &= ~0x07;
		LPC_IOCON->PIO2_10        |= 0x03;		/* Timer1_32 MAT2 */
	  }
	}
	/*  Timer1_32 I/O config */
	if ( match_enable & (0x1<<3) )
	{
	  if ( location == 0 )
	  {
		LPC_IOCON->PIO0_9         &= ~0x07;
		LPC_IOCON->PIO0_9         |= 0x04;		/* Timer1_32 MAT3 */
	  }
	  else if ( location == 1 )
	  {
#ifdef __SWD_DISABLED
		LPC_IOCON->SWCLK_PIO0_26  &= ~0x07;
		LPC_IOCON->SWCLK_PIO0_26  |= 0x04;		/* Timer1_32 MAT3 */
#endif
	  }
	  else if ( location == 2 )
	  {
		LPC_IOCON->PIO2_11        &= ~0x07;	
		LPC_IOCON->PIO2_11        |= 0x03;		/* Timer1_32 MAT3 */
	  }
	}
  }	
  return;
}

/******************************************************************************
** Function name:		Set_timer_reset
**
** Descriptions:		set timer external reset pins based on LOC number.
**
** parameters:			timer number and location number 
** Returned value:		None
** 
******************************************************************************/
void set_timer32_reset(uint8_t timer_num, uint8_t location)
{
  if ( timer_num == 0 )
  {
	/*  Timer0_32 I/O config */
	if ( location == 0 )
	{
	  LPC_IOCON->PIO0_23     &= ~0x07;
	  LPC_IOCON->PIO0_23     |= 0x05;		/* Timer0_32 RST_LOC0 */
	}
	else if ( location == 1 )
	{
	  LPC_IOCON->PIO2_12     &= ~0x07;
	  LPC_IOCON->PIO2_12     |= 0x02;		/* Timer0_32 RST_LOC1 */
	}
  }
  else if ( timer_num == 1 )
  {
	/*  Timer1_32 I/O config */
	if ( location == 0 )
	{
	  LPC_IOCON->PIO0_24     &= ~0x07;
	  LPC_IOCON->PIO0_24     |= 0x05;		/* Timer1_32 RST_LOC0 */
	}
	else if ( location == 1 )
	{
	  LPC_IOCON->PIO2_13     &= ~0x07;
	  LPC_IOCON->PIO2_13     |= 0x02;		/* Timer1_32 RST_LOC1 */
	}
  }
  else
  {
	while ( 1 );				/* Fatal timer number error */
  }
  return;		
}


/******************************************************************************
** Function name:		init_timer
**
** Descriptions:		Initialize timer, set timer interval, reset timer,
**						install timer interrupt handler
**
** parameters:			timer number and timer interval
** Returned value:		None
** 
******************************************************************************/
void init_timer32(uint8_t timer_num, uint8_t location, uint32_t TimerInterval) 
{
  uint32_t i;

  if ( timer_num == 0 )
  {
	/* Some of the I/O pins need to be clearfully planned if
	you use below module because JTAG and TIMER CAP/MAT pins are muxed. */
	LPC_SYSCON->PRESETCTRL |= (0x1<<6);
	LPC_SYSCON->SYSAHBCLKCTRL |= (0x1<<9);
	/*  Timer0_32 I/O config, default is capture. */
#if TIMER_32BIT_MATCH
	/* set all matches, enable match_enable field to 0x0F */
	set_timer32_match(timer_num, 0x0F, location );
#else
	set_timer32_capture(timer_num, location );
#endif

#if TIMER_32BIT_RESET_ENA
#if 1
    set_timer32_reset(timer_num, 0);
#else
    set_timer32_reset(timer_num, 1);
#endif
#endif
	for ( i = 0; i < 4; i++ )
	{
	  timer32_0_counter[i] = 0;
	  timer32_0_capture[i] = 0;
	}
	LPC_TMR32B0->MR0 = TimerInterval;
	LPC_TMR32B0->MR1 = TimerInterval;
	LPC_TMR32B0->MR2 = TimerInterval;
	LPC_TMR32B0->MR3 = TimerInterval;
#if TIMER_32BIT_MATCH
	LPC_TMR32B0->EMR &= ~(0xFF<<4);
	LPC_TMR32B0->EMR |= ((0x3<<4)|(0x3<<6)|(0x3<<8)|(0x3<<10));	/* MR0/1/2/3 Toggle */
#else
	/* Capture on rising edge 0~3. */
	LPC_TMR32B0->CCR = 0xB6D;		/* bit 0, 3, 6, 9 */
//	LPC_TMR32B0->CTCR = (0x1<<0)|(0x0<<2);
#endif
	/* Interrupt and Reset on MR0/1/2/3 */
	LPC_TMR32B0->MCR = (0x3<<0)|(0x3<<3)|(0x3<<6)|(0x3<<9);  

	/* Enable the TIMER0 Interrupt */
	NVIC_EnableIRQ(TIMER_32_0_IRQn);
  }
  else if ( timer_num == 1 )
  {
	/* Some of the I/O pins need to be clearfully planned if
	you use below module because JTAG and TIMER CAP/MAT pins are muxed. */
	LPC_SYSCON->PRESETCTRL |= (0x1<<7);
	LPC_SYSCON->SYSAHBCLKCTRL |= (1<<10);
	/*  Timer1_32 I/O config, default is capture. */
#if TIMER_32BIT_MATCH
	/* set all matches, enable match_enable field to 0x0F */
	set_timer32_match(timer_num, 0x0F, location );
#else
	set_timer32_capture(timer_num, location );
#endif

#if TIMER_32BIT_RESET_ENA
#if 1
    set_timer32_reset(timer_num, 0);
#else
    set_timer32_reset(timer_num, 1);
#endif
#endif
	for ( i = 0; i < 4; i++ )
	{
	  timer32_1_counter[i] = 0;
      timer32_1_capture[i] = 0;
	}
	LPC_TMR32B1->MR0 = TimerInterval;
	LPC_TMR32B1->MR1 = TimerInterval;
	LPC_TMR32B1->MR2 = TimerInterval;
	LPC_TMR32B1->MR3 = TimerInterval;
#if TIMER_32BIT_MATCH
	LPC_TMR32B1->EMR &= ~(0xFF<<4);
	LPC_TMR32B1->EMR |= ((0x3<<4)|(0x3<<6)|(0x3<<8)|(0x3<<10));	/* MR0/1/2 Toggle */
#else
	/* Capture on rising edge 0~3. */
	LPC_TMR32B1->CCR = 0xB6D;		/* bit 0, 3, 6, 9 */
//	LPC_TMR32B1->CTCR = (0x1<<0)|(0x0<<2);
#endif
	/* Interrupt and Reset on MR0/1/2/3 */
	LPC_TMR32B1->MCR = (0x3<<0)|(0x3<<3)|(0x3<<6)|(0x3<<9); 

	/* Enable the TIMER1 Interrupt */
	NVIC_EnableIRQ(TIMER_32_1_IRQn);
  }
  return;
}

/******************************************************************************
** Function name:		init_timer32PWM
**
** Descriptions:		Initialize timer as PWM
**
** parameters:			timer number, period and match enable:
**						match_enable[0] = PWM for MAT0 
**						match_enable[1] = PWM for MAT1
**						match_enable[2] = PWM for MAT2
** Returned value:		None
** 
******************************************************************************/
void init_timer32PWM(uint8_t timer_num, uint32_t period, uint8_t match_enable)
{
  disable_timer32(timer_num);
  if (timer_num == 1)
  {
    /* Some of the I/O pins need to be clearfully planned if
    you use below module because JTAG and TIMER CAP/MAT pins are muxed. */
    LPC_SYSCON->SYSAHBCLKCTRL |= (1<<10);

    /* Setup the external match register */
    LPC_TMR32B1->EMR = (1<<EMC3)|(1<<EMC2)|(2<<EMC1)|(1<<EMC0)|(1<<3)|(match_enable);

    /* Setup the outputs */
    /* If match0 is enabled, set the output, use location 0 for test. */
	set_timer32_match( timer_num, match_enable, 0 );

    /* Enable the selected PWMs and enable Match3 */
    LPC_TMR32B1->PWMC = (1<<3)|(match_enable);
 
    /* Setup the match registers */
    /* set the period value to a global variable */
    timer32_1_period = period;
    LPC_TMR32B1->MR3 = timer32_1_period;
    LPC_TMR32B1->MR0 = timer32_1_period/2;
    LPC_TMR32B1->MR1 = timer32_1_period/2;
    LPC_TMR32B1->MR2 = timer32_1_period/2;
    LPC_TMR32B1->MCR = 1<<10;				/* Reset on MR3 */
  }
  else
  {
    /* Some of the I/O pins need to be clearfully planned if
    you use below module because JTAG and TIMER CAP/MAT pins are muxed. */
    LPC_SYSCON->SYSAHBCLKCTRL |= (1<<9);

    /* Setup the external match register */
    LPC_TMR32B0->EMR = (1<<EMC3)|(2<<EMC2)|(1<<EMC1)|(1<<EMC0)|(1<<3)|(match_enable);
 
    /* Setup the outputs */
    /* If match0 is enabled, set the output, use location 0 for test. */
	set_timer32_match( timer_num, match_enable, 0 );

    /* Enable the selected PWMs and enable Match3 */
    LPC_TMR32B0->PWMC = (1<<3)|(match_enable);

    /* Setup the match registers */
    /* set the period value to a global variable */
    timer32_0_period = period;
    LPC_TMR32B0->MR3 = timer32_0_period;
    LPC_TMR32B0->MR0 = timer32_0_period/2;
    LPC_TMR32B0->MR1 = timer32_0_period/2;
    LPC_TMR32B0->MR2 = timer32_0_period/2;

    LPC_TMR32B0->MCR = 1<<10;				/* Reset on MR3 */
  }
}

/******************************************************************************
** Function name:		pwm32_setMatch
**
** Descriptions:		Set the pwm32 match values
**
** parameters:			timer number, match numner and the value
**
** Returned value:		None
** 
******************************************************************************/
void setMatch_timer32PWM (uint8_t timer_num, uint8_t match_nr, uint32_t value)
{
  if (timer_num)
  {
    switch (match_nr)
    {
      case 0:
        LPC_TMR32B1->MR0 = value;
      break;
      case 1: 
        LPC_TMR32B1->MR1 = value;
      break;
      case 2:
        LPC_TMR32B1->MR2 = value;
      break;
      case 3: 
        LPC_TMR32B1->MR3 = value;
      break;
      default:
      break;
    }	
  }
  else 
  {
    switch (match_nr)
    {
      case 0:
        LPC_TMR32B0->MR0 = value;
      break;
      case 1: 
        LPC_TMR32B0->MR1 = value;
      break;
      case 2:
        LPC_TMR32B0->MR2 = value;
      break;
      case 3: 
        LPC_TMR32B0->MR3 = value;
      break;
      default:
      break;
    }	
  }
}

/******************************************************************************
**                            End Of File
******************************************************************************/
