/**********************************************************************************************************
 *      This LPC1100 sample project allows the user to enter and exit the low power saving modes:
 *      1) Sleep Mode
 *      2) Deep-Sleep Mode
 *      3) Deep Power-down Mode
 *
 *      It also allows the user to measure the wake up times for each low power modes.
 *      This project displays a menu via the UART interface.
 *      See the "Using LPC111x/x01 and LPC11C1x/x01 power modes on the LPCXpresso" Application Note for further details.
 *
 *      For Sleep and Deep Sleep modes, the wake-up times can be measured by using the following pins:
 *		Trigger pin (PIO0_2)
 *		Used to get the device out of the sleep mode or deep-sleep mode.
 *		The pin is triggered externally (falling edge) to wake the device up using an external push button.
 *		
 *		Strobe pin (PIO2_0)
 *		After wake-up, the device returns into run mode, and this pin is set high within
 *		the WAKEUP_IRQHANDLER() subroutine.
 *
 *	    The wakeup time is the difference between the falling edge of the trigger pin
 *	    and the rising edge of the Strobe pin.
 * 
**********************************************************************************************************/
#ifdef __USE_CMSIS
#include "LPC11xx.h"
#endif

#include "uart.h"

#define UART_BAUD	9600

#define ENABLE_CLKOUT	0

extern volatile uint32_t UARTCount;
extern volatile uint8_t UARTBuffer[BUFSIZE];

/* Function returns 1st character from the received UARTBuffer */
uint8_t getchar(void) {
	uint8_t temp;

	while ( UARTCount == 0 );
	
	LPC_UART->IER = IER_THRE | IER_RLS;			/* Disable RBR */
	temp = UARTBuffer[0];
	UARTCount = 0;
	LPC_UART->IER = IER_THRE | IER_RLS | IER_RBR;	/* Re-enable RBR */
	return temp;
}

/* Function prints a menu of options onto the UART */
void menu(void) {

	uint8_t menu_array[] =
	/*            1         2         3         4         5         6*/
	/*	012345678901234567890123456789012345678901234567890123456789 */
	 "\r\nENTER WHICH MODE YOU WANT TO ENTER - LPC1100\r\n"
	   "SLEEP mode\r\n"
	   "    'a' - IRC (12MHz)\r\n"
	   "    'b' - System Oscillator (12MHz)\r\n"
	   "    'c' - System Oscillator with PLL (50MHz)\r\n"
	   "\r\n"
	   "DEEP SLEEP mode\r\n"
	   "    'd' - All OFF\r\n"
	   "\r\n"
	   "DEEP POWER DOWN mode\r\n"
	   "    'e' - Enter DEEP POWER DOWN\r\n"
	   "\r\n"
	   "Option: ";
	   UARTSend(menu_array, sizeof(menu_array));
}

/* Function turns off the UART peripheral for the current test */
void DeInit_UART(void){
	/* Disable NVIC UART interrupts */
	NVIC_DisableIRQ(UART_IRQn);

	/* Set the UART clock divider to 0, disabling the clock */
	LPC_SYSCON->UARTCLKDIV = 0;
		   
	/* Power down the UART */
	LPC_SYSCON->SYSAHBCLKCTRL &= ~(1<<12);
	return;
}

void delay(uint32_t count){
	while(count--);
	return;
}	
	 
void WAKEUP_IRQHandler(void){
	/* Signal scope that chip is in the interrupt handler */
	LPC_GPIO2->DATA |= (1<<0);

	/* Clear the pending interrupt of the start logic */
	LPC_SYSCON->STARTRSRP0CLR	|=	(1<<(2));

	return;
}

void config_ios(void);

int main (void) {

	uint8_t cmd = 0x00;

	/* Turn off all other peripheral dividers */
	LPC_SYSCON->SSP0CLKDIV = 0;
	LPC_SYSCON->SSP1CLKDIV = 0;
	LPC_SYSCON->WDTCLKDIV = 0;
	LPC_SYSCON->SYSTICKCLKDIV = 0;


	/* SYSAHBCLKCTRL while running in FLASH */
	LPC_SYSCON->SYSAHBCLKCTRL	=
								 (1<<0) 	//SYS Clock
								|(1<<1)		//ROM
								|(1<<2) 	//RAM
								|(1<<3) 	//FLASHREG
								|(1<<4) 	//FLASHARRAY
								|(1<<6) 	//GPIO
								|(1<<12)	//UART
								|(1<<16)	//IOCON
								;  

	#if ENABLE_CLKOUT
	/* Output the Clk onto the CLKOUT Pin PIO0_1 to monitor the freq on a scope */
	LPC_IOCON->PIO0_1	= (1<<0);
	/* Select the MAIN clock as the clock out selection since it's driving the core */
	LPC_SYSCON->CLKOUTCLKSEL = 3;
	/* Set CLKOUTDIV to 6 */
	LPC_SYSCON->CLKOUTDIV = 1;		//	CLKOUT Divider = 1
	/* Enable CLKOUT */
	LPC_SYSCON->CLKOUTUEN = 0;
	LPC_SYSCON->CLKOUTUEN = 1;
	while (!(LPC_SYSCON->CLKOUTUEN & 0x01));
	#endif
		  						 
	/* Specify the start logic to allow the chip to be waken up */
	LPC_SYSCON->STARTAPRP0 		&= ~(1<<(2));		// Falling edge
	LPC_SYSCON->STARTRSRP0CLR	|=	(1<<(2));		// Clear pending bit
	LPC_SYSCON->STARTERP0		|=	(1<<(2));		// Enable Start Logic

	/* Specify a pin as an output to be toggled once the chip has been brought out of sleep mode */
	LPC_GPIO2->DIR	|=	(1<<0);
	LPC_GPIO2->DATA &= ~(1<<0);
	
	SystemCoreClock = 12000000UL;
	/* NVIC is installed inside UARTInit file. */
	UARTInit(UART_BAUD);

	if(LPC_PMU->PCON & (1<<11)){
		UARTSend("\r\n!!!Reboot from Deep Power Down!!!\r\n",\
		  sizeof("\r\n!!!Reboot from Deep Power Down!!!\r\n"));

		/* Clear the Deep Power down flag from the PMU */
		LPC_PMU->PCON |= (1<<11);
	}

	menu();
    
  while (1) {				/* Loop forever */
	
	if(UARTCount != 0){
		
		cmd = getchar();
		UARTSend(&cmd, 1);
		delay(0xF000);
		
		switch(cmd){
			/* Sleep Mode Options */
			case 'a':
				/* Disable UART */
				DeInit_UART();

				/* Turn on all the IRC & Flash */
				LPC_SYSCON->PDRUNCFG &= ~((1<<0) | (1<<1) | (1<<2));			
				
				/* Switch MAINCLKSEL to IRC */
				LPC_SYSCON->MAINCLKSEL	= 0;
				LPC_SYSCON->MAINCLKUEN  = 0;
				LPC_SYSCON->MAINCLKUEN  = 1;
				while (!(LPC_SYSCON->MAINCLKUEN & 0x01));
	
				/* Make sure only the IRC is running */
				LPC_SYSCON->PDRUNCFG = ~((1<<0) | (1<<1) | (1<<2) | (1<<9)) ;
				
				/* Specify Sleep mode before entering mode */
				SCB->SCR	&=	~(1<<2);	//Clear SLEEPDEEP bit

				LPC_SYSCON->STARTRSRP0CLR |= (1<<(2));
				NVIC_ClearPendingIRQ(WAKEUP2_IRQn);
				NVIC_EnableIRQ(WAKEUP2_IRQn);

				/* Reconfigure the IOs */
				config_ios();

				/* Enter Sleep mode */
				__WFI();			  
				break;
	
			case 'b':
				/* Disable UART */
				DeInit_UART();

				/* Turn on all the system oscillator & Flash  */
				LPC_SYSCON->PDRUNCFG &= ~((1<<5) | (1<<2));
	
				/* Switch MAINCLKSEL to System oscillator */
				LPC_SYSCON->MAINCLKSEL	= 1;
				LPC_SYSCON->MAINCLKUEN  = 0;
				LPC_SYSCON->MAINCLKUEN  = 1;
				while (!(LPC_SYSCON->MAINCLKUEN & 0x01));
	
				/* Select SYSPLLCLKSEL to System oscillator */
				LPC_SYSCON->SYSPLLCLKSEL = 1;
				LPC_SYSCON->SYSPLLCLKUEN = 0;
				LPC_SYSCON->SYSPLLCLKUEN = 1;
				while (!(LPC_SYSCON->SYSPLLCLKUEN & 0x01));
		
				/* Make sure only the system oscillator & Flash are running now */
				LPC_SYSCON->PDRUNCFG = ~((1<<5) | (1<<2) | (1<<9) | (1<<12));
								
				/* Specify Sleep mode before entering mode */
				SCB->SCR	&=	~(1<<2);	//Clear SLEEPDEEP bit

				LPC_SYSCON->STARTRSRP0CLR |= (1<<(2));
				NVIC_ClearPendingIRQ(WAKEUP2_IRQn);
				NVIC_EnableIRQ(WAKEUP2_IRQn);
																   	
				/* Reconfigure the IOs */
				config_ios();

				/* Enter Sleep mode */
				__WFI();
				break;
	
			case 'c':
				/* Disable UART */
				DeInit_UART();

				/* Turn on all the system oscillator & Flash  */
				LPC_SYSCON->PDRUNCFG &= ~((1<<5) | (1<<2));
							
				LPC_SYSCON->SYSPLLCLKSEL  = 1;	//System oscillator
				LPC_SYSCON->SYSPLLCLKUEN  = 1;
				LPC_SYSCON->SYSPLLCLKUEN  = 0;
				LPC_SYSCON->SYSPLLCLKUEN  = 1;
				while (!(LPC_SYSCON->SYSPLLCLKUEN & 0x01));
				LPC_SYSCON->PDRUNCFG 	|=  (1<<7); 			/* Power down the PLL */
				LPC_SYSCON->SYSPLLCTRL    = (1<<5) | 3;			// 48Mhz
//				LPC_SYSCON->SYSPLLCTRL    = (2<<5) | 2;			// 36Mhz
//				LPC_SYSCON->SYSPLLCTRL    = (2<<5) | 1;			// 24Mhz
//				LPC_SYSCON->SYSPLLCTRL    = (3<<5) | 0;			// 12Mhz
				LPC_SYSCON->PDRUNCFG    &= ~(1 << 7);          	/* Power-up SYSPLL          */
				while (!(LPC_SYSCON->SYSPLLSTAT & 0x01));	    /* Wait Until PLL Locked    */
				
				/* Switch MAINCLKSEL to Sys PLL clock out */
				LPC_SYSCON->MAINCLKSEL	= 3;
				LPC_SYSCON->MAINCLKUEN  = 0;
				LPC_SYSCON->MAINCLKUEN  = 1;
				while (!(LPC_SYSCON->MAINCLKUEN & 0x01));
				
				/* Make sure only the system oscillator, PLL, & Flash are running now */
				LPC_SYSCON->PDRUNCFG = ~((1<<5) | (1<<7) | (1<<2) | (1<<9) | (1<<12));

				/* Specify Sleep mode before entering mode */
				SCB->SCR	&=	~(1<<2);	//Clear SLEEPDEEP bit

				LPC_SYSCON->STARTRSRP0CLR |= (1<<(2));
				NVIC_ClearPendingIRQ(WAKEUP2_IRQn);
				NVIC_EnableIRQ(WAKEUP2_IRQn);

				/* Reconfigure the IOs */
				config_ios();

				/* Enter Sleep mode */
				__WFI();
				break;


	
			/* Deep Sleep Mode Options */
			case 'd':
				/* Disable UART */
				DeInit_UART();

				/* Turn on all the IRC & Flash  */
				LPC_SYSCON->PDRUNCFG &= ~((1<<0) | (1<<1) | (1<<2));			
				
				/* Switch MAINCLKSEL to IRC */
				LPC_SYSCON->MAINCLKSEL	= 0;
				LPC_SYSCON->MAINCLKUEN  = 0;
				LPC_SYSCON->MAINCLKUEN  = 1;
				while (!(LPC_SYSCON->MAINCLKUEN & 0x01));
	
				/* Make sure only the IRC is running */
				LPC_SYSCON->PDRUNCFG = ~((1<<0) | (1<<1) | (1<<2) | (1<<9)) ;

				/* Clear the Deep Sleep Flag */
				LPC_PMU->PCON |= (1<<8);

				/* All OFF */
				LPC_SYSCON->PDSLEEPCFG |= 0x000018FF;

				/* Specify peripherals to be powered up again when returning from deep sleep mode */
				LPC_SYSCON->PDAWAKECFG = LPC_SYSCON->PDRUNCFG;
	
				/* Ensure DPDEN is disabled in the power control register */
				LPC_PMU->PCON	= (1<<11); //Clear DPDFLAG if it was set
	
				/* Specify Deep Sleep mode before entering mode */
				SCB->SCR	|=	(1<<2);		//Set SLEEPDEEP bit
				
				LPC_SYSCON->STARTRSRP0CLR |= (1<<(2));
				NVIC_ClearPendingIRQ(WAKEUP2_IRQn);
				NVIC_EnableIRQ(WAKEUP2_IRQn);

				/* Reconfigure the IOs */
				config_ios();

				/* Enter Deep Sleep mode */
				__WFI();
				break;
	


			/* Deep Power-down Mode Options */
			case 'e':
				/* Disable UART */
				DeInit_UART();

				/* Specify DPDEN to power control register */
				LPC_PMU->PCON	= (1<<1)|(1<<11);
	
				/* Specify Deep Power-down mode before entering mode */
				SCB->SCR	|=	(1<<2);		//Set SLEEPDEEP bit
	
				/* Enable IRC before deep power-down mode */
				LPC_SYSCON->PDRUNCFG &= ~((1<<0) | (1<<1));
				
				/* Reconfigure the IOs */
				config_ios();

				/* Enter Deep Power-down mode */
				__WFI();
				break;
	
			/* Default case whenever an incorrect cmd has been entered */
			default:
				UARTSend( "\r\n Incorrect Key\r\n", \
				   sizeof("\r\n Incorrect Key\r\n"));
	
		}
					
			NVIC_SystemReset();
	}
												  
  }
}

void config_ios(void){

	/* Configure all IOs as GPIO w/o pull-up & pull-down resistors */
	LPC_IOCON->PIO2_6			= 0xC0;
	LPC_IOCON->PIO2_0			= 0xC0;
	LPC_IOCON->RESET_PIO0_0 	= 0xC1;      // RST_BOOT/P0_0
#if !ENABLE_CLKOUT
	LPC_IOCON->PIO0_1			= 0xC0;      // CLKOUT/CT32B0_MAT2/USB_SOF
#endif /* ENABLE_CLKOUT */
	LPC_IOCON->PIO1_8			= 0xC0;
	LPC_IOCON->PIO0_2			= 0xC0;      // SSEL/CT16B0_CAP0	//Trigger input

	LPC_IOCON->PIO2_7			= 0xC0;
	LPC_IOCON->PIO2_8			= 0xC0;
	LPC_IOCON->PIO2_1			= 0xC0;
	LPC_IOCON->PIO0_3			= 0xC0;      // USB VBUS
	LPC_IOCON->PIO0_4			= 0xC0;      // I2C_SCL, no pull-up, inactive
	LPC_IOCON->PIO0_5			= 0xC0;      // I2C_SDA, no pull-up, inactive
	LPC_IOCON->PIO1_9			= 0xC0;      // CT16B1_MAT0
	LPC_IOCON->PIO3_4			= 0xC0;

	LPC_IOCON->PIO2_4			= 0xC0;
	LPC_IOCON->PIO2_5			= 0xC0;
	LPC_IOCON->PIO3_5			= 0xC0;
	LPC_IOCON->PIO0_6			= 0xC0;      // USB SoftConnect
	LPC_IOCON->PIO0_7			= 0xC0;
	LPC_IOCON->PIO2_9			= 0xC0;
	LPC_IOCON->PIO2_10			= 0xC0;
	LPC_IOCON->PIO2_2			= 0xC0;

	LPC_IOCON->PIO0_8			= 0xC0;		// SSP_MISO/CT16B0_MAT0/TRACE_CLK
	LPC_IOCON->PIO0_9			= 0xC0;		// SSP_MOSI/CT16B0_MAT1/TRACE_SWV
	LPC_IOCON->JTAG_TCK_PIO0_10	= 0xC1;		// JTAG_CLK/P0_10/SSP_CLK/CT16B0_MAT2
	LPC_IOCON->PIO1_10			= 0xC0;		// ADCIN6/CT16B1_MAT1
	LPC_IOCON->PIO2_11			= 0xC0;
	LPC_IOCON->JTAG_TDI_PIO0_11	= 0xC1;		// JTAG_TDI/P0_11/ADCIN0/CT32B0_MAT3
	LPC_IOCON->JTAG_TMS_PIO1_0  = 0xC1;		// JTAG_TMS/P1_0/ADCIN1/CT32B1_CAP0
	LPC_IOCON->JTAG_TDO_PIO1_1  = 0xC1;		// JTAG_TDO/P1_1/ADCIN2/CT32B1_MAT0

	LPC_IOCON->JTAG_nTRST_PIO1_2 = 0xC1;	// JTAG_TRST/P1_2/ADCIN3/CT32B1_MAT1
	LPC_IOCON->PIO3_0			= 0xC0;
	LPC_IOCON->PIO3_1			= 0xC0;
	LPC_IOCON->PIO2_3			= 0xC0;
	LPC_IOCON->ARM_SWDIO_PIO1_3	= 0xC1;		// ARM_SWD/P1_3/ADCIN4/CT32B1_MAT2
	LPC_IOCON->PIO1_4			= 0xC0;		// ADCIN5/CT32B1_MAT3/WAKEUP
	LPC_IOCON->PIO1_11			= 0xC0;		// ADCIN7
	LPC_IOCON->PIO3_2			= 0xC0;

	LPC_IOCON->PIO1_5			= 0xC0;      // UART_DIR/CT32B0_CAP0
	LPC_IOCON->PIO1_6			= 0xC0;      // UART_RXD/CT32B0_MAT0
	LPC_IOCON->PIO1_7			= 0xC0;      // UART_TXD/CT32B0_MAT1
	LPC_IOCON->PIO3_3			= 0xC0;

	/* GPIOs at outputs */
	LPC_GPIO0->DIR = ~(1<<2);	//Trigger Pin input only
	//LPC_GPIO1->DIR = ~(1<<4);	//WAKEUP Pin input only
	//PIO1_6/RXD is configured as input since the transceiver pin is output.
	//if PIO1_6/RXD is configured as output, driven low and the transceiver is driving high then extra current
	LPC_GPIO1->DIR = ~((1<<4)|(1<<6));	//WAKEUP Pin and PIO1_6/RXD input only
	LPC_GPIO2->DIR = 0xFFFFFFFF;
	LPC_GPIO3->DIR = 0xFFFFFFFF;

	/* GPIO outputs to LOW */
	LPC_GPIO0->DATA  = 0;
	LPC_GPIO1->DATA  = 0;
	LPC_GPIO2->DATA  = 0;
	LPC_GPIO3->DATA  = 0;

	return;
}
