/*****************************************************************************
 *   pmuIPD.c:  PMU test module file for NXP LPC29xx Family Microprocessors
 *
 *   Copyright(C) 2007, NXP Semiconductor
 *   All rights reserved.
 *
 *   History
 *   2009.01.14  ver 1.00    First Release
 *
******************************************************************************/
#include "LPC29xx.h"			/* LPC29xx definitions */
#include "type.h"
#include "target.h"
#include "irq.h"
#include "LCD.h"
#include "extint.h"
#include "flashoff.h"   

#define SWITCH_CLOCKS_TO_XTAL 0         // 1: Switch all clocks to Xtal before PD to save some additional power
                                        // 0: only use the PMU PD features    
void GOSleepMode( void );

extern volatile DWORD eint_flag;

/*****************************************************************************
** Function name:		main
**
** Descriptions:		main routine for PMU module test
**
** parameters:			None
** Returned value:		int
**
*****************************************************************************/
int main( void )
{
    DWORD cnt=25; 
    unsigned long i;

    LCD_init ();
    LCD_cls ();
    LCD_print (0, 0, "LPC2900 PD DEMO ");
    LCD_print (0, 1, "  www.NXP.com   ");
    
    // Delay is necessary to allow reflashing, else the clocks could be disabled before
    // the JTAG communication can initialeze and you brick you MCU.
    for (i=0;i<=0x2FFFFF;i++){}

    GPIO1_DR = (1<<24)|(1<<25)|(1<<26)|(1<<27)|(1<<28)|(1<<29);
    
    // external interrupts are used as a wakeup event, select the event in exint.h
    EventRouter_Init();

    while ( 1 )
    {
        // external interrupt occurs.
        if ( eint_flag > 0 )
        {
            if(eint_flag & (1<<EXINT_sleep))
            {
                GOSleepMode(); // got to sleep
            }
            if(eint_flag & (1<<EXINT_wake))
            {
                GPIO1_OR ^= (1<<26); // toggle led
            }
            eint_flag = 0;
        }
        
        // blink led on port 1.27 to indicate we are running the main loop waiting for a switch 
        GPIO1_OR ^= (1<<27); 
        LCD_bargraph (0, 1, 16, cnt&0x7F); /* Display bargraph according to cnt   */                                         
        cnt++;        
    }
}


/******************************************************************************
** Function name:		GOSleepMode
**
** Descriptions:		Set clocks and go to sleep mode, the sleep mode
**						is turning off all the clocks, then, put the
**						the micro to power down mode which will turn off
**						the clocks with wakeup enabled. The wakeup souce
						is configured before going to sleep mode(power down).
**
** parameters:			None
** Returned value:		None
**
******************************************************************************/
void GOSleepMode( void )
{
    LCD_print (0, 0, "Sleeping..ZZzzz");

   // any pending interrupts will prevent the device from going in to power down mode, 
   // so all pending interrupts need to be handled before we can proceed. The pending 
   // interrupts in the VIC are meant with this, the interrupts in the core do not 
   // prevent the device from going to sleep.

   // Before we proceed we will disable the interrupt in the core. These wont interfere 
   // with going to sleep but will cause problems when waking up. If we wakeup by a interrupt 
   // the CPU will try to branch to a interrupt handler and approach hardware that isn't 
   // available just jet, so it will then jump to one of the exeption handlers what we don't want. 
   __disable_irq();
   __disable_fiq();

    // If LP_OSC and PLL are not present, no need to go further 
    while (!(CGU_RDET & (0x01|PLL_PRESENT)));

    // Set the system clock to run from LPosc
    SYS_CLK_CONF = AUTOBLK | DIV1;

#if SWITCH_CLOCKS_TO_XTAL
      // All the subsystem switched to run from X_tal. 
    IVNSS_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;
    MSCSS_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;
    UART_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;
    SPI_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;
    ADC_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;

    ICLK0_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;
    ICLK1_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;

    USB_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;
    USB_I2C_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;
    OUT_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;
    CTEST_CLK_CONF = CLK_SEL_XTAL | AUTOBLK | DIV1;
#endif

    // All the branch clocks are set as "wake-up enabled".
    // The CPU, SYS, PCR clockes can  only be disabled with the PMU_PD bit. 
    // bit2     bit1   bit0
    // WAKEUP | AUTO | RUN //
    PMU_CLK_CFG_CPU = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_SYS = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_PCR = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_FMC = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_RAM0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_RAM1 = (0x1<<2)|(0x1<<1)|(0x1<<0);

    // The following clocks can not be accessed unless XX_CLK_CONF are set,
    // can be either LP_OSC, ext. OSC, or PLL, or FDIVx
    // set all clocks to run, auto and wakeup enabled.
    PMU_CLK_CFG_SMC = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_GESS = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_VIC = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_PESS = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_GPIO0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_GPIO1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_GPIO2 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_GPIO3 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_IVNSSA = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_MSCSSA = (0x1<<2)|(0x1<<1)|(0x1<<0); 
    PMU_CLK_CFG_GPIO4 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_GPIO5 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_DMA = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_USB = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_PCR_IP = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_IVNSS_VPB = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_CANCA = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_CANC0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_CANC1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_I2C0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_I2C1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_LIN0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_LIN1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_MSCSS_VPB = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_MTMR0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_MTMR1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_PWM0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_PWM1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_PWM2 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_PWM3 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_ADC0_VPB = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_ADC1_VPB = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_ADC2_VPB = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_QEI = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_OUT_CLK = (0x1<<2)|(0x1<<1)|(0x1<<0); 
    PMU_CLK_CFG_UART0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_UART1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_SPI0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_SPI1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_SPI2 = (0x1<<2)|(0x1<<1)|(0x1<<0); 
    PMU_CLK_CFG_TMR0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_TMR1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_TMR2 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_TMR3 = (0x1<<2)|(0x1<<1)|(0x1<<0); 
    PMU_CLK_CFG_ADC0 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_ADC1 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_ADC2 = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_TSSHELL = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_USB_I2C = (0x1<<2)|(0x1<<1)|(0x1<<0);
    PMU_CLK_CFG_USB_CLK = (0x1<<2)|(0x1<<1)|(0x1<<0);

#if SWITCH_CLOCKS_TO_XTAL
    // power down clock slices on the subsystem.
    USB_CLK_CONF |= 0x01;
    USB_I2C_CLK_CONF |= 0x01;
    OUT_CLK_CONF |= 0x01;

    IVNSS_CLK_CONF |= 0x01;
    MSCSS_CLK_CONF |= 0x01;
    UART_CLK_CONF |= 0x01;
    SPI_CLK_CONF |= 0x01;
    ADC_CLK_CONF |= 0x01;
    TMR_CLK_CONF |= 0x01;
    ICLK0_CLK_CONF |= 0X01;
    ICLK1_CLK_CONF |= 0x01;
#endif

    CGU1_PLL_CTRL = 1; //Power Down CGU1 PLL
    CGU_PLL_CTRL = 1;  //Power Down CGU PLL
   
    // Power down OSC pads, still leave HF bit as default.
    CGU_OSC_CTRL = (0x1<<2) | (0x0<<0);
    
    EEPWRDWN =1;  // Power Down the EEPROM controller

    sleepFlashOff();// SleepflashOff is placed in TCM so flash can be disabled. see "flashoff.c"
                    // To locate the code in TCM the nice feature of the uVision is used.

                    //void sleepFlashOff( void )
                    //{
                    //    /* Disable flash (Power Down) */
                    //    FCTR |=0x200;
                    //	
                    //    /* Switch selected clocks off (resumed at event) */
                    //    PMU_PM =0x1;
                    //
                    //    /* Enable flash */
                    //    FCTR &=~0x200;
                    //
                    //	return;
                    //}

#if SWITCH_CLOCKS_TO_XTAL

    TargetResetInit();	// a quick way to get all clock configured again after wakeup
    EventRouter_Init();
#else
    CGU_OSC_CTRL |= (0x1<<2) | (0x1<<0);		// re-enable the osc
    while ( !(CGU_RDET & XTAL_PRESENT) ){ NOP; }    // wait for osc to start up
    
    // Restart PLL
    CGU_PLL_CTRL = PLL_XTAL_SEL | (PLL_M_VALUE<<MSEL_SHIFT) | P23EN;
    
    // Check lock bit, if unlocked, PLL_LOCK is always 0
    while ( !(CGU_PLL_STAT & PLL_LOCK) ) { NOP; }
    // Check clock detection register to make sure PLL is present now.
    while ( !(CGU_RDET & PLL_PRESENT) ) { NOP; }
    
    CGU_PLL_CTRL = PLL_XTAL_SEL | (PLL_M_VALUE<<MSEL_SHIFT) | P23EN;
    
    // PLL is 250Mhz, SYS_CLK and TMR_CLK is 125Mhz. This line depends on the board 
    // and configuration of clocks used see TargetResetInit() in target.c
    SYS_CLK_CONF = CLK_SEL_PLL | AUTOBLK | DIV2;
#endif
    
    // At this point all interrupts are still disabled and it's now save to 
    // re-enable them as all the hardware is up and running again.   
    __enable_fiq();
    __enable_irq();    

    LCD_print (0, 0, "     Awake      ");

    return;
}
/******************************************************************************
**                            End Of File
******************************************************************************/
