 /****************************************************************
 *		multi_tone_gen.c   -  Multi Tone Generator
 *
 *	    This module generates multiple sine waves that are combined
 *		to generate tones that contain a fundamental and 4 harmonics
 *		per the IEC60601-1-8 Medical Alarm specification 
 *
 *
 *   Copyright(C) 2007, NXP Semiconductor
 *   All rights reserved.
 *
 ****************************************************************/


// ----- Parameters and Variables for Sine Wave Generator ----- 

#include <stdio.h>                         /* standard I/O .h-file */
#include <LPC23xx.H>                       /* LPC23xx definitions  */
#include <math.h>
#include "timer.h"
#include "seqencer.h"


#define Amp 200                 // Output Amplitute     
#define PI  3.1415926
#define Fsample  25000             // Timer  Reload Frequency               

struct wave  {                      // struct for Sine Wave Generator Signal   
  short coef;                      	// IIR filter coefficient                
  long y1;                     		// y[-1] value                            
  long y2;                        	// y[-2] value      
};  

struct wave Waves[9][5];	// 'Waves' holds tone gen coefficients and variables
							// the coefficients are calculated during initialization.
long output;
long output_old;

enum {C4,D4,E4,F4,Fsharp4,G4,A4,B4,C5};	  // Can address array rows with notes

float const FreqArray[][5]= {{261.626,523.252,784.878,1046.50,1308.13}, 	// C4
							 {293.67,587.34,881.01,1174.7,1468.3},	   		// D4
							 {329.63,659.26,988.89,1318.52,1648.15},   		// E4
							 {349.23,698.46,1047.69,1396.9,1746.15},		// F4
							 {369.99,739.98,1109.97,1479.96,1849.95},		// FSharp4
							 {392.00,784.00,1176.0,1568.0,1960.0},			// G4
							 {440.000,880.00,1320.0,1760.00,2200.00},		// A4
							 {493.88,987.76,1481.64,1975.5,2469.4},			// B4
							 {523.251,1046.50,1569.756,2093.00,2616.25}};	// C5
	  
unsigned char ToneWeights[] = {255,255,255,255,255};	// used for test and  
														// ajusting harmonic levels
void InitToneCoefArray(void)   	// generate the coefficients and init array for tones 
{
  unsigned char n;
  unsigned char j;
  for (j=0;j<9;j++)			    // Initialize all nine scale tones (C4-C5)
  {
  	for (n=0;n<5;n++)	   		// fundamental and 4 harmonics for IEC60601-1-8
 	{
    	Waves[j][n].coef = ((cos (2*PI*(float)(FreqArray[j][n]/Fsample)))* 32768) ;	   // 2* taken out, put in final calc as a shift
  		Waves[j][n].y1  = 0;
  		Waves[j][n].y2  = ((sin (2*PI*(float)((FreqArray[j][n]/Fsample))) * Amp * 32768));	// Try 8388608 (+8 bits) w/ long coef
  	}
  }
}
void GenerateMultiTone (struct wave *t)  
{
  long y;
  unsigned char i;
  unsigned char env_weights;
  output = 0;				// clear output accumulator 
  for (i=0;i<5;i++)			// cycle through the 5 structures in the array
  {
  y = ((t->coef *(long long)(t->y1)>>14)) - t->y2;	 // Goertzel Calculation
  t->y2 = t->y1;									 // store for next time
  t->y1 = y;										 // store for next time
  env_weights = envelope * ToneWeights[i]>>8;
  output += ((t->y1* env_weights)>>8);	  // sum fundamental and harmonics
  t++;									  // increment structure pointer
  }
  DACR = ((output >> 10) & 0xFFC0) + 0x8000;  // make unsigned and output to DAC
  if ((output >= 0)&& (output_old <= 0))		// zero crossing detect
  {
     if (envelope_off && (note_on==0))
	 {
	  	envelope_on = 0;      // sychronizes turn off with zero cross
		envelope_off = 0;	  // reset envelope flag 
	 }
  }	 
  output_old = output;
}

void OutputTones(unsigned char note, unsigned char level)
{
  note_level = level;				   
  GenerateMultiTone (&Waves[note][0]);
}



