/*!
* @defgroup accelerometer Accelerometer
*
* @details Accelerometer module initializes and obtains acceleration measurements from the NXP FXOS8700CQ accelerometer plus magnetometer sensor.
* APIs to initialize, configure, write and read the accelerometer sensor are included in the FXOS8700CQ module.
* @{
*******************************************************************************/
/*!
* @file FXOS8700CQ.h
*
* @author  
*
* @version 1.0
*
* @date Mar-14-2016
*
* @brief 
*
********************************************************************************
*
* Copyright (c) 2016, Freescale Semiconductor.
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
* \defgroup fxos8700cq FXOS8700CQ
* @{
* The FXOS8700CQ module implements the functions to initialize, configure, and read the FXOS8700CQ
* accelerometer + magnetometer sensor from NXP. The sensor registers are described in the
* @ref fxos8700cq_registers section.
* 
* The current driver version is 1.0 and it does not include these features:
*
* 1.  Accelerometer filter
* 2.  Accelerometer pulse detection
* 3.  Accelerometer motion and free-fall
* 4.  Accelerometer acceleration transient
* 5.  Accelerometer orientation detection
* 6.  Accelerometer vector magnitude change
* 7.  FXOS8700 low-power configuration
* 8.  Magnetometer magnetic threshold
* 9.  Magnetometer vector-magnitude
* 10. Magnetometer magnetic Min/Max detection
* 11. FIFO configuration
* 12. SPI transport
*/

#ifndef _FXOS8700CQ_H_
#define _FXOS8700CQ_H_

/************************************************************************************
* Project Headers
************************************************************************************/
#include "FXOS8700CQ_registers.h"

/* KSDK */
#include "board.h"
#include "fsl_device_registers.h"
#include "EmbeddedTypes.h"

/************************************************************************************
* Macros
************************************************************************************/
/* Select just one transport. If both are selected, I2C takes precedence */
#define FXOS8700CQ_TRANSPORT_I2C                1                       /*!< I2C transport selected. Select just one transport. If both are selected, I2C takes precedence */
//No implementation for SPI on this driver version!! #define FXOS8700CQ_TRANSPORT_SPI                0                       /*!< SPI transport selected. Select just one transport. If both are selected, I2C takes precedence. */

/* I2C definitions */
#define FXOS8700CQ_I2C_ADDRESS                  0x1F                    /*!< I2C address configured for the FXOS8700CQ */
#define FXOS8700CQ_I2C_INSTANCE                 1                       /*!< I2C module instance to use */
#define FXOS8700CQ_I2C_BAUDRATE_KBPS            400                     /*!< Baudrate (in kbit/s) */

/* General definitions */
#define FXOS8700CQ_TIMEOUT_MS                   OSA_WAIT_FOREVER        /*!< Timeout for the blocking functions */

/************************************************************************************
* Type definitions
************************************************************************************/
/*!
 * Status responses for the FXOS8700CQ driver functions
 */
typedef enum _FXOS8700CQ_status{
  kStatusSuccess,                                               /*!< No error occured */
  kStatusTimeOutError,                                          /*!< Timeout error occured */
  kStatusInitializationError,                                   /*!< Error during initialization */
  kStatusTransportBusyError,                                    /*!< Transport is busy */
  kStatusCommunicationsError,                                   /*!< Error communicating with the FXOS8700CQ */
  kStatusMemoryAllocationError                                  /*!< Error trying to allocate memory */
}FXOS8700CQ_status_t;   

/*!
 * Configuration structure for the FXOS8700CQ initialization
 */
typedef struct _FXOS8700CQ_config{
  FXOS8700CQ_data_rate_hz_t outputDataRate;                     /*!< Set the output data rate */
  FXOS8700CQ_sensor_enable_t enabledSensors;                    /*!< Set the sensors to enable */
}FXOS8700CQ_config_t;
  
/*!
 * Configuration structure for the FXOS8700CQ interruptions initialization
 */
typedef struct _FXOS8700CQ_interrupt_config{    
  FXOS8700CQ_interrupt_sources_t interruptSources;              /*!< Interrupt sources to enable separated by | operator */
  FXOS8700CQ_interrupt_pin_map_t interruptPinMap;               /*!< Interrupt sources to map to INT2 separated by | operator (mapped to INT1 when clear) */
}FXOS8700CQ_interrupt_config_t;

/*!
 * Configuration structure for the accelerometer sensor initialization
 */
typedef struct _FXOS8700CQ_accelerometer_config{        
  FXOS8700CQ_sensitivity_t sensitivity;                         /*!< Accelerometer sensitivity */
  bool_t fastModeEnabled;                                       /*!< Enable fast mode; 8-bit output resolution when enabled */
  bool_t lowNoiseEnabled;                                       /*!< Enable low noise; does not work with +-8g sensitivity */
  FXOS8700CQ_oversampling_mods_t oversamplingMod;               /*!< Select the ADC oversampling mod */
}FXOS8700CQ_accelerometer_config_t;   

/*!
 * Configuration structure for the magnetometer sensor initialization
 */
typedef struct _FXOS8700CQ_magnetometer_config{
  FXOS8700CQ_magnetometer_osr_t oversamplingRatio;              /*!< Magnetometer oversampling ratio */
  FXOS8700CQ_magnetic_sensor_reset_t autoSensorResetFreq;       /*!< Magnetic sensor reset (degaussing) frequency */
  bool_t autoCalibrationEnabled;                                /*!< Enable sensor autocalibration */
}FXOS8700CQ_magnetometer_config_t;

/*!
 * Sensor ouput data presentation structure
 */
typedef struct _FXOS8700CQ_output_data{  
  uint8_t      status;                                          /*!< Output data status flags */
  int16_t      xAxisData;                                       /*!< X-Axis data in a signed 16-bit format */
  int16_t      yAxisData;                                       /*!< Y-Axis data in a signed 16-bit format */
  int16_t      zAxisData;                                       /*!< Z-Axis data in a signed 16-bit format */
}FXOS8700CQ_output_data_t;

/*!
 * FXOS8700CQ non-blocking functions callback type
 */
typedef void(*FXOS8700CQ_callback_function_t) (uint8_t* dataSource, uint8_t byteCount);

/************************************************************************************
* Function prototypes
************************************************************************************/

/*!****************************************************************************************
 * @brief Initializes the FXOS8700CQ module over the selected transport
 *
 * @param[in] pConfigStruct Module configuration structure (See @ref FXOS8700CQ_config_t )
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_init (FXOS8700CQ_config_t* pConfigStruct);

/*!****************************************************************************************
 * @brief Starts the FXOS8700CQ module
 *
 * @param[in] None
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_start (void);

/*!****************************************************************************************
 * @brief Stops the FXOS8700CQ module
 *
 * @param[in] None
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_stop (void);

/*!****************************************************************************************
 * @brief Executes communications tests by reading the WHO AM I register
 *
 * @param[in] None
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_communication_test(void);

/*!****************************************************************************************
 * @brief Configures the FXOS8700CQ module to generate interrupt requests using the INT1
 *        and INT2 pins.
 *
 * @param[in] pConfigurationParameters Interrupt configuration parameters (See @ref FXOS8700CQ_interrupt_config_t)
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_interrupt_configuration (FXOS8700CQ_interrupt_config_t* pConfigurationParameters);

/*!****************************************************************************************
 * @brief Gets the current interrupt status register
 *
 * @param[out] interruptStatus Pointer to the 8-bit variable where the result is to be stored
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_get_interrupt_status (uint8_t* interruptStatus);

/*!****************************************************************************************
 * @brief Configures the accelerometer sensor
 *
 * @param[in] pConfigurationParameters Configuration structure for the accelerometer (see @ref FXOS8700CQ_accelerometer_config_t)
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_accelerometer_configuration (FXOS8700CQ_accelerometer_config_t* pConfigurationParameters);

/*!****************************************************************************************
 * @brief Reads the accelerometer data from the FXOS8700CQ
 *
 * @param[out] pAccelerometerData Pointer to the structure where the results are to be stored
 *            (see @ref FXOS8700CQ_output_data_t)
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_get_accelerometer_readings (FXOS8700CQ_output_data_t* pAccelerometerData);

/*!****************************************************************************************
 * @brief Configures the magnetometer sensor
 *
 * @param[in] pConfigurationParameters Configuration structure for the magnetometer (see @ref FXOS8700CQ_magnetometer_config_t)
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_magnetometer_configuration (FXOS8700CQ_magnetometer_config_t* pConfigurationParameters);

/*!****************************************************************************************
 * @brief Reads the magnetometer data from the FXOS8700CQ
 *
 * @param[out] pMagnetometerData Pointer to the structure where the results are to be stored
 *            (see @ref FXOS8700CQ_output_data_t)
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_get_magnetometer_readings (FXOS8700CQ_output_data_t* pMagnetometerData);

/*!****************************************************************************************
 * @brief Reads the accelerometer and magnetometer data in a single sequence
 *
 * @param[out] pAccelerometerData Pointer to the structure where the results are to be stored
 *            (see @ref FXOS8700CQ_output_data_t)
 * @param[out] pMagnetometerData Pointer to the structure where the results are to be stored
 *            (see @ref FXOS8700CQ_output_data_t)
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 *
 * @note When using this function to read the magnetometer data, the magnetometer STATUS byte in the 
 *       struct is not updated.
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_get_hybrid_sensor_readings (FXOS8700CQ_output_data_t* pAccelerometerData, FXOS8700CQ_output_data_t* pMagnetometerData);

/****************************** Auxiliary Functions ********************************/

/*!****************************************************************************************
 * @brief Reads a sequence of the FXOS8700CQ registers
 *
 * @param[in] startRegisterAddress Address for the first register to read
 * @param[in] byteCount Number of registers to read
 * @param[in] onCompletionCallback Callback function to execute upon read complete. This function
 *            must be compatible with the @ref FXOS8700CQ_callback_function_t type.
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_get_registers (uint8_t startRegisterAddress, uint8_t byteCount, FXOS8700CQ_callback_function_t onCompletionCallback);

/*!****************************************************************************************
 * @brief Reads a sequence of the FXOS8700CQ registers blocking the execution until completion.
 *
 * @param[in] startRegisterAddress Address for the first register to read
 * @param[in] byteCount Number of registers to read
 * @param[out] pOutBuffer Pointer to the buffer where the data are to be stored
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_get_registers_blocking (uint8_t startRegisterAddress, uint8_t byteCount, uint8_t* pOutBuffer);

/*!****************************************************************************************
 * @brief Starts a FXOS8700CQ register write.
 *
 * @param[in] registerAddress Address for the register to write.
 * @param[in] registerDataPtr Pointer to the variable containing the data to write.
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_set_register (uint8_t registerAddress, uint8_t* registerDataPtr);

/*!****************************************************************************************
 * @brief Starts a FXOS8700CQ register write and waits until it is complete.
 *
 * @param[in] registerAddress Address for the register to write.
 * @param[in] registerDataPtr Pointer to the variable containing the data to write.
 *
 * @return @ref FXOS8700CQ_status_t Function execution status
 ******************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_set_register_blocking (uint8_t registerAddress, uint8_t* registerDataPtr);

/*!
* @} End of fxos8700cq
*/

/*!
* @} End of accelerometer
*/

/**********************************************************************************/
#endif //End of file