/*!
* Copyright (c) 2015, Freescale Semiconductor, Inc.
* All rights reserved.
*
* \file PhyISR.c
* PHY ISR Functions
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor, Inc. nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*! *********************************************************************************
*************************************************************************************
* Include
*************************************************************************************
********************************************************************************** */
#include "EmbeddedTypes.h"
#include "Phy.h"
#include "PhyInterface.h"
#include "FunctionLib.h"

#include "fsl_os_abstraction.h"
#include "fsl_device_registers.h"

#include "KW4xXcvrDrv.h"

/*! *********************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
********************************************************************************** */
#define gPhyIrqPriority_c     (0x80)

#if gUsePBTransferThereshold_d
  #define mPhyGetPBTransferThreshold(len) ((len) - 2)
#endif


/*! *********************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
********************************************************************************** */
extern Phy_PhyLocalStruct_t     phyLocal;
static volatile phyRxParams_t * mpRxParams        = NULL;
static const uint8_t            mPhyInstance      = 0;
uint8_t                         mPhyLastRxLQI     = 0;
uint8_t                         mPhyLastRxRSSI    = 0;
uint8_t                         mPhyIrqDisableCnt = 1;
bool_t                          mPhyForceFP = FALSE;


/*! *********************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
********************************************************************************** */
static void PhyIsrSeqCleanup();
static void PhyIsrTimeoutCleanup();
static void Phy_GetRxParams(void);
static uint8_t Phy_LqiConvert(uint8_t hwLqi);


/*! *********************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
********************************************************************************** */

/*! *********************************************************************************
* \brief  Disable the 802.15.4 radio IRQ
*
********************************************************************************** */
void ProtectFromXcvrInterrupt(void)
{
    OSA_EnterCritical(kCriticalDisableInt);
   
    if( mPhyIrqDisableCnt == 0 )
    {
        ZLL_BWR_PHY_CTRL_TRCV_MSK(ZLL, 1);
    }
    
    mPhyIrqDisableCnt++;
    
    OSA_ExitCritical(kCriticalDisableInt);
}

/*! *********************************************************************************
* \brief  Enable the 802.15.4 radio IRQ
*
********************************************************************************** */
void UnprotectFromXcvrInterrupt(void)
{
    OSA_EnterCritical(kCriticalDisableInt);
    
    if( mPhyIrqDisableCnt )
    {
        mPhyIrqDisableCnt--;
        
        if( mPhyIrqDisableCnt == 0 )
        {
            ZLL_BWR_PHY_CTRL_TRCV_MSK(ZLL, 0);
        }
    }

    OSA_ExitCritical(kCriticalDisableInt);
}

/*! *********************************************************************************
* \brief  Sets the location of the Rx parameters
*
* \param[in]  pRxParam pointer to Rx parameters
*
********************************************************************************** */
void PhyIsrPassRxParams
(
  volatile phyRxParams_t * pRxParam
)
{
    mpRxParams = pRxParam;
}

/*! *********************************************************************************
* \brief  Clear and mask PHY IRQ, set sequence to Idle
*
********************************************************************************** */
static void PhyIsrSeqCleanup
(
  void
)
{
    uint32_t irqSts;

    /* Mask SEQ, RX, TX and CCA interrupts */
    ZLL_PHY_CTRL |= ZLL_PHY_CTRL_CCAMSK_MASK |
                    ZLL_PHY_CTRL_RXMSK_MASK  |
                    ZLL_PHY_CTRL_TXMSK_MASK  |
                    ZLL_PHY_CTRL_SEQMSK_MASK;
    /* Set the PHY sequencer back to IDLE */
    ZLL_PHY_CTRL &= ~(ZLL_PHY_CTRL_XCVSEQ_MASK);

    while( ZLL_SEQ_STATE & ZLL_SEQ_STATE_SEQ_STATE_MASK );

    irqSts = ZLL_IRQSTS;
    /* Mask TMR3 interrupt */
    irqSts |= ZLL_IRQSTS_TMR3MSK_MASK;
    /* Clear transceiver interrupts except TMRxIRQ */
    irqSts &= ~( ZLL_IRQSTS_TMR1IRQ_MASK |
                 ZLL_IRQSTS_TMR2IRQ_MASK |
                 ZLL_IRQSTS_TMR3IRQ_MASK |
                 ZLL_IRQSTS_TMR4IRQ_MASK );
    ZLL_IRQSTS = irqSts;
#if gMWS_Enabled_d
    XcvrChangeMode(BLE);
#endif
}

/*! *********************************************************************************
* \brief  Clear and mask PHY IRQ, disable timeout, set sequence to Idle
*
********************************************************************************** */
static void PhyIsrTimeoutCleanup
(
  void
)
{
    uint32_t irqSts;

    /* Mask SEQ, RX, TX and CCA interrupts */
    ZLL_PHY_CTRL |= ZLL_PHY_CTRL_CCAMSK_MASK |
                    ZLL_PHY_CTRL_RXMSK_MASK  |
                    ZLL_PHY_CTRL_TXMSK_MASK  |
                    ZLL_PHY_CTRL_SEQMSK_MASK;
    /* Disable TMR3 comparator and timeout */
    ZLL_PHY_CTRL &= ~(ZLL_PHY_CTRL_TMR3CMP_EN_MASK | ZLL_PHY_CTRL_TC3TMOUT_MASK);
    /* Set the PHY sequencer back to IDLE */
    ZLL_PHY_CTRL &= ~(ZLL_PHY_CTRL_XCVSEQ_MASK);

    while( ZLL_SEQ_STATE & ZLL_SEQ_STATE_SEQ_STATE_MASK );

    irqSts = ZLL_IRQSTS;
    /* Mask TMR3 interrupt */
    irqSts |= ZLL_IRQSTS_TMR3MSK_MASK;
    /* Clear transceiver interrupts except TMR1IRQ, TMR2IRQ and TMR4IRQ. */
    irqSts &= ~( ZLL_IRQSTS_TMR1IRQ_MASK |
                 ZLL_IRQSTS_TMR2IRQ_MASK |
                 ZLL_IRQSTS_TMR4IRQ_MASK );
    ZLL_IRQSTS = irqSts;
#if gMWS_Enabled_d
    XcvrChangeMode(BLE);
#endif
}

/*! *********************************************************************************
* \brief  Scales energy level to 0-255
*
* \param[in]  energyLevel  the energy level reported by HW
*
* \return  uint8_t  the energy level scaled in 0x00-0xFF
*
********************************************************************************** */
uint8_t Phy_GetEnergyLevel
(
uint8_t energyLevel /* [dbm] */
)
{
    int32_t temp = (int8_t)energyLevel;

    if( temp <= -78 )
        return 0x00;

    if( temp >=0 )
        return 0xFF;

    /* Convert energy level from dbm into a 0x00-0xFF value */
    temp = 255*temp/78 + 255;
 
    return (uint8_t)temp;
}

/*! *********************************************************************************
* \brief  This function returns the LQI for the las received packet
*
* \return  uint8_t  LQI value
*
********************************************************************************** */
uint8_t PhyGetLastRxLqiValue(void)
{
    return mPhyLastRxLQI;
}

/*! *********************************************************************************
* \brief  This function returns the RSSI for the las received packet
*
* \return  uint8_t  RSSI value
*
********************************************************************************** */
uint8_t PhyGetLastRxRssiValue(void)
{
    /*RSSI*/
    /*
    **       LQI
    **RSSI = ---  - LQI_COMP
    **       2.25
    */
    uint8_t LQI_COMP = ZLL_RD_CCA_LQI_CTRL_LQI_OFFSET_COMP(ZLL);
    int32_t lqi_to_rssi= ((mPhyLastRxRSSI * 456) >> 10) - LQI_COMP;
    
    return (uint8_t)lqi_to_rssi;
}
    
/*! *********************************************************************************
* \brief  This function converts the LQI reported by the PHY into an signed RSSI value
*
* \param[in]  LQI  the LQI reported by the PHY
*
* \return  the RSSI value in dbm
*
********************************************************************************** */
int8_t PhyConvertLQIToRSSI(uint8_t lqi)
{
    int32_t rssi = 6*lqi/17 - 95;

    return (int8_t)rssi;
}

/*! *********************************************************************************
* \brief  PHY ISR
*
********************************************************************************** */
void PHY_InterruptHandler(void)
{
    uint8_t xcvseqCopy;
    uint32_t irqSts;

    /* Mask XCVR interrupts */
    ProtectFromXcvrInterrupt();

    /* Read current XCVRSEQ and interrup status */
    xcvseqCopy = ZLL_PHY_CTRL & ZLL_PHY_CTRL_XCVSEQ_MASK;
    irqSts     = ZLL_IRQSTS;
    /* Clear all xcvr interrupts */
    ZLL_IRQSTS = irqSts;

    /* Flter Fail IRQ */
    if( (irqSts & ZLL_IRQSTS_FILTERFAIL_IRQ_MASK) &&
       !(ZLL_PHY_CTRL & ZLL_PHY_CTRL_FILTERFAIL_MSK_MASK) )
    {
        Radio_Phy_PlmeFilterFailRx(mPhyInstance);
    }
    /* Rx Watermark IRQ */
    else if( (irqSts & ZLL_IRQSTS_RXWTRMRKIRQ_MASK) &&
            !(ZLL_PHY_CTRL & ZLL_PHY_CTRL_RX_WMRK_MSK_MASK) )
    {
        Radio_Phy_PlmeRxWatermark(mPhyInstance, ZLL_RD_IRQSTS_RX_FRAME_LENGTH(ZLL));
    }

    /* Timer 1 Compare Match */
    if( (irqSts & ZLL_IRQSTS_TMR1IRQ_MASK) &&
       !(irqSts & ZLL_IRQSTS_TMR1MSK_MASK) )
    {
        PhyTimeDisableWaitTimeout();

        Radio_Phy_TimeWaitTimeoutIndication(mPhyInstance);
    }

    /* Sequencer interrupt, the autosequence has completed */
    if( (irqSts & ZLL_IRQSTS_SEQIRQ_MASK) &&
       !(ZLL_PHY_CTRL & ZLL_PHY_CTRL_SEQMSK_MASK))
    {
        /* PLL unlock, the autosequence has been aborted due to PLL unlock */
        if( irqSts & ZLL_IRQSTS_PLL_UNLOCK_IRQ_MASK )
        {
            PhyIsrSeqCleanup();
            Radio_Phy_PlmeSyncLossIndication(mPhyInstance);
            UnprotectFromXcvrInterrupt();
            return;
        }

        /* TMR3 timeout, the autosequence has been aborted due to TMR3 timeout */
        if( (irqSts & ZLL_IRQSTS_TMR3IRQ_MASK) &&
           !(irqSts & ZLL_IRQSTS_RXIRQ_MASK) &&
            (gTX_c != xcvseqCopy) )
        {
            PhyIsrTimeoutCleanup();
            Radio_Phy_TimeRxTimeoutIndication(mPhyInstance);
            UnprotectFromXcvrInterrupt();
            return;
        }

        PhyIsrSeqCleanup();

        switch(xcvseqCopy)
        {
        case gTX_c:
            if( (irqSts & ZLL_IRQSTS_CCA_MASK ) &&
               (ZLL_PHY_CTRL & ZLL_PHY_CTRL_CCABFRTX_MASK) )
            {
                Radio_Phy_PlmeCcaConfirm(gPhyChannelBusy_c, mPhyInstance);
            }
            else
            {
                Radio_Phy_PdDataConfirm(mPhyInstance, FALSE);
            }
            break;

        case gTR_c:
            if( (irqSts & ZLL_IRQSTS_CCA_MASK ) &&
                (ZLL_PHY_CTRL & ZLL_PHY_CTRL_CCABFRTX_MASK) )
            {
                Radio_Phy_PlmeCcaConfirm(gPhyChannelBusy_c, mPhyInstance);
            }
            else
            {
                Phy_GetRxParams();
                if( irqSts & ZLL_IRQSTS_RX_FRM_PEND_MASK )
                {
                    Radio_Phy_PdDataConfirm(mPhyInstance, TRUE);
                }
                else
                {
                    Radio_Phy_PdDataConfirm(mPhyInstance, FALSE);
                }
            }
            break;

        case gRX_c:
            /* Check SAA0 and SAA1 (source address absent) */
            if( irqSts & ZLL_IRQSTS_PI_MASK )
            {
                /* Save the state of the FP bit sent in ACK frame */
                if( PhyPpIsTxAckDataPending() )
                {
                    phyLocal.flags |= gPhyFlagTxAckFP_c;
                }
                else
                {
                    phyLocal.flags &= ~gPhyFlagTxAckFP_c;
                }

                if( ZLL_SAM_MATCH & (ZLL_SAM_MATCH_SAA0_ADDR_ABSENT_MASK | ZLL_SAM_MATCH_SAA1_ADDR_ABSENT_MASK) )
                {
                    mPhyForceFP = TRUE;
                }
            }
            
            Phy_GetRxParams();
            Radio_Phy_PdDataIndication(mPhyInstance);
            break;

        case gCCA_c:
            if( ZLL_RD_PHY_CTRL_CCATYPE(ZLL) == gCcaED_c )
            {
                Radio_Phy_PlmeEdConfirm( ZLL_RD_LQI_AND_RSSI_CCA1_ED_FNL(ZLL), mPhyInstance );
            }
            else /* CCA */
            {
                if( irqSts & ZLL_IRQSTS_CCA_MASK )
                {
                    Radio_Phy_PlmeCcaConfirm(gPhyChannelBusy_c, mPhyInstance);
                }
                else
                {
                    Radio_Phy_PlmeCcaConfirm(gPhyChannelIdle_c, mPhyInstance);
                }
            }
            break;

        case gCCCA_c:
            Radio_Phy_PlmeCcaConfirm(gPhyChannelIdle_c, mPhyInstance);
            break;

        default:
            Radio_Phy_PlmeSyncLossIndication(mPhyInstance);
            break;
        }
    }
    /* Timers interrupt */
    else
    {
        /* Timer 2 Compare Match */
        if( (irqSts & ZLL_IRQSTS_TMR2IRQ_MASK) &&
           !(irqSts & ZLL_IRQSTS_TMR2MSK_MASK) )
        {
            PhyTimeDisableEventTrigger();

            if( gIdle_c != xcvseqCopy )
            {
                Radio_Phy_TimeStartEventIndication(mPhyInstance);
            }
        }

        /* Timer 3 Compare Match */
        if( (irqSts & ZLL_IRQSTS_TMR3IRQ_MASK) &&
           !(irqSts & ZLL_IRQSTS_TMR3MSK_MASK) )
        {
            PhyTimeDisableEventTimeout();

            /* Ensure that we're not issuing TimeoutIndication while the Automated sequence is still in progress */
            /* TMR3 can expire during R-T turnaround for example, case in which the sequence is not interrupted */
            if( gIdle_c == xcvseqCopy )
            {
                Radio_Phy_TimeRxTimeoutIndication(mPhyInstance);
            }
        }

        /* Timer 4 Compare Match */
        if( (irqSts & ZLL_IRQSTS_TMR4IRQ_MASK) &&
           !(irqSts & ZLL_IRQSTS_TMR4MSK_MASK) )
        {
            uint32_t irqSts = ZLL_IRQSTS;

            /* Disable TMR4 comparator */
            ZLL_BWR_PHY_CTRL_TMR4CMP_EN(ZLL, 0);
            /* Mask and clear TMR4 interrupt (do not change other IRQ status) */
            irqSts &= ~( ZLL_IRQSTS_TMR1MSK_MASK |
                         ZLL_IRQSTS_TMR2MSK_MASK |
                         ZLL_IRQSTS_TMR3MSK_MASK);
            irqSts |= ZLL_IRQSTS_TMR4IRQ_MASK | ZLL_IRQSTS_TMR4MSK_MASK;
            ZLL_IRQSTS = irqSts;
        }
    }

    UnprotectFromXcvrInterrupt();
}

/*! *********************************************************************************
* \brief  This function installs the PHY ISR
*
********************************************************************************** */
void PHY_InstallIsr( void )
{
    OSA_InstallIntHandler(ZigBee_IRQn, PHY_InterruptHandler);

    /* enable transceiver SPI interrupt request */
    NVIC_ClearPendingIRQ(ZigBee_IRQn);
    NVIC_EnableIRQ(ZigBee_IRQn);
    
    /* set transceiver interrupt priority */
    NVIC_SetPriority(ZigBee_IRQn, gPhyIrqPriority_c >> (8 - __NVIC_PRIO_BITS));
    UnprotectFromXcvrInterrupt();
}

/*! *********************************************************************************
 * \brief Check is an XCVR IRQ is pending
 *
 * \return  TRUE if ISR pending, else FALSE
 *
 ********************************************************************************** */
bool_t PHY_isIrqPending(void)
{
    uint32_t temp;
    
    if( !(ZLL_PHY_CTRL & ZLL_PHY_CTRL_TRCV_MSK_MASK) )
    {
        /* Check usual ZLL IRQs */
        temp = ZLL_PHY_CTRL & ( ZLL_PHY_CTRL_SEQMSK_MASK         |
                                ZLL_PHY_CTRL_TXMSK_MASK          |
                                ZLL_PHY_CTRL_RXMSK_MASK          |
                                ZLL_PHY_CTRL_CCAMSK_MASK         |
                                ZLL_PHY_CTRL_RX_WMRK_MSK_MASK    |
                                ZLL_PHY_CTRL_FILTERFAIL_MSK_MASK |
                                ZLL_PHY_CTRL_PLL_UNLOCK_MSK_MASK );
        temp = temp >> ZLL_PHY_CTRL_SEQMSK_SHIFT;
        
        if( ZLL_IRQSTS & temp )
        {
            return TRUE;
        }
        
        /* Check ZLL Timers IRQs */
        temp = ZLL_IRQSTS & ( ZLL_IRQSTS_TMR1IRQ_MASK |
                              ZLL_IRQSTS_TMR2IRQ_MASK |
                              ZLL_IRQSTS_TMR3IRQ_MASK |
                              ZLL_IRQSTS_TMR4IRQ_MASK );
        temp = temp >> 4;
        
        if( ZLL_IRQSTS & temp )
        {
            return TRUE;
        }
    }

    return FALSE;
}

/*! *********************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
********************************************************************************* */

/*! *********************************************************************************
* \brief  Fill the Rx parameters: RSSI, LQI, Timestamp and PSDU length
*
********************************************************************************** */
static void Phy_GetRxParams(void)
{
    if(NULL != mpRxParams)
    {
        /* Reports value of 0x00 for -105 dBm of received input power 
        and 0xFF for 0 dBm of received input power */
        mPhyLastRxRSSI = ZLL_RD_LQI_AND_RSSI_LQI_VALUE(ZLL);
        mPhyLastRxLQI = Phy_LqiConvert(mPhyLastRxRSSI);
        mpRxParams->linkQuality = mPhyLastRxLQI;
        mpRxParams->timeStamp = ZLL_TIMESTAMP;
        mpRxParams->psduLength = ZLL_RD_IRQSTS_RX_FRAME_LENGTH(ZLL); /* Including FCS (2 bytes) */
        mpRxParams = NULL;
    }
}

/*! *********************************************************************************
* \brief  Scales LQI to 0-255
*
* \param[in]  hwLqi  the LQI reported by HW
*
* \return  uint8_t  the LQI scaled in 0x00-0xFF
*
********************************************************************************** */
static uint8_t Phy_LqiConvert
(
uint8_t hwLqi
)
{
    uint8_t LQI_COMP = ZLL_RD_CCA_LQI_CTRL_LQI_OFFSET_COMP(ZLL);
    /* Convert the LQI reported by HW into RSSI (dbm) */
    int32_t rssi = ((hwLqi * 456) >> 10) - LQI_COMP;

    if( rssi <= -95 )
        return 0x00;

    if( rssi >= -5 )
        return 0xFF;

    /* Convert the RSSI value into a 0x00-0xFF value */
    rssi = ((343*rssi)>>7) + 255;

    return rssi;
}