/*! *********************************************************************************
 * \addtogroup FRDM-KW40Z Demo Custom Profile
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * \file remote_controller_service.c
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
#include "FunctionLib.h"
#include "ble_general.h"
#include "gatt_db_app_interface.h"
#include "gatt_server_interface.h"
#include "gap_interface.h"
#include "remote_controller_interface.h"
#include "ir_controller.h"
#include "controller_parameter_list.h"
/************************************************************************************
*************************************************************************************
* Private constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
/*! Remote Controller Service - Subscribed Client*/
static deviceId_t mRcs_SubscribedClientId;

//Characteristic declarations
const bleUuid_t controllerCommandCharacteristicUuid128 = {
  .uuid128 = {0xE0, 0x1C, 0x4B, 0x5E, 0x1E, 0xEB, 0xA1, 0x5C, 0xEE, 0xF4, 0x5E, 0xBA, 0x08, 0x57, 0xFF, 0x02}
};

const bleUuid_t controllerConfigurationCharacteristicUuid128 = {
  .uuid128 = {0xE0, 0x1C, 0x4B, 0x5E, 0x1E, 0xEB, 0xA1, 0x5C, 0xEE, 0xF4, 0x5E, 0xBA, 0x09, 0x57, 0xFF, 0x02}
};

uint8_t currentController = 0;

/************************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
bleResult_t Rcs_Start (rcsConfig_t *pServiceConfig)
{    
    mRcs_SubscribedClientId = gInvalidDeviceId_c;
    
    return Rcs_SetConfiguration (pServiceConfig->serviceHandle, 
                                 pServiceConfig->controllerConfiguration);
}

bleResult_t Rcs_Stop (rcsConfig_t *pServiceConfig)
{
    return Rcs_Unsubscribe();
}

bleResult_t Rcs_Subscribe(deviceId_t deviceId)
{
    mRcs_SubscribedClientId = deviceId;

    return gBleSuccess_c;
}

bleResult_t Rcs_Unsubscribe()
{
    mRcs_SubscribedClientId = gInvalidDeviceId_c;
    return gBleSuccess_c;
}

bleResult_t Rcs_SetCommand (uint16_t serviceHandle, uint8_t commandValue){
  uint16_t  handle;
  bleResult_t result;
  
  /* Make sure that the command is valid */
  if(commandValue >= kRcsInvalidCommand)
    return gCustomCommandNotSupported_c;
  
  /* Make sure that the IR Controller module is not currently executing a command */
  if(irControllerStatusOk != ir_controller_get_current_status()){
    return gCustomRemoteControllerBusy_c;
  }
  
  /* Execute command */
  ir_controller_send_command(&controllerParameterList[currentController].controllerTimingParameters,
                             &controllerParameterList[currentController].controllerCommands[commandValue], 
                             controllerParameterList[currentController].bitsPerCommand, 
                             controllerParameterList[currentController].commandRepeatTimes);
  
  /* Update status in GATT database */
  /* Get handle of Remote Controller Command characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType128_c, (bleUuid_t*)&controllerCommandCharacteristicUuid128, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value */
    result = GattDb_WriteAttribute(handle, sizeof(uint8_t), (uint8_t*)&commandValue);

    if (result != gBleSuccess_c)
        return result;
    
    return gBleSuccess_c;
}

bleResult_t Rcs_SetConfiguration (uint16_t serviceHandle, uint8_t configurationValue){
  uint16_t  handle;
  bleResult_t result;
  
  /* Make sure that the configuration is valid */
  if(configurationValue >= kRcsBrandInvalid)
    return gCustomBrandSelectionNotSupported_c;
  
  /* Update configuration */
  currentController = configurationValue;
  
  /* Update status in GATT database */
  /* Get handle of Remote Controller Configuration characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType128_c, (bleUuid_t*)&controllerConfigurationCharacteristicUuid128, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value */
    result = GattDb_WriteAttribute(handle, sizeof(uint8_t), (uint8_t*)&configurationValue);

    if (result != gBleSuccess_c)
        return result;
    
    return gBleSuccess_c;
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/*! *********************************************************************************
 * @}
 ********************************************************************************** */
