/*! *********************************************************************************
 * \addtogroup FRDM-KW40Z Demo Custom Profile
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * \file led_control_service.c
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
#include "FunctionLib.h"
#include "ble_general.h"
#include "gatt_db_app_interface.h"
#include "gatt_server_interface.h"
#include "gap_interface.h"
#include "led_control_interface.h"
#include "led_control.h"
/************************************************************************************
*************************************************************************************
* Private constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
/*! LED Control Service - Subscribed Client*/
static deviceId_t mLcs_SubscribedClientId;

//Characteristic declarations
const bleUuid_t ledStatusCharacteristicUuid128 = {
  .uuid128 = {0xE0, 0x1C, 0x4B, 0x5E, 0x1E, 0xEB, 0xA1, 0x5C, 0xEE, 0xF4, 0x5E, 0xBA, 0x00, 0x57, 0xFF, 0x02}
};

const bleUuid_t ledControlCharacteristicUuid128 = {
  .uuid128 = {0xE0, 0x1C, 0x4B, 0x5E, 0x1E, 0xEB, 0xA1, 0x5C, 0xEE, 0xF4, 0x5E, 0xBA, 0x01, 0x57, 0xFF, 0x02}
};

/************************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
************************************************************************************/
//static void Hts_SendTemperatureMeasurementNotification
//(
//    uint16_t handle
//);
/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
bleResult_t Lcs_Start (lcsConfig_t *pServiceConfig)
{    
    mLcs_SubscribedClientId = gInvalidDeviceId_c;
    
    return Lcs_RecordNewLedStatus (pServiceConfig->serviceHandle, 
                                   pServiceConfig->ledStatus);
}

bleResult_t Lcs_Stop (lcsConfig_t *pServiceConfig)
{
    return Lcs_Unsubscribe();
}

bleResult_t Lcs_Subscribe(deviceId_t deviceId)
{
    mLcs_SubscribedClientId = deviceId;

    return gBleSuccess_c;
}

bleResult_t Lcs_Unsubscribe()
{
    mLcs_SubscribedClientId = gInvalidDeviceId_c;
    return gBleSuccess_c;
}

bleResult_t Lcs_RecordNewLedStatus (uint16_t serviceHandle, uint8_t ledStatus)
{
    uint16_t  handle;
    bleResult_t result;

    /* Get handle of LED Status characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType128_c, (bleUuid_t*)&ledStatusCharacteristicUuid128, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value */
    result = GattDb_WriteAttribute(handle, sizeof(uint8_t), (uint8_t*)&ledStatus);

    if (result != gBleSuccess_c)
        return result;

    return gBleSuccess_c;
}

bleResult_t Lcs_SetNewLedValue (lcsConfig_t* pLedConfiguration){
    uint16_t  handle = pLedConfiguration->serviceHandle;
    bleResult_t result;
    uint8_t ledStatus;
    uint16_t byteCount;
    led_control_status_t ledControlStatus;
      
    /* Update LED */
    ledControlStatus = led_control_update_led (pLedConfiguration->ledControl.ledNumber, (led_control_command_t)pLedConfiguration->ledControl.ledCommand);
    
    /* Check for errors */
    if(ledControlStatus == kStatusLedNumberOutOfRange)
      return gCustomLedOutOfRange_c;
    
    if(ledControlStatus == kStatusInvalidCommand)
      return gCustomCommandNotSupported_c;
    
    if(ledControlStatus == kStatusUnexpectedError)
      return gBleUnexpectedError_c;
    
    /* Update LED status in GATT database */
    /* Get handle of LED Control characteristic */
    result = GattDb_FindCharValueHandleInService(handle,
        gBleUuidType128_c, (bleUuid_t*)&ledStatusCharacteristicUuid128, &handle);

    if (result != gBleSuccess_c)
        return result;
    
    /* Read current led status in ATT database */
    result = GattDb_ReadAttribute(handle, sizeof(uint8_t), (uint8_t*)&ledStatus, (uint16_t*)&byteCount);

    if (result != gBleSuccess_c)
        return result;
    
    /* Modify LED status with latest changes */
    switch(pLedConfiguration->ledControl.ledCommand){
    case kLedOff:
      ledStatus &= ~(1<<pLedConfiguration->ledControl.ledNumber);
      break;
      
    case kLedOn:
      ledStatus |= (1<<pLedConfiguration->ledControl.ledNumber);
      break;
      
    case kLedToggle:
      ledStatus ^= (1<<pLedConfiguration->ledControl.ledNumber);
      break;
      
    default:
      break;
    }
    
    /* Update characteristic value */
    result = GattDb_WriteAttribute(handle, sizeof(uint8_t), (uint8_t*)&ledStatus);

    if (result != gBleSuccess_c)
        return result;
    
    
    return gBleSuccess_c;
}


/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/*! *********************************************************************************
 * @}
 ********************************************************************************** */
