/*! *********************************************************************************
 * \addtogroup Temperature Custom Profile
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * \file smart_lock_service.c
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
#include "FunctionLib.h"
#include "ble_general.h"
#include "gatt_db_app_interface.h"
// Include this file for the 128 bit characteristic UUIDs. Do not access the handles directlty!
#include "gatt_db_handles.h"
#include "gatt_server_interface.h"
#include "gap_interface.h"
#include "smart_lock_interface.h"
/************************************************************************************
*************************************************************************************
* Private constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

/*! Smart lock Service - Subscribed Client*/
static deviceId_t mSml_SubscribedClientId;

/************************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
************************************************************************************/
static bleResult_t Sml_SendSmartLockNotification(uint16_t handle);
static bleResult_t Sml_SendSmartLockIndication(uint16_t handle);
/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
bleResult_t Sml_Start (smartlockConfig_t *pServiceConfig)
{    
    mSml_SubscribedClientId = gInvalidDeviceId_c;
    
    return Sml_RecordSmartLockControl(pServiceConfig->serviceHandle, 
                                             pServiceConfig->initialLockStatus);
}

bleResult_t Sml_Stop (smartlockConfig_t *pServiceConfig)
{
    return Sml_Unsubscribe();
}

bleResult_t Sml_Subscribe(deviceId_t deviceId)
{
    mSml_SubscribedClientId = deviceId;

    return gBleSuccess_c;
}

bleResult_t Sml_Unsubscribe()
{
    mSml_SubscribedClientId = gInvalidDeviceId_c;
    return gBleSuccess_c;
}

bleResult_t Sml_SetSmartLockStatus(uint16_t serviceHandle, uint8_t lock_status)
{
    uint16_t  hSmartLockStatus;
    bleResult_t result;
    bleUuid_t *pUuid = (bleUuid_t*)&uuid_char_smart_lock_status;

    /* Get handle or Heart Rate Measurement characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType128_c, pUuid, &hSmartLockStatus);

    if (result != gBleSuccess_c)
    {
        return result;
    }
    
    return GattDb_WriteAttribute(hSmartLockStatus, sizeof(uint8_t), &lock_status);
}

bleResult_t Sml_RecordSmartLockControl(uint16_t serviceHandle, uint8_t lock_control)
{
    uint16_t  hSmartLockStatus;
    bleResult_t result;
    bleUuid_t *pUuid = (bleUuid_t*)&uuid_char_smart_lock_status;

    /* Get handle of Temperature characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType128_c, pUuid, &hSmartLockStatus);

    if (result != gBleSuccess_c)
    {
        return result;
    }

    /* Update characteristic value */
    result = GattDb_WriteAttribute(hSmartLockStatus, sizeof(uint8_t), (uint8_t *)&lock_control);

    if (result != gBleSuccess_c)
        return result;

    Sml_SendSmartLockNotification(hSmartLockStatus);

    return gBleSuccess_c;
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

static bleResult_t Sml_SendSmartLockNotification(uint16_t handle)
{
    uint16_t  hCccd;
    bool_t isNotificationActive;
    bleResult_t result;

    /* Get handle of CCCD */
    result = GattDb_FindCccdHandleForCharValueHandle(handle, &hCccd);
    if (result != gBleSuccess_c)
    {
        return result;
    }

    result = Gap_CheckNotificationStatus(mSml_SubscribedClientId, hCccd,
                                         &isNotificationActive);
    if (result != gBleSuccess_c)
    {
        return result;
    }
    
    if (TRUE == isNotificationActive)
    {
        return GattServer_SendNotification(mSml_SubscribedClientId, handle);
    }
    else
    {
        return gBleSuccess_c;
    }
}

static bleResult_t Sml_SendSmartLockIndication(uint16_t handle)
{
    uint16_t    hCccd;
    bool_t      isIndicationActive;
    bleResult_t result;
    
    /* Get handle of CCCD */
    result = GattDb_FindCccdHandleForCharValueHandle (handle, &hCccd);
    if (result != gBleSuccess_c)
    {
        return result;
    }
    
    result = Gap_CheckIndicationStatus (mSml_SubscribedClientId,
                                        hCccd,
                                        &isIndicationActive);
    
    if (result != gBleSuccess_c)
    {
        return result;
    }
    
    if (TRUE == isIndicationActive)
    {
        return GattServer_SendIndication (mSml_SubscribedClientId,
                                          handle);
    }
    else
    {
        return gBleSuccess_c;
    }
}

/*! *********************************************************************************
 * @}
 ********************************************************************************** */
