#! /bin/bash
#
#**********************************************************************
# Copyright 2005-2014 by Freescale Semiconductor, Inc.
# All modifications are confidential and proprietary information
# of Freescale Semiconductor, Inc. ALL RIGHTS RESERVED.
#***********************************************************************
#============================================================
#                   make_project.sh
#
# Description: Create a project for KM user case.
# Author:   Terry Lu
# Data:     12-22-2014
# History:
#
#   12-22-2014      1.0     First release
#============================================================

#unalias mkdir
#unalias cp
#unalias grep
#unalias sed
#unalias pwd

readonly WORK_DIR="$(pwd)"
readonly REPOS_DIR="$WORK_DIR/.."
readonly PRJ_BUILD_DIR_ROOT="$REPOS_DIR/build"
readonly PRJ_SRC_DIR_ROOT="$REPOS_DIR/src"

PROJECT_NAME=""
DBG_DEVICE="PE-Micro"
Target_Device="MKMxxZ64"

readonly IAR_SUPPORT_TOOLCHAIN="
7.30
7.10
6.70
6.60
6.50
6.40"

readonly DBG_SUPPORT_INTERFACE="
J-Link
J-Trace
PE-Micro
CMSIS-DAP
Simulator"

readonly Device_Support="
MKMxxZ256:FlashKMxx256K:MKMxxZ256_flash.icf
MKMxxZ128:FlashKMxx128K:MKMxxZ128_flash.icf
MKMxxZ64:FlashKMxx64K:MKMxxZ64_flash.icf"

BUILD_DEVICE_DEFAULT_BOARD="FlashKMxx256K"
BUILD_DEVICE_FILE="project.board"
BUILD_PROJECT_FILES="
project.ewd
project.ewp
project.eww
"
SRC_PROJECT_TEMPLATE_FILE="project.c"
SRC_PROJECT_ADDTIONAL_FILE="
appconfig.h
freemaster_cfg.h
"

print_usage()
{
    echo "+----------------------------------------------------------+"
    echo "+ Usage:"
    echo "+ "
#    echo "+   1. ./$(basename $0) -n <prj_name> -t <iar_version> -d <dbg_interface> -f <device_selection>"
    echo "+   1. ./$(basename $0) -n <prj_name> -d <dbg_interface> -f <device_selection>"
    echo "+   -n Project Name"
#    echo "+   -t IAR Tool chain version, which can be 7.10, 6.70, 6.60, 6.50 or 6.40, default is 7.10."
    echo "+   -d Debug interface, which can be J-Link, J-Trace, PE-Micro, CMSIS-DAP or Simulator, default is PE-Micro."
    echo "+   -f Device selection, which can be MKMxxZ128 or MKMxxZ64, default is MKMxxZ64"
    echo "+   -h Help"
    echo "+   2. ./$(basename $0)"
    echo "+   Use interactive menu."
    echo "+"
    echo "+----------------------------------------------------------+"

    exit ${1}
}

check_parameter()
{
   local get_param=$1
   local support_list=$2

   for foo in $support_list; do
       if [ X"$foo" == X"$get_param" ]; then
           return 0
           break
       fi
   done

   return 1
}

create_setting_menu()
{
    local MENU_ARG=$1
    local NUM_OF_ITEM=$(echo $MENU_ARG | wc -l)
    local MENU_LIST=""
    local MENU_PREFIX=$2

    for each_item in $MENU_ARG; do
        MENU_LIST="$MENU_LIST ${MENU_PREFIX}${each_item}"
    done
    MENU_LIST="$MENU_LIST"

    OLD_IFS=$IFS
    IFS=$(echo -ne " ")
    PS3=">"
    echo "$3"
    select opt in $MENU_LIST; do
        if [ -z "$opt" ]; then
            echo "No selection! Exit!"
            USER_SEL_ITEM=""
        else
            if [ X"$opt" = "XExit"  ]; then
                USER_SEL_ITEM=""
            else
                USER_SEL_ITEM=$opt
            fi
        fi
        break
    done
    IFS=$OLD_IFS
}

interactive_menu()
{

    echo "+ Make project - 1.0 DEMO, 2014, Freescale Semiconductor, Inc."
    echo ""

    INTER_PREFIX=""
    INTER_MENU=$DBG_SUPPORT_INTERFACE
    INTER_PS="+ Select debugger support for IAR EWARM:"
    create_setting_menu "$INTER_MENU" "$INTER_PREFIX" "$INTER_PS"
    if [ -z "$USER_SEL_ITEM" ]; then
        echo "+ Wrong selection, Exit!"
        exit 0
    fi
    DBG_DEVICE=$USER_SEL_ITEM

    INTER_PREFIX=""
    INTER_MENU=""
    for each_item in $Device_Support; do
        INTER_MENU="$INTER_MENU $(echo $each_item | cut -d: -f1)"
    done
    INTER_PS="+ Select device"
    create_setting_menu "$INTER_MENU" "$INTER_PREFIX" "$INTER_PS"
    if [ -z "$USER_SEL_ITEM" ]; then
        echo "+ Wrong selection, Exit!"
        exit 0
    fi
    Target_Device=$USER_SEL_ITEM

    echo "+ Enter project name:"
    echo -n ">"
    read PROJECT_NAME
    if [ -z "$PROJECT_NAME" ]; then
        echo "+ No project name, Exit!"
        exit 0
    fi
}

if [ $# -eq 0 ]; then
    interactive_menu
else
    while [ -n "$1" ]; do
        case $1 in
            -n )
                PROJECT_NAME=$2
                shift
                shift
                ;;
            -d )
                DBG_DEVICE=$2
                if ! check_parameter $DBG_DEVICE "$DBG_SUPPORT_INTERFACE"; then
                    echo "+ Invalid Debug Interface $DBG_DEVICE!"
                    exit 1
                fi
                shift
                shift
                ;;
            -f )
                Target_Device=$2
                Device_List=""
                for each_item in $Device_Support; do
                    device_name=$(echo $each_item | cut -d: -f1)
                    Device_List="$Device_List $device_name"
                done
                if ! check_parameter $Target_Device "$Device_List"; then
                    echo "+ Invalid Device!"
                    exit 1
                fi
                shift
                shift
                ;;
            -h | -H )
                print_usage
                exit 0
                ;;
            * )
                echo "+ Invalid argument!"
                exit 1
                ;;
        esac

    done

fi

if [ -z $PROJECT_NAME ]; then
    echo "+ No project name specified!" 
    exit 1
fi

cd $WORK_DIR

# Check project dir and create new if not existed.

if [ ! -d $PRJ_BUILD_DIR_ROOT ]; then
    echo "+ build directory not found!"
    exit 1
fi

if [ ! -d $PRJ_SRC_DIR_ROOT ]; then
    echo "+ source directory not found!"
    exit 1
fi

if [ ! -d ${PRJ_BUILD_DIR_ROOT}/${PROJECT_NAME} ]; then
    mkdir -p ${PRJ_BUILD_DIR_ROOT}/${PROJECT_NAME}
fi

if [ ! -d ${PRJ_SRC_DIR_ROOT}/${PROJECT_NAME} ]; then
    mkdir -p ${PRJ_SRC_DIR_ROOT}/${PROJECT_NAME}
fi

# Copy template dir

PRJ_BUILD_DIR="${PRJ_BUILD_DIR_ROOT}/${PROJECT_NAME}"
PRJ_SRC_DIR="${PRJ_SRC_DIR_ROOT}/${PROJECT_NAME}"
TOOLCHAIN_TEMPLATE_DIR="${WORK_DIR}"

# Copy device file and change according to settings

cp -f ${TOOLCHAIN_TEMPLATE_DIR}/$BUILD_DEVICE_FILE $PRJ_BUILD_DIR/$BUILD_DEVICE_FILE

# Get flash file name
for each_item in $Device_Support; do
    device_name=$(echo $each_item | cut -d: -f1)
    if [ X"$device_name" == X"$Target_Device" ]; then
        board_file_name=$(echo $each_item | cut -d: -f2)
        linker_file_name=$(echo $each_item | cut -d: -f3)
        break
    fi
done

# Change board file name
if [ -z $board_file_name ]; then
    echo "+ No board file name!"
    exit 1
fi
# Seems no need to change
#sed -i "s/${BUILD_DEVICE_DEFAULT_BOARD}/${board_file_name}/" $PRJ_BUILD_DIR/$BUILD_DEVICE_FILE

# Copy project file and change according to settings
for each_item in $BUILD_PROJECT_FILES; do
    file_ext=$(echo $each_item | cut -d. -f2)
    prj_file_name=${PROJECT_NAME}.$file_ext
    prj_file_path=${PRJ_BUILD_DIR}/${PROJECT_NAME}.$file_ext
    cp -f ${TOOLCHAIN_TEMPLATE_DIR}/$each_item ${prj_file_path}

    case "$file_ext" in
        eww )
            sed -i "s/project.ewp/${PROJECT_NAME}.ewp/" $prj_file_path
            echo "+ Project build file $prj_file_name copied!"
            ;;
        ewp )
            sed -i "s/_template_/${PROJECT_NAME}/g" $prj_file_path
            sed -i "s/_linker_/${linker_file_name}/g" $prj_file_path
            echo "+ Project build file $prj_file_name copied!"
            ;;
        ewd )
            case "$DBG_DEVICE" in
                J-Link | J-Trace )
                    sed -i "s/_debuggertool_/JLINK_ID/" $prj_file_path
                    ;;
                PE-Micro )
                    sed -i "s/_debuggertool_/PEMICRO_ID/" $prj_file_path
                    ;;
                CMSIS_DAP )
                    sed -i "s/_debuggertool_/CMSISDAP_ID/" $prj_file_path
                    ;;
                Simulator )
                    sed -i "s/_debuggertool_/ARMSIM_ID/" $prj_file_path
                    ;;
                * )
                    echo "+ Unknown debugger interface!"
                    exit 1
                    ;;
            esac
            echo "+ Project build file $prj_file_name copied!"
            ;;
        * )
            echo "+ Unknown project file ext!"
            exit 1
            ;;
    esac
done

# Copy source file
# Copy addtional files
for each_item in $SRC_PROJECT_ADDTIONAL_FILE; do
    cp -f ${WORK_DIR}/$each_item ${PRJ_SRC_DIR}/$each_item
    echo "+ Project source file $each_item copied!"
done

# Copy project source file
cp -f ${WORK_DIR}/${SRC_PROJECT_TEMPLATE_FILE} ${PRJ_SRC_DIR}/${PROJECT_NAME}.c
echo "+ Project source file ${PROJECT_NAME}.c copied!"

echo ""
echo "+++++++++++++++++++++++++++++++++++++++++++++++++"
echo "+ Project $PROJECT_NAME created"
echo "+++++++++++++++++++++++++++++++++++++++++++++++++"
echo ""

