/******************************************************************************
 * (c) Copyright 2010-2015, Freescale Semiconductor Inc.
 * ALL RIGHTS RESERVED.
 ***************************************************************************/
#include "common.h"
#include "metering_modules.h"
#include "drivers.h"
#include <string.h>
#include <stdio.h>

/*
    This file implements work mode functions
*/

static int32_t enter_vlps_flag = 0;
static int32_t enter_vlpr_flag = 0;
static int32_t exit_vlpr_flag = 0;

/* Some interrupt may be enabled when system launched, clear it */
static int32_t work_mode_clear_init_interrupt(void)
{
    /* GPIO */
    if (SIM_SCGC5 & SIM_SCGC5_PORTA_MASK)
        PORTA_ISFR = (uint8_t)PORTA_ISFR; /* read and clear interrupt flags   */

    if (SIM_SCGC5 & SIM_SCGC5_PORTB_MASK)
        PORTB_ISFR = (uint8_t)PORTB_ISFR; /* read and clear interrupt flags   */

    if (SIM_SCGC5 & SIM_SCGC5_PORTC_MASK)
        PORTC_ISFR = (uint8_t)PORTC_ISFR; /* read and clear interrupt flags   */

    if (SIM_SCGC5 & SIM_SCGC5_PORTD_MASK)
        PORTD_ISFR = (uint8_t)PORTD_ISFR; /* read and clear interrupt flags   */

    if (SIM_SCGC5 & SIM_SCGC5_PORTE_MASK)
        PORTE_ISFR = (uint8_t)PORTE_ISFR; /* read and clear interrupt flags   */

    if (SIM_SCGC5 & SIM_SCGC5_PORTF_MASK)
        PORTF_ISFR = (uint8_t)PORTF_ISFR; /* read and clear interrupt flags   */

    if (SIM_SCGC5 & SIM_SCGC5_PORTG_MASK)
        PORTG_ISFR = (uint8_t)PORTG_ISFR; /* read and clear interrupt flags   */

    if (SIM_SCGC5 & SIM_SCGC5_PORTH_MASK)
        PORTH_ISFR = (uint8_t)PORTH_ISFR; /* read and clear interrupt flags   */

    if (SIM_SCGC5 & SIM_SCGC5_PORTI_MASK)
        PORTI_ISFR = (uint8_t)PORTI_ISFR; /* read and clear interrupt flags   */

    return 0;
}

int32_t work_mode_vlpr_mode_module_init(void)
{
    pulse_out_deinit();
#if POWER_METERING_SMART_CARD_SUPPORT
    smart_card_deinit();
#endif
    sf_access_deinit();
    rs485_parse_deinit();
    freq_detect_deinit();
    vref_comp_deinit();
    rtc_comp_deinit();
    slcd_disp_deinit();
#if POWER_METERING_IRDA_SUPPORT
    irda_deinit();
#endif
    return 0;
}

int32_t work_mode_run_mode_module_init(void)
{
    /* XBAR common init                                                         */
    XBAR_Init (XBAR_MODULE_NO_EDGE_DETECT_CONFIG, PRI_LVL0, NULL);

    /* Init rtc functions, 1Hz compensation, tamper etc. */
    pulse_out_init();

    rtc_comp_init();

#if POWER_METERING_SLCD_SUPPORT
    slcd_disp_init();
#endif

    key_handler_init();

    LED_Init();
#if POWER_METERING_SMART_CARD_SUPPORT
    smart_card_init();
#endif
#if POWER_METERING_SF_SUPPORT
    sf_access_init();
#endif
#if POWER_METERING_RS485_SUPPORT
    rs485_parse_init();
    uart_handler_init();
#endif

    pit_trigger_init();
#if POWER_METERING_IRDA_SUPPORT
    //irda_init(IRDA_XFER_NON_BLOCKING, IRDA_XFER_BLOCKING);
#endif
    work_mode_clear_init_interrupt();

    return 0;
}

int32_t work_mode_run_mode_freq_init(void)
{
#if (72000000 == POWER_METERING_SYS_CLK_FREQ)
    /* clock mode 3:1:1, sysclock is 72MHz, so bus clock and flash clock is 24MHz */
    /* Bus clock up to 25Mhz */
    /* Flash colck up to 25MHz */
    SIM_SetClkMode(SYSCLK_MODE2);
    SIM_SetClkDiv(SYSCLK_DIV1);
#if POWER_METERING_SYS_CLK_FEE_MODE
    FLL_Init(FLL_MODULE_FEE_72MHZ_CONFIG);
#elif POWER_METERING_SYS_CLK_FEI_MODE
    FLL_Init(FLL_MODULE_FEI_72MHZ_CONFIG);
#else
    #error "One clock mode config need to be used!"
#endif
#elif (48000000 == POWER_METERING_SYS_CLK_FREQ)
    /* Default is 48MHz */
    /* clock mode 2:1:1, 48MHz                                                 */
    SIM_SetClkMode(SYSCLK_MODE1); 
    SIM_SetClkDiv(SYSCLK_DIV1);
#if POWER_METERING_SYS_CLK_FEE_MODE
    FLL_Init(FLL_MODULE_FEE_48MHZ_CONFIG);
#elif POWER_METERING_SYS_CLK_FEI_MODE
  FLL_Init(FLL_MODULE_FEI_48MHZ_CONFIG);
#else
#error "One clock mode config need to be used!"
#endif
#elif (24000000 == POWER_METERING_SYS_CLK_FREQ)
    /* clock mode 1:1:1, 24MHz                                                 */
    SIM_SetClkMode(SYSCLK_MODE0); 
    SIM_SetClkDiv(SYSCLK_DIV1);
#if POWER_METERING_SYS_CLK_FEE_MODE
    FLL_Init(FLL_MODULE_FEE_24MHZ_CONFIG);
#elif POWER_METERING_SYS_CLK_FEI_MODE
    FLL_Init(FLL_MODULE_FEI_24MHZ_CONFIG);
#else
    #error "One clock mode config need to be used!"
#endif
#else
    #error "Wrong system clock frequency!"
#endif

    return 0;
}

/* This function will disable interrupt, so should be called in main */
int32_t work_mode_enter_vlpr(void)
{
    DisableInterrupts();

    work_mode_vlpr_mode_module_init();

    AFE_Disable();

    SIM_Init(SIM_MODULE_POWER_METERING_VLPR_MODE_CONFIG);

#ifdef POWER_METERING_DEBUG
    /* Output RTC clock to XBAR_OUT4 pin */
    PORT_Init(PORTE,PORT_MODULE_ALT5_MODE, PIN0);
    SIM_SelClkout(CLKOUT_SRC7);
#endif

    /* clock mode 2:1:1, sysclock is 4MHz, so bus clock and flash clock is 1MHz */
    MCG_FEE_To_FBI(IRC_FAST_FREQ, IRC_FAST);
    MCG_FBI_To_BLPI(IRC_FAST_FREQ, IRC_FAST);
    PMC_SetBGEN();

    /* switch to VLPR mode                                                      */
    SMC_SetMode(VLPR);

    metering_func_vlpr_hardware_init();

    EnableInterrupts();

    /* Delay about 1000 / 20 = 50ms */
    arch_delay(IRC_FAST_FREQ / 50);

    return 0;
}

int32_t work_mode_exit_vlpr(void)
{
    DisableInterrupts();
    AFE_Disable();

    work_mode_run_mode_freq_init();
    MCG_BLPI_To_FBI(IRC_FAST_FREQ, IRC_FAST);
    MCG_FBI_To_FEE(ERC_SLOW_FREQ, 0, 0);

    //SMC_Init(SMC_MODULE_VLPS_OFF_VLLS_OFF_CONFIG);
    SIM_Init(SIM_MODULE_POWER_METERING_RUN_MODE_CONFIG);
    SMC_SetMode(RUN);

    work_mode_run_mode_module_init();

    metering_func_hardware_init();
    EnableInterrupts();

    return 0;
}

int32_t work_mode_enter_vlps(void)
{
    //SIM_CtrlPLLVLP(FALSE);
    //SMC_SetMode(VLPS);

    return 0;
}

int32_t work_mode_set_enter_vlpr_flag(int32_t flag)
{
    enter_vlpr_flag = flag;

    return 0;
}

int32_t work_mode_get_enter_vlpr_flag(void)
{
    return enter_vlpr_flag;
}

int32_t work_mode_set_enter_vlps_flag(int32_t flag)
{
    AFE_Disable();

    enter_vlps_flag = flag;

    return 0;
}

int32_t work_mode_get_enter_vlps_flag(void)
{
    return enter_vlps_flag;
}

int32_t work_mode_set_exit_vlpr_flag(int32_t flag)
{
    AFE_Disable();

    exit_vlpr_flag = flag;

    return 0;
}

int32_t work_mode_get_exit_vlpr_flag(void)
{
    return exit_vlpr_flag;
}

/******************************************************************************
 * End of module                                                              *
 ******************************************************************************/
