/*
 * Copyright (c) 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __SCR_TTL_H__
#define __SCR_TTL_H__

#include "common.h"
#include "metering_modules.h"
#include "drivers.h"

/*******************************************************************************
 * Defination
 ******************************************************************************/

//ATR Parameter Definitions
#define ATR_DURATION       (20160 - 10)
#define SETUP_LENGTH       15
#define HIST_LENGTH        15
#define ISO7816_CARD_SUPPORT

//APDU Length Definitions
#define LC_MAX             32
#define LE_MAX             32

//ATR information print or not MACRO, define it to 1 show details of ATR
#define ATR_PARSE_PRINT  1

//PPS print
#define PPS_INFO_PRINT   1

//APDU print information or not MACRO, define 1 to print information via UART
#define GET_RESPONSE_PRINT 1
#define SEND_CLA_PRINT     1
#define RECV_STATUS_PRINT  1

//Transfer Parity Check Parameter
enum
{
  kEven,
  kOdd
};

//Card type parameter
enum
{
  kTType0,
  kTType1
};


/* C-ADPU-Header command structure ---------------------------------------------*/
typedef struct _sc_adpu_header_t
{
  uint8_t CLA;  /* Command class */
  uint8_t INS;  /* Operation code */
  uint8_t P1;   /* Selection Mode */
  uint8_t P2;   /* Selection Option */
  uint8_t P3;   /* Selection Option, LC or LE */
} sc_adpu_header_t;

/* C-ADPU-Body command structure -----------------------------------------------*/
typedef struct _sc_adpu_body_t
{
  uint8_t Data[LC_MAX];  /* Command parameters */
  uint16_t LC;
  uint16_t LE;
} sc_adpu_body_t;

/* C-ADPU Command structure ----------------------------------------------------*/
typedef struct _sc_adpu_command_t
{
  sc_adpu_header_t adpuHeader;
  sc_adpu_body_t adpuBody;
} sc_adpu_command_t;

/* SC response structure -----------------------------------------------------*/
typedef struct _sc_adpu_response_t
{
  uint8_t Data[LE_MAX];  /* Data returned from the card */
  uint8_t SW1;          /* Command Processing status */
  uint8_t SW2;          /* Command Processing qualification */
  uint8_t dataRequired; /* If SC need more data*/
  uint8_t cmdResend;    /* If need to resend command to smart card*/
  uint8_t cmdGetResp;   /* next command is get response*/
} sc_adpu_response_t;

/* SC Paramter */
typedef struct
{
  uint32_t sCClock;     /*!< Smart card Clock ranges from 1 to 5 MHz */
  uint32_t sCClockKHz;  /*!< Smart card Clock set in KHz */
  uint8_t TA1;          /*!< Smart Card ATR TA1>*/
  uint8_t TA2;          /*!< Smart Card ATR TA2>*/
  uint8_t TC1;          /*!< Smart Card ATR TC1>*/    
  uint8_t WI;           /*!< 8 bits, iso range = 1 to 255 */
  uint16_t Fi;          /*!< 4 bits Fi */
  uint8_t fMax;         /*!< Maximum Smart card frequency in MHz */
  uint8_t currentD;     /*!< 4 bits Di */
  uint8_t Di;           /*!< 4 bits Di */
  uint8_t BWI;          /*!< 4 bits BWI */
  uint8_t CWI;          /*!< 4 bits CWI */
  uint8_t BGI;          /*!< 4 bits BWI */
  uint8_t GTN;          /*!< 8 bits GTN */
  uint8_t t0Indicated;  /*!< If T=0 indicated in TD1 byte */
  uint8_t t1Indicated;  /*!< If T=1 indicated in TD2 byte */
  uint8_t sCPort;       /*!< Smart card port */
  uint32_t baudrate;    /*!< baudrate value */
  uint8_t ttype;        /*!< Transfer type */
  UART_MemMapPtr uartMemMapPtr; /*!< UART module register block pointer */
  uint8_t adtExpired;   /*!< Indicates whether ADT timer expired */
  uint8_t wwtExpired;   /*!< Indicates whether WWT timer expired */
  uint8_t cwtExpired;   /*!< Indicates whether CWT timer expired */
  uint8_t bwtExpired;   /*!< Indicates whether BWT timer expired */
  uint8_t ppsDone;      /*!< Indicates whether the PPS was successfully executed or not */
  uint8_t resetType;    /*!< Indicates whether a Cold reset or Warm reset was requested. */
  uint8_t vcc;          /*!< Indicates the voltage class */  
  uint8_t IFSC;         /*!< Indicates IFSC value of the card */ 
  uint8_t IFSD;         /*!< Indicates IFSD value of the terminal */ 
  uint8_t parity;       /*!< Indicates current parity even/odd */ 
  uint8_t modeNegotiable; /*!< Indicates if the card acts in negotiable or specific mode. */
  uint8_t invalidInitChar;/*!< Indicates whether Invalid Init Character has been detected*/
  uint8_t rxtCrossed;     /*!< Indicates whether RXT threshols has been crossed*/
  uint8_t txtCrossed;     /*!< Indicates whether TXT threshols has been crossed*/
  uint8_t initDelayExpired;/*!<Indiactes whether Initiation time delay after RST has expired before init character*/
  uint8_t initDectected;   /*!<Indicates initial char dectected>*/
  uint32_t error_isr_count;
} smartcardreader_params_t;

/* TTL command structure */
typedef struct
{
    uint8_t cardType;
    uint8_t NAD;
    uint8_t PCB;
    uint8_t LEN;
    sc_adpu_command_t * sc_adpu_command;         /*!< command > */
    uint8_t LRC;
    uint16_t length;      /*!< command length */
}ttl_cmd_t;

/* TTL response structure */
typedef struct
{
    uint8_t NAD;
    uint8_t PCB;
    uint8_t LEN;
    sc_adpu_response_t * sc_adpu_response;        /*!< response pointer */
    uint8_t LRC;
    uint16_t length;      /*!< response length */
}ttl_resp_t;


/* Exported macro ------------------------------------------------------------*/
extern uint8_t gTxBuf[32];
extern uint8_t gRxBuf[32];
extern sc_adpu_command_t sc_adpu_command;
extern smartcardreader_params_t sCReaderParams;
extern sc_adpu_command_t sc_adpu_command;
extern sc_adpu_response_t sc_adpu_response;

/* Exported functions ------------------------------------------------------- */
/* TTL APPLICATION LAYER ---------------------------------------------------------*/

/*!
 * @name Terminal Transport Layer Application Functions
 * @{
 */

/*!
 * @brief Send comamnd with command data package
 *
 * @param sc_adpu_command_t  Terminal Application layer command pointer
 * @param sc_adpu_response_t Terminal Application layer response pointer
 * @return Error or success status returned by API.
 */
int32_t sc_send_command_with_data(sc_adpu_command_t *sc_adpu_command, sc_adpu_response_t *sc_adpu_response);

/*!
 * @brief Send comamnd with command without data package
 *
 * @param sc_adpu_command_t  Terminal Application layer command pointer
 * @param sc_adpu_response_t Terminal Application layer response pointer
 * @return Error or success status returned by API.
 */
int32_t sc_send_command_without_data(sc_adpu_command_t *sc_adpu_command, sc_adpu_response_t *sc_adpu_response);

/*!
 * @brief Get status comamnd to receive data package from smart card
 *
 * @param sc_adpu_command_t  Terminal Application layer command pointer
 * @param sc_adpu_response_t Terminal Application layer response pointer
 * @return Error or success status returned by API.
 */
int32_t sc_get_response(sc_adpu_command_t *sc_adpu_command, sc_adpu_response_t *sc_adpu_response);

/*!
 * @brief Send T1 frame package and receive data package from smart card
 *
 * @param ttl_cmd  Terminal Terminal layer command pointer
 * @param ttl_resp Terminal Terminal layer response pointer
 * @return Error or success status returned by API.
 */
int32_t sc_t1_send_command(ttl_cmd_t *ttl_cmd, ttl_resp_t *ttl_resp);

/*!
 * @brief Send T1 frame package and receive data package from smart card
 *
 * @param info    Information pointor
 * @param buf     Data pointor
 * @param length  Data length to print
 * @return NA
 */
void sc_print(char* info,uint8_t *buf, uint8_t length);

/*!
 * @brief Report R-ADPU received
 *
 * @param smartcardreader_params    Smart card information pointor
 * @param sc_adpu_response          adpu response pointer
 * @param buf                       receive data buf pointer
 * @param length                    receive data buf length
 * @return NA
 */
void sc_response_status_check(smartcardreader_params_t *sCReaderParamsPtr, sc_adpu_response_t *sc_adpu_response, uint8_t * buf, uint8_t length);



#endif /* __FSL_SCR_TTL_H__ */

/*******************************************************************************
 * EOF
 ******************************************************************************/
