/*
 * Copyright (c) 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "common.h"
#include "metering_modules.h"
#include "drivers.h"
#include "scr_ttl.h"
#include "scr_interface.h"
#include <string.h>


/*******************************************************************************
 * Defines
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
* Code
******************************************************************************/

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_init
 * Description   : Smart card reader initialization function.
 * This function will initialization smart card reader interface
 *
 *END**************************************************************************/
int8_t scr_init(smartcardreader_params_t *sCReaderParamsPtr)
{
    scr_interface_init();
    sCReaderParamsPtr->sCClockKHz = scr_clock_init() / 1000;
    scr_emv_init(sCReaderParamsPtr);

    return 0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_receive_atr_connect
 * Description   : Receive ATR and decode it
 * This function receives an ATR from the card and performs various prescribed 
 * checks and takes prescribed actions 
 *
 *END**************************************************************************/
int16_t scr_receive_atr_connect(smartcardreader_params_t *sCReaderParamsPtr, uint8_t *buff)
{
    int16_t retLength ,temp;        
    uint16_t length = 260;   /* 22 for 6.5 RevC, 23 for 7.5 RevD */
    int32_t timeout = 100000;

    /*STart proceedings for cold ATR*/ 
    scr_cold_reset(sCReaderParamsPtr);

    scr_setup_atr(sCReaderParamsPtr);
    scr_enable_init_detect(sCReaderParamsPtr);

    sCReaderParamsPtr->adtExpired = 0;
    scr_reset_wwt_timer(sCReaderParamsPtr);
    scr_enable_adt(sCReaderParamsPtr);
    scr_disable_wwt(sCReaderParamsPtr);

    /*Wait for Init char.In case wrong Init char comes, reject the card
    In case, Init char comes after 42000 clock cycles of RST , reject the card*/
    while (timeout--)
    {
        if((sCReaderParamsPtr->initDectected == 1)
          || (sCReaderParamsPtr->adtExpired == 1))
            break;
        NOP();
    }

     if (timeout <= 0)
         return -1;

    if (sCReaderParamsPtr->adtExpired)
    {
        sCReaderParamsPtr->adtExpired = 0;
        sc_debug("Expired TS time found\r\n");
        /*reject the card*/
        goto deactivation;
    }
    else
    {
        /* Start WWT,CWT and ADT timer */
        sCReaderParamsPtr->wwtExpired = 0;
        sCReaderParamsPtr->cwtExpired = 0;
        /* Enable WWT and ADT timer before starting T=0 transport */
        scr_enable_wwt(sCReaderParamsPtr); 
    
        /*Receiving cold ATR*/    
        retLength = scr_receive_chars_atr(sCReaderParamsPtr, buff, length);
        scr_disable_adt(sCReaderParamsPtr);
        scr_disable_wwt(sCReaderParamsPtr);
    
        /*In case there is a parity error in ATR,reject the card*/
        if(sCReaderParamsPtr->rxtCrossed == 0x01)
        {
            sCReaderParamsPtr->rxtCrossed = 0x00;
            goto deactivation;
        }
        else
        {
            temp = scr_parse_atr(sCReaderParamsPtr, buff, retLength);
            /*Parsing and checking cold ATR*/
            if (temp == -1)
            {
                if (sCReaderParamsPtr->wwtExpired)
                { 
                    /*Card has to be rejected as WT expired while full ATR was not received*/
                    sCReaderParamsPtr->wwtExpired = 0;
                    goto deactivation;
                }
                else if (sCReaderParamsPtr->adtExpired)
                { 
                    /*Card has to be rejected as ADT expired while full ATR was not received*/
                    sCReaderParamsPtr->adtExpired = 0;
                    goto deactivation;
                } 
                else
                {
                    /*ATR has to be rejected only and start warm reset sequence*/
                    goto warm_reset;
                }
            }
            else
            {
                /*ATR has been succefully received; return the length of ATR received*/
                return retLength;
            }

warm_reset:
            /*Start proceedings for warm ATR*/  
            scr_warm_reset(sCReaderParamsPtr);

            sCReaderParamsPtr->adtExpired = 0;
            scr_enable_adt(sCReaderParamsPtr);
            scr_disable_wwt(sCReaderParamsPtr);
        
            /*Wait for Init char.In case wrong Init char comes, reject the card
            In case, Init char comes after 42000 clock cycles of RST , reject the card*/
            while (timeout--)
            {
                if((sCReaderParamsPtr->initDectected == 1)
                    || (sCReaderParamsPtr->adtExpired == 1))
                    break;
                NOP();
            }

            if (timeout <= 0)
                return -1;
        
            if (sCReaderParamsPtr->invalidInitChar)
            {
                sCReaderParamsPtr->invalidInitChar = 0;
                /*reject the card*/
                goto deactivation;
            }
            else if (sCReaderParamsPtr->initDelayExpired)
            {
                sCReaderParamsPtr->initDelayExpired = 0;
                /*reject the card*/
                goto deactivation;
            }
            else
            {
                /* Start WWT,CWT and ADT  timer */
                sCReaderParamsPtr->wwtExpired = 0;
                sCReaderParamsPtr->cwtExpired = 0;
        
                scr_reset_wwt_timer(sCReaderParamsPtr);
        
                /* Enable WWT and ADT timer before starting T=0 transport */
                scr_enable_wwt(sCReaderParamsPtr); 
        
                /*Receiving warm ATR*/    
                retLength = scr_receive_chars_atr(sCReaderParamsPtr, buff, length);
        
                scr_disable_adt(sCReaderParamsPtr);
                scr_disable_wwt(sCReaderParamsPtr); 
        
                /*In case there is a parity error in ATR,reject the card*/
                if (sCReaderParamsPtr->rxtCrossed == 0x01)
                {
                    sCReaderParamsPtr->rxtCrossed = 0x00;
                    goto deactivation;
                }
                else
                {
                    temp = scr_parse_atr(sCReaderParamsPtr,buff,retLength);
                    /*Parsing and checking warm ATR*/
                    if (temp == -1)
                    {
                        /*ATR has to be rejected ; hence begin deactivation sequence */
                        goto deactivation;
                    }
                    else
                    {
                        /*WARM ATR has been succefully received; return the length of ATR received*/
                        return retLength;
                    }
                }
            }
        }
    }

deactivation :
    /*Start deactivation sequence;send the command to HAL*/
    scr_reject_card(sCReaderParamsPtr);
  
    return -1;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_parse_atr
 * Description   : Parse ATR
 * This function parses the received ATR from the card and re-populates the
 * sCReaderParamsPtr with the card parameters
 *
 *END**************************************************************************/
int16_t scr_parse_atr(smartcardreader_params_t *sCReaderParamsPtr, uint8_t *buff, int16_t length)
{
    uint16_t Fi;        
    uint8_t ch;
    uint16_t i;
    uint8_t TCK = 0;
    uint8_t Yi = 0;
    uint8_t Y1;
    uint8_t Y2;
    uint8_t typeTi = 0;
    uint8_t historicalBytes = 0;
    uint8_t searchTA3 = 1;
    uint8_t searchTB3 = 1;
    uint8_t searchTC3 = 1;
    sCReaderParamsPtr->modeNegotiable = 1; 
  
    for (i=0; i < length; i++)
    {
        TCK ^= buff[i];
    }
#if  (ATR_PARSE_PRINT == 1)
    sc_debug("ATR CRC check result TCK = 0x%x\r\n", TCK);
    if (TCK != 0)
    {
        sc_debug("TCK is wrong or absent in ATR\r\n");
    }
    else
    {
        sc_debug("TCK is correct\r\n");
    }
#endif
  
    if (length == 0)
        goto out;
  
    /* parsing data */
    i = 0;
  
    /*STORING y1 TO CHECK WHICH COMPONENTS ARE AVAILABLE*/
    ch = buff[i]; 
    Y1 = ch >> 4;
    historicalBytes = ch & 0x0f;
    length -= historicalBytes; 
    i++;
  
#if (ATR_PARSE_PRINT == 1)
    sc_debug("T0 = 0x%x , Y1 = 0x%X, K = %d\r\n",ch,Y1,historicalBytes);
#endif  
  
    if(Y1 & 0x1)
    { 
        /*TA1 present*/
        if(i >= length)
        /*TA1 not found,hence generate error*/
            goto out;
        ch = buff[i]; 
        Fi = ch >> 4; 
        sCReaderParamsPtr->TA1 = ch; 
        /*switch to the real Fi value, according to the table of Spec-3, chapter 8.3 */
        switch (Fi)
        {
        /*Fi = 1 is only allowed value*/
#if defined(POWER_METERING_SMART_CARD_ISO7816_CARD) 
        case 0 :
            sCReaderParamsPtr->Fi =372;
            sCReaderParamsPtr->fMax = 4;
            break;
#endif
        case 1 :
            sCReaderParamsPtr->Fi = 372;
            sCReaderParamsPtr->fMax = 5;
            break;
#if defined(POWER_METERING_SMART_CARD_ISO7816_CARD)
        case 2:
            sCReaderParamsPtr->Fi = 558;
            sCReaderParamsPtr->fMax = 6;
            break;
        case 3:
            sCReaderParamsPtr->Fi = 744;
            sCReaderParamsPtr->fMax = 8;
            break;
        case 4:
            sCReaderParamsPtr->Fi = 1116;
            sCReaderParamsPtr->fMax = 12;
            break;
        case 5:
            sCReaderParamsPtr->Fi = 1488;
            sCReaderParamsPtr->fMax =16;
            break;
        case 6:
            sCReaderParamsPtr->Fi = 1860;
            sCReaderParamsPtr->fMax = 20;
            break;
        case 9:
            sCReaderParamsPtr->Fi = 512;
            sCReaderParamsPtr->fMax = 5;
            break;
        case 10:
            sCReaderParamsPtr->Fi = 768;
            sCReaderParamsPtr->fMax = 7;
            break;
        case 11:
            sCReaderParamsPtr->Fi = 1024;
            sCReaderParamsPtr->fMax = 10;
            break;
        case 12:
            sCReaderParamsPtr->Fi = 1536;
            sCReaderParamsPtr->fMax = 15;
            break;
        case 13:
            sCReaderParamsPtr->Fi = 2048;
            sCReaderParamsPtr->fMax = 20;
#endif      
        default:
            sCReaderParamsPtr->Fi =372;
            sCReaderParamsPtr->fMax = 5;
            goto out;
            break;
        }
    
        sCReaderParamsPtr->Di = ch&0xf;
        switch(sCReaderParamsPtr->Di)
        {
        /*Di = 4 is only allowed value*/
        case 1:
            sCReaderParamsPtr->Di = 1;
            break;
        case 2:
            sCReaderParamsPtr->Di = 2;
            break;
        case 3:
            sCReaderParamsPtr->Di = 4;
            break;
#if defined(POWER_METERING_SMART_CARD_ISO7816_CARD)
        case 4:
            sCReaderParamsPtr->Di = 8;
            break;
        case 5:
            sCReaderParamsPtr->Di = 16;
            break;
        case 6:
            sCReaderParamsPtr->Di = 32;
            break;
        case 7:
            sCReaderParamsPtr->Di = 64;
            break;
        case 8:
            sCReaderParamsPtr->Di = 12;
            break;
        case 9:
            sCReaderParamsPtr->Di = 20;
            break;
#endif
        default:
            sCReaderParamsPtr->Di = 1;
            goto out;
            break;
        }    
        i++;
    }
#if (ATR_PARSE_PRINT == 1)
    sc_debug("TA1 = 0x%x, Fi = %d, Di = %d, fMax = %dMHz\r\n",
           ch, sCReaderParamsPtr->Fi, sCReaderParamsPtr->Di, sCReaderParamsPtr->fMax);
#endif  
    if (Y1 & 0x2)
    {
        /*TB1 present*/
        if(i>=length)
            /*TB1 not found,hence generate error*/
            goto out;
        ch = buff[i]; 
        if ((sCReaderParamsPtr->resetType == kColdReset) && (ch > 0x00))
            goto out;
        i++;
    }
#if (ATR_PARSE_PRINT == 1)
    sc_debug("TB1 = 0x%x\r\n",ch);
#endif    
    if (Y1 & 0x4)
    {
        /*TC1 present*/
        if(i>=length)
        {
            /*TC1 not found,hence generate error*/
            goto out;
        }
        ch = buff[i];
        sCReaderParamsPtr->TC1 = ch;
        if(ch == 0xff)
            sCReaderParamsPtr->GTN = 0xFF;
        else
            sCReaderParamsPtr->GTN = ch;
    
        i++;
    }
    else
        sCReaderParamsPtr->GTN = 0xFF;
#if (ATR_PARSE_PRINT == 1)
    sc_debug("Guard Time TC1 = 0x%x\r\n",ch);
#endif  
    if (Y1 & 0x8)
    {
        /*TD1 present*/
        if (i >= length)
        {
            /*TD1 not found,hence generate error*/
            goto out;
        }
        ch = buff[i];
        if ((buff[i] & 0xf) == 0)
        {
            /*T=0 is first offered protocol and hence must be used until TD2 is present
            indicating presence of T=1 and TA2 is absent indicating negotiable mode */
            sCReaderParamsPtr->t0Indicated = 1;
            sCReaderParamsPtr->t1Indicated = 0;
        }
        else if((buff[i] & 0xf) == 1)
        {
            /*T=1 is first offered protocol and hence must be used 
            TA2 is absent indicating negotiable mode */
            sCReaderParamsPtr->t1Indicated = 1;
            sCReaderParamsPtr->t0Indicated = 0;
        }
        else
        {
            goto out;
        }
        Y2 = ch >> 4;
        i++;
#if (ATR_PARSE_PRINT == 1)
        sc_debug("TD1 = 0x%x, Y2 = 0x%X\r\n",ch,Y2);
#endif
    }
    else 
    {
        /*TD1 absent,meaning only T= 0 has to be used*/
        sCReaderParamsPtr->t0Indicated = 1; 
        sCReaderParamsPtr->t1Indicated = 0;
        Y2 = 0;
    }
  
    if (Y2 & 0x1)
    {
        /*TA2 present*/
        if (i >= length)
        {
            /*TA2 not found,hence generate error*/    
            goto out; 
        }
        sCReaderParamsPtr->modeNegotiable = 0;
        ch = buff[i];
        sCReaderParamsPtr->TA2 = ch;
#if (ATR_PARSE_PRINT == 1)
        sc_debug("TA2 = 0x%x\r\n",ch);
#endif
        if ((ch & 0x0f) == 0)
        {
            /*T = 0 specfic mode*/
            sCReaderParamsPtr->t0Indicated = 1; 
            sCReaderParamsPtr->t1Indicated = 0;
        }
        else if ((ch & 0x0f) == 1)
        {
            /*T = 1 specfic mode*/
            if (sCReaderParamsPtr->t1Indicated ==0)
                goto out;
            else
                 sCReaderParamsPtr->t0Indicated = 0;
        }
        else
        {
            /*Unsupported Protocol type*/
            goto out;
        }
    
        if (ch & 0x10)
        {
            /*b5 should be zero;hence generate error*/
            goto out;
        }
    
        if (ch & 0x80)
        {
            /*Card is still in negotiable mode*/
            sCReaderParamsPtr->modeNegotiable = 1;
        }
    
        i++;    
        }
    else
    {
        /*TA2 not present*/
        sCReaderParamsPtr->modeNegotiable = 1; 
    }
  
    if (Y2 & 0x2)
    {
        /*TB2 present*/
        if (i >= length)
        {
            /*TB2 not found,hence generate error*/ 
            goto out;
        }      
        i++;
        /*TB2 is not supported by EMV ;hence generate error*/
        goto out;
    }
  
    if (Y2 & 0x4)
    {
        /*TC2 present*/
        if(i >= length)
        {
            /*TC2 not found,hence generate error*/ 
            goto out; 
        }
    
        /*ch contains temp value of WI sent by ICC*/
        ch = buff[i];
#if (ATR_PARSE_PRINT == 1)
        sc_debug("TC2 = 0x%x, WI = 0x%x\r\n",ch, ch);
#endif

        if (ch == 0)
        {
            /*WI can't be zero*/
            goto out;
        }
        else
        {
            sCReaderParamsPtr->WI = ch;
        }
        i++;
    }
    else
        /*TC2 absent;hence WI = 0x0A ; default value*/
        sCReaderParamsPtr->WI = 0x0A;
  
  
    if (Y2 & 0x8)
    {
        /*TD2 present*/
        if (i >= length)
        {
            /*TD2 not found,hence generate error*/ 
            goto out;
        }
    
        if ((buff[i] & 0xf) == 0)
        {
            /*lsb can't be zero*/
            goto out;
        }
        else if ((buff[i] & 0xf) == 1)
        {
            if(sCReaderParamsPtr->modeNegotiable == 1)
              sCReaderParamsPtr->t1Indicated = 1;
         }
        else if ((buff[i] & 0xf) == 0x0E)
        {
            if (sCReaderParamsPtr->t1Indicated == 0)
            {
               /*Do nothing*/
            }
            else
            {
                goto out;
            }
        }
        else
        {
            /*Unsupported values*/
            goto out;
        }
    
        Yi = buff[i] >> 4;                
#if (ATR_PARSE_PRINT == 1)
        sc_debug("TD2 = 0x%x, Yi = %d\r\n",buff[i],Yi);
#endif
        i++;
    }
    else
    {
        Yi = 0;
    }
  
    while (Yi)
    {
        if (Yi & 0x1)
        {
            /*TAi present*/
            if (i >= length)
            {
                /*TAi not found;hence generate error*/
                goto out;
            }
      
            if (searchTA3)
            {
                if ((buff[i] >= 0x10) && (buff[i] <= 0xFE))
                {
                    sCReaderParamsPtr->IFSC = buff[i];
#if (ATR_PARSE_PRINT == 1)
                    sc_debug("Card Max block receive size, IFSC TA3 = 0x%x\r\n",buff[i]);
#endif
                }
                else
                {
                    /*Unsupported value of TA3*/
                    goto out;
                }
                searchTA3 = 0;
            }      
            i++;
        }
        else
        {
            /*TAi absent*/
            if(searchTA3)
            {
                /*Default value of TA3*/
                sCReaderParamsPtr->IFSC = 0x20;
                searchTA3 = 0;
            } 
        }
    
        if (Yi & 0x2)
        {
            /*TBi present*/
            if(i >= length)
            {
                /*TBi not found;hence generate error*/
                goto out;
            }
      
            if(searchTB3)
            {
#if !defined(POWER_METERING_SMART_CARD_ISO7816_CARD)
                if(((buff[i] >> 4) <= 4) & ((buff[i] & 0x0F) <= 5))
                {
                    sCReaderParamsPtr->BWI = buff[i] >> 4;
                    sCReaderParamsPtr->CWI = buff[i] & 0x0F;  
#if (ATR_PARSE_PRINT == 1)
                    sc_debug("TB3 = 0x%x, BWI = 0x%x, CWI = 0x%x\r\n",buff[i],sCReaderParamsPtr->BWI, sCReaderParamsPtr->CWI);
#endif
                }
                else
                {
                    goto out;
                }
#else
                sCReaderParamsPtr->BWI = buff[i] >> 4;
                sCReaderParamsPtr->CWI = buff[i] & 0x0F; 
#if (ATR_PARSE_PRINT == 1)
                sc_debug("TB3 = 0x%x, BWI = 0x%x, CWI = 0x%x\r\n",buff[i],sCReaderParamsPtr->BWI, sCReaderParamsPtr->CWI);
#endif
#endif
                searchTB3 = 0;
            }
            i++;
        }
        else
        /*TBi abesent*/
        {
            if (searchTB3)
            {
                sCReaderParamsPtr->BWI = 4;
                sCReaderParamsPtr->CWI = 5;  
                searchTB3 = 0;
#if (ATR_PARSE_PRINT == 1)
                sc_debug("TB3 absent, however set BWI = 0x%x, CWI = 0x%x\r\n",sCReaderParamsPtr->BWI, sCReaderParamsPtr->CWI);
#endif
            }
        }
    
        if (Yi&0x4)
        {
            /*TCi present*/
            if (i>=length)
            {
                /*TCi not found;hence generate error*/
                goto out; 
            }
      
            if (searchTC3)
            {
                searchTC3 = 0x00;
                if(buff[i] != 0x00)
                {
                    goto out;
                }
            }
            i++;
        }
        else
        {
            /*TCi absent*/
            searchTC3 = 0;
        }
    
        if (Yi & 0x8)
        {
            /*TDi present*/
            if(i>=length)
            {
                /*TDi not found;hence generate error*/
                goto out;
            }
            Yi = buff[i]>>4;
            typeTi = buff[i]&0xf;      
            i++;
        }
        else
        {
            /*TDi absent*/
            Yi = 0;
        }
    
    }//while(Yi)          
    /*CWT should be greater than CGT*/
    if (sCReaderParamsPtr->t1Indicated == 1)
    {
        if ((sCReaderParamsPtr->GTN) != 0xFF)
        {
            if (((2 << (sCReaderParamsPtr->CWI)) < (sCReaderParamsPtr->GTN) + 1))
            {
                goto out;
            }
        }   
    }
      
    return 0;
  
out:        
    return -1;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_pps
 * Description   : ISO7816 PPS section
 * This function send PPS comand to smart card and received the result then 
 * re-configure for proper PPS settings
 *
 *END**************************************************************************/
void scr_pps(smartcardreader_params_t *sCReaderParamsPtr)
{
    uint16_t length;
    uint16_t i;
    
    if (sCReaderParamsPtr->modeNegotiable == 0)
    {
        sc_debug("TA2 is present.Not in Negotiable mode \r\n");
    }
    else
    {
        sc_debug("TA2 is absent.In Negotiable mode \r\n");
    }

    if (sCReaderParamsPtr->t1Indicated)
    {
        sc_debug("T1 card in nogotialbe mode, NO PPS sent\r\n");
        return;
        
    }
    else
    {
        //current PPS, only reset Fi/Di to the CARD
        gTxBuf[0] = 0xFF;
        gTxBuf[1] = 0x10;
        gTxBuf[2] = sCReaderParamsPtr->TA1;
        gTxBuf[3] = gTxBuf[0]^gTxBuf[1]^gTxBuf[2];
        length = 4;
    }
#if    (PPS_INFO_PRINT == 1)
    sc_debug("PPS Section: data send: ");
    for(i = 0; i <length; i++)
    {
        sc_debug("%02X", gTxBuf[i]);
    }
    sc_debug("\r\n");
#endif
    length = scr_send_pps_data(sCReaderParamsPtr, gTxBuf, gRxBuf, length);
 
#if     (PPS_INFO_PRINT == 1)
    sc_debug("PPS Section: data recv: ");
    for (i = 0; i <length; i++)
    {
        sc_debug("%02X", gRxBuf[i]);
    }
    sc_debug("\r\n");
#endif
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_send_pps_data
 * Description   : ISO7816 PPS section data send and receive
 * This function send PPS comand data to smart card and received the data  
 *
 *END**************************************************************************/
int16_t scr_send_pps_data(smartcardreader_params_t *sCReaderParamsPtr, uint8_t *ppsRequest, uint8_t *ppsResponse, uint16_t length)
{
    int16_t rcvLength;
    int16_t expPPSRespLenght;
 
    sCReaderParamsPtr->wwtExpired = 0;
    sCReaderParamsPtr->cwtExpired = 0;
    sCReaderParamsPtr->bwtExpired = 0;
  
    /* Reset WWT timer */
    scr_reset_wwt_timer(sCReaderParamsPtr);
  
    /* Enable WWT timer before starting T=0 transport */
    scr_enable_wwt(sCReaderParamsPtr);  
  
    scr_transmit_chars_polling(sCReaderParamsPtr, ppsRequest, length);  
  
    expPPSRespLenght = length;
    rcvLength = scr_receive_chars_polling(sCReaderParamsPtr, ppsResponse, length);
  
    scr_disable_wwt(sCReaderParamsPtr);  
  
    sCReaderParamsPtr->ppsDone = (rcvLength != expPPSRespLenght)?0:1;

    //reconfigure the bandrate after PPS
    scr_configure_baudrate(sCReaderParamsPtr);

    return rcvLength;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_send_command_with_data
 * Description   : ISO7816 data linker lay data send and receive
 * This function send comand with data to smart card and received the data  
 *
 *END**************************************************************************/
int32_t sc_send_command_with_data(sc_adpu_command_t *sc_adpu_command, sc_adpu_response_t *sc_adpu_response)
{
    uint32_t length;
    uint32_t i;
    
    memset(sc_adpu_response, 0, LE_MAX+3);
    gTxBuf[0] = sc_adpu_command->adpuHeader.CLA;
    gTxBuf[1] = sc_adpu_command->adpuHeader.INS;
    gTxBuf[2] = sc_adpu_command->adpuHeader.P1;
    gTxBuf[3] = sc_adpu_command->adpuHeader.P2;
    gTxBuf[4] = sc_adpu_command->adpuHeader.P3;

#if (SEND_CLA_PRINT == 1)
    sc_print("SEND CLA -> INS", gTxBuf, 5);
#endif
    //send command to smart card, 5 bytes
    scr_transmit_chars_polling(&sCReaderParams, &gTxBuf[0], 5);
    
    //get procedure bytes from smart card, 1-2 bytes
    length = scr_receive_chars_polling(&sCReaderParams, gRxBuf, 2);

#if (SEND_CLA_PRINT == 1)
    sc_print("SEND CLA <-RESP", gRxBuf, length);
#endif
    sc_response_status_check(&sCReaderParams, sc_adpu_response, gRxBuf, length);

    while (sc_adpu_response->SW1 != 0x90)
    { 
        if((sc_adpu_response->SW1 == gTxBuf[1])||(sc_adpu_response->SW1 == (gTxBuf[1]^0xFF)))
        {
            sc_adpu_response->dataRequired = 1;
        }
        else if(sc_adpu_response->SW1 == 0x6C)
        {
            sc_adpu_response->cmdResend = 1;
        }
        else if((sc_adpu_response->SW1 == 0x61)||(sc_adpu_response->SW1 == 0x9F)||(sc_adpu_response->SW1 == 0x91))
        {
            sc_adpu_response->cmdGetResp = 1;
            sc_adpu_command->adpuBody.LE    = gRxBuf[1];
        }
        else if(sc_adpu_response->SW1 == 0x62)
        {
            sc_adpu_response->cmdGetResp = 1;
            sc_adpu_command->adpuBody.LE    = 0;
        }
        else if((sc_adpu_response->SW1 == 0x6D))
        {
            break;
        }
        else if((sc_adpu_response->SW1 == 0xB0))
        {
            sc_debug("GET IDDIC command\r\n");
            break;
        }
        else
        {
            break;
        }
 
        //if more command data needed
        if(sc_adpu_response->dataRequired)
        {
            for(i = 0; i<sc_adpu_command->adpuBody.LC; i++)
            {
                gTxBuf[i] = sc_adpu_command->adpuBody.Data[i];
            }
#if (SEND_CLA_PRINT == 1)
            sc_print("SEND CLA ->DATA", gTxBuf, sc_adpu_command->adpuBody.LC);
#endif
    
            scr_transmit_chars_polling(&sCReaderParams, &gTxBuf[0], sc_adpu_command->adpuBody.LC);
    
            length = scr_receive_chars_polling(&sCReaderParams, gRxBuf, 2);

#if (SEND_CLA_PRINT == 1)
            sc_print("SEND CLA <-RESP", gRxBuf, length);
#endif
            sc_response_status_check(&sCReaderParams, sc_adpu_response, gRxBuf, length);
        }
        //if resend command needed, with SW2 as P3
        if (sc_adpu_response->cmdResend)
        {
            gTxBuf[4] = gRxBuf[1];
#if (SEND_CLA_PRINT == 1)
            sc_print("SEND CLA -> INS", gTxBuf, 5);
#endif
    
            scr_transmit_chars_polling(&sCReaderParams, &gTxBuf[0], 5);
    
            length = scr_receive_chars_polling(&sCReaderParams, gRxBuf, 2);
#if (SEND_CLA_PRINT == 1)
            sc_print("SEND CLA <-RESP", gRxBuf, length);
#endif
            sc_response_status_check(&sCReaderParams, sc_adpu_response, gRxBuf, length);
        }    
        //if get response command needed, with SW2 as P3
        if (sc_adpu_response->cmdGetResp)
        {
            sc_adpu_command->adpuHeader.INS = 0xC0;
            sc_adpu_command->adpuHeader.P1  = 0;
            sc_adpu_command->adpuHeader.P2  = 0;
            
            sc_get_response(sc_adpu_command, sc_adpu_response);
        }
        sc_adpu_response->dataRequired = 0;
        sc_adpu_response->cmdResend = 0;
        sc_adpu_response->cmdGetResp = 0;

    }
    return length;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_send_command_without_data
 * Description   : ISO7816 data linker lay data send and receive
 * This function send comand without data to smart card and received the data  
 *
 *END**************************************************************************/
int32_t sc_send_command_without_data(sc_adpu_command_t *sc_adpu_command, sc_adpu_response_t *sc_adpu_response)
{
    uint16_t length;
    uint32_t i;
    
    memset(sc_adpu_response, 0, LE_MAX+3);
    gTxBuf[0] = sc_adpu_command->adpuHeader.CLA;
    gTxBuf[1] = sc_adpu_command->adpuHeader.INS;
    gTxBuf[2] = sc_adpu_command->adpuHeader.P1;
    gTxBuf[3] = sc_adpu_command->adpuHeader.P2;
    gTxBuf[4] = sc_adpu_command->adpuHeader.P3;

    length = (gTxBuf[4] == 0) ? 256 : gTxBuf[4];

#if (SEND_CLA_PRINT == 1)
    sc_print("SEND CLA -> INS", gTxBuf, 5);
#endif
    //send command to smart card, 5 bytes
    scr_transmit_chars_polling(&sCReaderParams, &gTxBuf[0], 5);
    
    //get procedure bytes from smart card, length + 3 bytes
    length = scr_receive_chars_polling(&sCReaderParams, gRxBuf, length+3);

    if (length == 2)
    {
#if (SEND_CLA_PRINT == 1)
        sc_print("SEND CLA ->RESP", gRxBuf, length);
#endif
        sc_response_status_check(&sCReaderParams, sc_adpu_response, gRxBuf, length);
        //in case, data required length is not correct, reset it and resend the command
        if(gRxBuf[0] == 0x67)
        {
            gTxBuf[4] = gRxBuf[1];
        }
#if (SEND_CLA_PRINT == 1)
        sc_print("SEND CLA -> INS", gTxBuf, 5);
#endif
         //send command to smart card, 5 bytes
        scr_transmit_chars_polling(&sCReaderParams, &gTxBuf[0], 5);
    
        //get procedure bytes from smart card, length + 3 bytes
        length = scr_receive_chars_polling(&sCReaderParams, gRxBuf, sc_adpu_command->adpuHeader.P3+3);
    }
    
    if (length == 2)
    {
        //in case, data required length is not correct, reset it and resend the command
        sc_debug("Wrong P3 used, please check your code\r\n");
        return length;
    }
    if (length >= 2)
    {
        for (i = 0; i < length - 2; i++)
        {
             sc_adpu_response->Data[i + 1] = gRxBuf[i + 1];
        }
    }

#if (SEND_CLA_PRINT == 1)
    sc_print("SEND CLA <-RESP", gRxBuf, length);
#endif
    sc_response_status_check(&sCReaderParams, sc_adpu_response, gRxBuf, length);
  
    return length;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_get_response
 * Description   : ISO7816 data linker layer get response command
 * This function received data from smart card  
 *
 *END**************************************************************************/
int32_t sc_get_response(sc_adpu_command_t *sc_adpu_command, sc_adpu_response_t *sc_adpu_response)
{
    uint32_t length;
    uint32_t i;
    
    gTxBuf[0] = sc_adpu_command->adpuHeader.CLA;
    gTxBuf[1] = 0xC0;
    gTxBuf[2] = sc_adpu_command->adpuHeader.P1;
    gTxBuf[3] = sc_adpu_command->adpuHeader.P2;
    gTxBuf[4] =
        (sc_adpu_command->adpuBody.LE == 256) ? 0 : sc_adpu_command->adpuBody.LE;
    
#if (GET_RESPONSE_PRINT == 1)
    sc_print("GET RESP -> CLA", gTxBuf, 5);
#endif
    scr_transmit_chars_polling(&sCReaderParams, &gTxBuf[0], 5);
    
    length = scr_receive_chars_polling(&sCReaderParams, gRxBuf, (sc_adpu_command->adpuBody.LE) + 3);
    
    for(i = 0; i < length - 2; i++)
        sc_adpu_response->Data[i+1] = gRxBuf[i+1];

#if (GET_RESPONSE_PRINT == 1)
    sc_print("GET RESP <-RESP", gRxBuf, length);
#endif
    
    sc_response_status_check(&sCReaderParams, sc_adpu_response, gRxBuf, length);

    return length;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_t1_send_command
 * Description   : ISO7816 data linker layer send T1 package and receive
 * This function send T1 package data and receive package data from smart card  
 *
 *END**************************************************************************/
int32_t sc_t1_send_command(ttl_cmd_t *ttl_cmd, ttl_resp_t *ttl_resp)
{
    uint32_t length;
    uint32_t i;
    length = 0;
    i = 0;
    memset(gTxBuf, 0, LE_MAX+4);
    gTxBuf[0] = ttl_cmd->NAD;//NAD
    gTxBuf[1] = ttl_cmd->PCB;//PCB
    gTxBuf[2] = ttl_cmd->LEN;
    gTxBuf[3] = ttl_cmd->sc_adpu_command->adpuHeader.CLA;
    gTxBuf[4] = ttl_cmd->sc_adpu_command->adpuHeader.INS;
    gTxBuf[5] = ttl_cmd->sc_adpu_command->adpuHeader.P1;
    gTxBuf[6] = ttl_cmd->sc_adpu_command->adpuHeader.P2;
    if ((ttl_cmd->sc_adpu_command->adpuBody.LC))
    {
        length = ttl_cmd->sc_adpu_command->adpuBody.LC;
        gTxBuf[7] = ttl_cmd->sc_adpu_command->adpuBody.LC;
        for (i = 0; i < length; i++)
            gTxBuf[i + 8] = ttl_cmd->sc_adpu_command->adpuBody.Data[i];
        length++;
    }
    if(ttl_cmd->sc_adpu_command->adpuBody.LE)
        gTxBuf[length + 7] = ttl_cmd->sc_adpu_command->adpuBody.LE;
    gTxBuf[ttl_cmd->LEN + 3] = 0;
    for(i = 0; i<ttl_cmd->LEN + 3; i++)
        gTxBuf[ttl_cmd->LEN+3] ^= gTxBuf[i];
    ttl_cmd->LRC = gTxBuf[ttl_cmd->LEN + 3];
#if (SEND_CLA_PRINT == 1)
    sc_print("SEND CLA -> INS", gTxBuf, gTxBuf[2] + 4);
#endif
    //send command to smart card, 5 bytes
    scr_transmit_chars_polling(&sCReaderParams, &gTxBuf[0], 9);
    
    //get procedure bytes from smart card, 1-2 bytes
    length = scr_receive_chars_polling(&sCReaderParams, gRxBuf, 5);

#if (SEND_CLA_PRINT == 1)
    sc_print("SEND CLA <-RESP", gRxBuf, length);
#endif
    sc_response_status_check(&sCReaderParams, ttl_resp->sc_adpu_response, gRxBuf, length);
   
    return length;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_print
 * Description   : print information
 * This function print information 
 *
 *END**************************************************************************/
void sc_print(char* info,uint8_t *buf, uint8_t length)
{
    uint8_t i;
    sc_debug("%s(%2d):  ", info,length);
    for(i = 0; i < length; i++)
    {
        sc_debug("%02X ", buf[i]);
    }
    sc_debug("\r\n");
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_response_status_check
 * Description   : Check out T1/T0 status
 * This function check status received from smart card 
 *
 *END**************************************************************************/
void sc_response_status_check(smartcardreader_params_t *sCReaderParamsPtr, sc_adpu_response_t *sc_adpu_response, uint8_t * buf, uint8_t length)
{
    uint16_t i;
    uint8_t temp = 0;
    //T1
    if(sCReaderParamsPtr->t1Indicated)
    {
        if(length < 5)
        {
#if (RECV_STATUS_PRINT == 1)
            sc_debug("T1 response package error\r\n, please do check your code and setup\r\n");
            return;
#endif            
        }
        else
        {
#if (RECV_STATUS_PRINT == 1)
            sc_debug("INFO Field size is %d\r\n", buf[2]);
#endif
            sc_adpu_response->SW1 = buf[length - 3];
            sc_adpu_response->SW2 = buf[length - 2];
            for (i = 0; i < length; i++)
            {
                temp ^= buf[i]; 
            }
            if(temp != 0)
            {
#if (RECV_STATUS_PRINT == 1)
                sc_debug("EDC not zero, error dectected\r\n");
                return;
#endif
            }
        }
    }
    else//T0
    {
        if (length == 1)
            sc_adpu_response->SW1 = buf[0];
        else
        {
            sc_adpu_response->SW1 = buf[length - 2];
            sc_adpu_response->SW2 = buf[length - 1];
        }
    }
    //we should provide more details information, but now only these
#if (RECV_STATUS_PRINT == 1)
    if(length == 1)
        sc_debug("SW1 SW2 received: INS or ~INS 0x%02X\r\n", sc_adpu_response->SW1);
    else
    {
        sc_debug("SW1 SW2 received: 0x%02X%02X\r\n", sc_adpu_response->SW1, sc_adpu_response->SW2);
    }
#endif
    
}

/*******************************************************************************
 * EOF
 ******************************************************************************/
