/*
 * Copyright (c) 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "common.h"
#include "metering_modules.h"
#include "drivers.h"
#include "scr_interface.h"
#include "scr_gsm11.h"


/*******************************************************************************
 * Defines
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
* Code
******************************************************************************/

static void scr_interface_delay_cycles(uint32_t cycles)
{
    uint32_t ins_f = POWER_METERING_SYS_CLK_FREQ / POWER_METERING_SMART_CARD_CLOCK_FREQ + 1;

    while (cycles--)
        arch_delay(ins_f);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_interface_init
 * Description   : Smart card reader interface initialization function.
 * This function will initialization smart card reader interface
 *
 *END**************************************************************************/
void scr_interface_init(void)
{
    /* Reset Pin, PTE3 */
    SMART_CARD_RESET_PIN_INIT();

    /* VCC Pin, PTK5 */
    SMART_CARD_VOLTAGE_PIN_INIT();

    /* Clk pin, PTE5, QTMR0_TMR3 */
    SMART_CARD_CLK_PIN_INIT();

    /* IO pin, PTK6, UART1_TXD */
    SMART_CARD_IO_PIN_INIT();
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_clock_init
 * Description   : Smart card reader interface clock initialization function.
 * This function will initialization smart card reader clock
 *
 *END**************************************************************************/
uint32_t scr_clock_init(void)
{
    uint16_t mod_val;

    /* Clock should between 1M to 5M */

    /* CLK Pin, XBAR_OUT0, We use QTMR0_Tmr3 to output clock */
    mod_val = (POWER_METERING_BUS_CLK_FREQ / POWER_METERING_SMART_CARD_CLOCK_FREQ) / 2 - 1;
    TMR_Init(CH3,
            TMR_CH_FIXED_FREQ_PWM_MODE_CONFIG(BUS_CLK_DIV1),
            mod_val,
            0, 0, mod_val, 0, PRI_LVL2, (TMR_CH_CALLBACK)NULL);
    //TMR_Enable(CH3);

    return (POWER_METERING_BUS_CLK_FREQ / ((mod_val + 1) << 1));
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_clock_active
 * Description   : Smart card reader interface clock activate function.
 * This function will activate smart card reader clock
 *
 *END**************************************************************************/
static void scr_clock_active(void)
{
    TMR_Enable(CH3);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_clock_deactive
 * Description   : Smart card reader interface clock deactivate function.
 * This function will deactivate smart card reader clock
 *
 *END**************************************************************************/
static void scr_clock_deactive(void)
{
    TMR_Disable(CH3);
}


/*FUNCTION**********************************************************************
 *
 * Function Name : scr_interface_deactivate
 * Description   : Smart card reader interface deactivate function.
 * This function will deactivate smart card reader interface
 *
 *END**************************************************************************/
void scr_interface_deactivate(smartcardreader_params_t *sCReaderParamsPtr)
{
    /*  Smart Deinit sequence:
     * 1. Reset.
     * 2. Clock
     * 3. VCC
     */

    /* Need to shutdown UART first, or Party error interrupt will be triggered. */
    scr_emv_deinit(sCReaderParamsPtr);

    /* Disable CMDVCC */

    /* Deinitializes the SC_PIN_RESET_GPIO, RST to 0 */
    SMART_CARD_RESET(0);
    scr_interface_delay_cycles(1000);

    /* CLK to 0 */
    scr_clock_deactive();

    scr_interface_delay_cycles(1000);

    /* VCC to 0 */
    SMART_CARD_VOLTAGE(0);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_interface_activate
 * Description   : Smart card reader interface activate function.
 * This function will activate smart card reader interface
 *
 *END**************************************************************************/
void scr_interface_activate(smartcardreader_params_t *sCReaderParamsPtr, uint8_t vcc, uint8_t resetType)
{
    /*Enable Initial character detection first*/
    scr_enable_init_detect(sCReaderParamsPtr);

    /*  Smart Init sequence:
     * 1. Ground.
     * 2. VCC
     * 3. Clock
     * 4. Reset
     */
    /* Set reset to 0 */
    SMART_CARD_RESET(0);

    /* Enable VCC, VCC to 1 */
    SMART_CARD_VOLTAGE(1);
    scr_interface_delay_cycles(1000);

    scr_clock_active();
    scr_interface_delay_cycles(40000);
  
    /* RST to 1 */
    SMART_CARD_RESET(1);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_configure_baudrate
 * Description   : Smart card reader interface UART BPS initialization function.
 * This function will initialize smart card reader interface UART bps
 *
 *END**************************************************************************/
void scr_configure_baudrate(smartcardreader_params_t *sCReaderParamsPtr)
{
    register uint16_t sbr, brfa;
    uint8_t temp;
    uint32_t core_clk_hz;

    core_clk_hz = POWER_METERING_SYS_CLK_FREQ;

    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    /* Calculate baud settings */

    /* UART baud rate = UART module clock / (16  (SBR[12:0] + BRFD)), BRFD = 0
       so SBR[12:0] = UART module clock / (UART baud rate * 16) */
    /* etu = F / (D * fs) and etu = 1 / UART baud rate, so UART baud rate = (D * fs) / F */
    /* So, SBR[12:0] = UART module clock / (((D * fs) / F) * 16) = (UART module clock * F) / (D * fs * 16) */
    sbr = (uint16_t)((core_clk_hz / 1000) * sCReaderParamsPtr->Fi / (sCReaderParamsPtr->Di * sCReaderParamsPtr->sCClockKHz * 16));

    /* Save off the current value of the UARTx_BDH except for the SBR field */
    temp = UART_BDH_REG(ptr) & ~(UART_BDH_SBR(0x1F));

    UART_BDH_REG(ptr) = temp |  UART_BDH_SBR(((sbr & 0x1F00) >> 8));
    UART_BDL_REG(ptr) = (uint8_t)(sbr & UART_BDL_SBR_MASK);

    /* Determine if a fractional divider is needed to get closer to the baud rate */
    brfa = (32 * (core_clk_hz / 1000) * sCReaderParamsPtr->Fi * 1000/(sCReaderParamsPtr->Di * sCReaderParamsPtr->sCClockKHz * 16)) - (sbr * 32);

    /* Save off the current value of the UARTx_C4 register except for the BRFA field */
    temp = UART_C4_REG(ptr) & ~(UART_C4_BRFA(0x1F));

    UART_C4_REG(ptr) = temp |  UART_C4_BRFA(brfa);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_cold_reset
 * Description   : Smart card reader interface cold reset function.
 * This function will cold reset smart card reader interface 
 *
 *END**************************************************************************/
void scr_cold_reset(smartcardreader_params_t *sCReaderParamsPtr)
{
    scr_interface_deactivate(sCReaderParamsPtr);
    scr_interface_activate(sCReaderParamsPtr, sCReaderParamsPtr->vcc, kColdReset);
   /*Enable the receiver*/
    UART_C3_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C3_TXDIR_MASK;
    UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C2_RE_MASK;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_warm_reset
 * Description   : Smart card reader interface warm reset function.
 * This function will warm reset smart card reader interface 
 *
 *END**************************************************************************/
void scr_warm_reset(smartcardreader_params_t *sCReaderParamsPtr)
{
    scr_interface_activate(sCReaderParamsPtr, sCReaderParamsPtr->vcc, kWarmReset);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_reject_card
 * Description   : Smart card reader interface reject function.
 * This function will reject smart card reader interface 
 *
 *END**************************************************************************/
void scr_reject_card(smartcardreader_params_t *sCReaderParamsPtr)
{
    uint32_t tempDelay = 50000000;
    scr_interface_deactivate(sCReaderParamsPtr);
  
    /*Wait for 100 ms*/
    while (tempDelay--)
        NOP();
}

void scr_emv_deinit(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~(UART_C2_TE_MASK | UART_C2_RE_MASK);
    UART_C1_REG(sCReaderParamsPtr->uartMemMapPtr) = 0;
    UART_C3_REG(sCReaderParamsPtr->uartMemMapPtr) = 0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_emv_init
 * Description   : Smart card reader interface UART initialization function.
 * This function will initialize smart card reader interface UART into ISO7816 mode
 *
 *END**************************************************************************/
void scr_emv_init(smartcardreader_params_t *sCReaderParamsPtr)
{
    uint16_t temp16;

    scr_configure_baudrate(sCReaderParamsPtr);

    //Clear all pending status if set
    UART_IS7816_REG(sCReaderParamsPtr->uartMemMapPtr) = 0xFF;
    //Set Wait FD to 1
    UART_WF7816_REG(sCReaderParamsPtr->uartMemMapPtr) = sCReaderParamsPtr->Di;

    /* 
    * EMV expectation: WWT = (960 x D x WI) + (D x 480) = 480xD x (2xWI + 1)
    * UART formula: WI x 480
    */
    temp16 = sCReaderParamsPtr->Di * ((2 * sCReaderParamsPtr->WI) + 1);
    UART_WP7816A_T0_REG(sCReaderParamsPtr->uartMemMapPtr) = (temp16 & 0xFF00) >> 8;      
    UART_WP7816B_T0_REG(sCReaderParamsPtr->uartMemMapPtr) = temp16 & 0xFF;

    /* 
    * EMV expectation: GT = GTN not equal to 255 -> 12 + GTN
    = GTN equal to 255 -> 12
    * UART formula: same as above
    */  
    UART_WN7816_REG(sCReaderParamsPtr->uartMemMapPtr) = sCReaderParamsPtr->GTN;  

    UART_C1_REG(sCReaderParamsPtr->uartMemMapPtr) = 0|
                                                  UART_C1_LOOPS_MASK |
                                                  UART_C1_RSRC_MASK |
                                                  UART_C1_M_MASK |
                                                  UART_C1_PE_MASK;

    if (sCReaderParamsPtr->parity)
    {
        UART_C1_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C1_PT_MASK;
    }

    UART_S2_REG(sCReaderParamsPtr->uartMemMapPtr) = 0;

    //UART_MODEM_REG(sCReaderParamsPtr->uartMemMapPtr) = 0;

    UART_C4_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~(UART_C4_MAEN1_MASK|UART_C4_MAEN2_MASK);
    UART_C5_REG(sCReaderParamsPtr->uartMemMapPtr) = 0;

    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) = UART_C7816_INIT_MASK |
                                                     UART_C7816_ISO_7816E_MASK| 
                                                     UART_C7816_ONACK_MASK| 
                                                     UART_C7816_ANACK_MASK;

    /* 
    * Error detection and Correction in T=0 mode.
    * TXTHRESHOLD is set to '4' as per EMV requirement. 
    */
    UART_ET7816_REG(sCReaderParamsPtr->uartMemMapPtr) = 0x40;
    /* interrupt setup*/
    UART_IE7816_REG(sCReaderParamsPtr->uartMemMapPtr) = UART_IE7816_TXTE_MASK;        

    UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) = UART_C2_RE_MASK;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_transmit_chars_polling
 * Description   : Smart card reader interface UART TX data function(T0/1).
 * This function transmit data via UART
 *
 *END**************************************************************************/
void scr_transmit_chars_polling(smartcardreader_params_t *sCReaderParamsPtr, uint8_t *buf, int32_t length)
{
    volatile uint8_t temp;
    uint16_t txBufIndex;

    UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C2_RE_MASK;  
    UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C2_TE_MASK; /* disable firstly */
    UART_C3_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C3_TXDIR_MASK; /* set the TX pin as output  */  
    
    if (sCReaderParamsPtr->ttype == kTType1)
    {
        sCReaderParamsPtr->cwtExpired = 0;
        scr_reset_cwt_timer(sCReaderParamsPtr);
        scr_enable_cwt(sCReaderParamsPtr);
        /* the TLEN does not count the NAD, PCB, LEN, LRC, if use CRC, it should + 1 */
        UART_TL7816_REG(sCReaderParamsPtr->uartMemMapPtr) = length-4;
    }
    else //kTType0
    {
        /* Start WWT timer */
        sCReaderParamsPtr->wwtExpired = 0;
        scr_reset_wwt_timer(sCReaderParamsPtr);
        /* Enable WWT timer before starting T=0 transport */
        scr_enable_wwt(sCReaderParamsPtr);
    }
  
    UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C2_TE_MASK; /* enable TX */
    UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C2_RE_MASK; /* enable RX */

    txBufIndex = 0;
 
    /* wait for tx is done from the interrupt handler */
    if (sCReaderParamsPtr->ttype == kTType0)
    {
        while((!sCReaderParamsPtr->wwtExpired) && (txBufIndex != length))
        {
            if (((UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_TC_MASK) == UART_S1_TC_MASK) &&
            ((UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_TDRE_MASK) == UART_S1_TDRE_MASK)  
             )
            {
                UART_D_REG(sCReaderParamsPtr->uartMemMapPtr) = *buf++;   /* this clears UART_S1_REG as well */
                txBufIndex++;
            }
        }
    }
    else
    {
        while ((!sCReaderParamsPtr->cwtExpired) && (txBufIndex != length))
        {
            if (((UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_TC_MASK) == UART_S1_TC_MASK) &&
              ((UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_TDRE_MASK) == UART_S1_TDRE_MASK)  
              )
            {
                UART_D_REG(sCReaderParamsPtr->uartMemMapPtr) = *buf++;   /* this clears UART_S1_REG as well */
                txBufIndex++;
            }
        }
        scr_disable_cwt(sCReaderParamsPtr->uartMemMapPtr);
    }

    /* Wait for TC bit to clear - last byte transmission has started */
    while (!(UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_TC_MASK))
    {
    }
  
    UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C2_TE_MASK; /* disable after transmit */
    UART_C3_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C3_TXDIR_MASK;

    if (sCReaderParamsPtr->ttype == kTType1)
    {
        sCReaderParamsPtr->cwtExpired = 0;
        scr_reset_cwt_timer(sCReaderParamsPtr);
    }
    else //kTType0
    {
        /* Start WWT timer */
        sCReaderParamsPtr->wwtExpired = 0;
        scr_reset_wwt_timer(sCReaderParamsPtr);
    }
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_receive_chars_polling
 * Description   : Smart card reader interface UART RX data function(T0/1).
 * This function receive data via UART
 *
 *END**************************************************************************/
int16_t scr_receive_chars_polling(smartcardreader_params_t *sCReaderParamsPtr, uint8_t* buf, uint16_t length)
{
    int16_t retLength = 0;
    uint16_t i;
      
    if (sCReaderParamsPtr->ttype == kTType1)
    {
        sCReaderParamsPtr->cwtExpired = 0;
        scr_reset_cwt_timer(sCReaderParamsPtr);
        scr_enable_cwt(sCReaderParamsPtr);
        /* the TLEN does not count the NAD, PCB, LEN, LRC, if use CRC, it should + 1 */
        UART_TL7816_REG(sCReaderParamsPtr->uartMemMapPtr) = length - 4;
    }
    else //kTType0
    {
        /* Start WWT timer */
        sCReaderParamsPtr->wwtExpired = 0;
        scr_reset_wwt_timer(sCReaderParamsPtr);
        /* Enable WWT timer before starting T=0 transport */
        scr_enable_wwt(sCReaderParamsPtr);
    }
  
    if (sCReaderParamsPtr->ttype == kTType0)
    {
        /*Set the GTN to 12 etus while receiving */
        UART_WN7816_REG(sCReaderParamsPtr->uartMemMapPtr) = 0;
        //clear any pending RDRF first if have
//        buf[0]=UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr);
//        buf[0]=UART_D_REG(sCReaderParamsPtr->uartMemMapPtr);
        
        UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C2_RE_MASK;
        for (i = 0; i < length; i++)
        {
            while((!(UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr)&UART_S1_RDRF_MASK)))
            {
                if (sCReaderParamsPtr->wwtExpired)
                {        
                    retLength = i;
                    return retLength;
                 }            
            }
        
            buf[i]=UART_D_REG(sCReaderParamsPtr->uartMemMapPtr);
        
            if (i == 0)
            {
                if ( ((buf[0] == sc_adpu_command.adpuHeader.INS)||(buf[0] ==(sc_adpu_command.adpuHeader.INS^0xFF))) 
                  && (buf[0] != SC_INS_GET_RESPONCE )  //some T0 command reply has no procedure byte, 
                  && (buf[0] != SC_INS_READ_BINARY )  //but first data is INS and following replied data
                  && (buf[0] != SC_INS_STATUS )  //in this case, we can't jump out of for loop too early,
                  && (buf[0] != SC_INS_READ_RECORD )  //Keep receiving data from Smart Card
                  && (buf[0] != 0xFF )
                 )
                {
                    length = 1;
                    return length;
                } 
            }           
        }      
        retLength = i;
    }
    else
    {
        sCReaderParamsPtr->cwtExpired = 0;
        scr_reset_cwt_timer(sCReaderParamsPtr);
        scr_enable_cwt(sCReaderParamsPtr);
    
        /* TTYpe = 1 */      
        UART_IS7816_REG(sCReaderParamsPtr->uartMemMapPtr) = 0xFF; //clear all flag if set
        UART_C3_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C3_TXDIR_MASK;
        UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C2_RE_MASK;
      
        /* wait for the 1st character of the return block */
        while (!(UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_RDRF_MASK))
        {
            if (sCReaderParamsPtr->bwtExpired)
            {
                 return -1;//did not receive the return block within BWT 
            }
        }
      
        buf[0] = UART_D_REG(sCReaderParamsPtr->uartMemMapPtr);
      
        for (i = 1; i < length; i++)
        {
            while (!(UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_RDRF_MASK) && (!sCReaderParamsPtr->cwtExpired));
            if (UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_RDRF_MASK)
            {
                buf[i]=UART_D_REG(sCReaderParamsPtr->uartMemMapPtr);
            }
        
            if (sCReaderParamsPtr->cwtExpired)
            {
                UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C2_RE_MASK;              
                retLength = i;
                return retLength;
            }
            if (i == 2)
            {
                length = buf[2] + 4;
            }
        }
        UART_C2_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C2_RE_MASK;
        scr_disable_cwt(sCReaderParamsPtr->uartMemMapPtr);
        retLength = i;
    }
    return retLength;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_receive_chars_atr
 * Description   : Smart card reader interface UART RX ATR data function(T0/1).
 * This function receive ATR data via UART
 *
 *END**************************************************************************/
int16_t scr_receive_chars_atr(smartcardreader_params_t *sCReaderParamsPtr, uint8_t *buf, uint16_t length)
{
    int16_t retLength = 0;
    uint16_t atrLoop = 0;
     
    for (atrLoop = 0; atrLoop < length; atrLoop++)
    {
        while ((!(UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_RDRF_MASK))
              &&(!sCReaderParamsPtr->wwtExpired) && (!sCReaderParamsPtr->adtExpired));
        
        if (UART_S1_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_S1_RDRF_MASK)
        {
            buf[atrLoop] = UART_D_REG(sCReaderParamsPtr->uartMemMapPtr);
        }
        
        if (sCReaderParamsPtr->wwtExpired)
        {        
            retLength = atrLoop;
            return retLength;
        }
        
        if (sCReaderParamsPtr->adtExpired)
        {        
            retLength = atrLoop;
            return retLength;
        }
    }      
    return atrLoop;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_reset_wwt_timer
 * Description   : Smart card reader interface UART reset WT timer function.
 * This function reset UART WT timer
 *
 *END**************************************************************************/
void scr_reset_wwt_timer(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_C7816_REG(ptr) &= ~UART_C7816_ISO_7816E_MASK;//reset WT timer//reset WT timer
    UART_IS7816_REG(ptr) = UART_IS7816_WT_MASK;
    UART_C7816_REG(ptr) |= UART_C7816_ISO_7816E_MASK;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_reset_cwt_timer
 * Description   : Smart card reader interface UART reset CWT timer function.
 * This function reset UART CWT timer
 *
 *END**************************************************************************/
void scr_reset_cwt_timer(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_C7816_REG(ptr) &= ~UART_C7816_ISO_7816E_MASK;//reset WT timer//reset CWT timer
    UART_IS7816_REG(ptr) =  UART_IS7816_CWT_MASK;
    UART_C7816_REG(ptr) |=  UART_C7816_ISO_7816E_MASK;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_reset_bwt_timer
 * Description   : Smart card reader interface UART reset BWT timer function.
 * This function reset UART BWT timer
 *
 *END**************************************************************************/
void scr_reset_bwt_timer(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_C7816_REG(ptr) &= ~UART_C7816_ISO_7816E_MASK;//reset WT timer//reset BWT timer
    UART_IS7816_REG(ptr) = UART_IS7816_BWT_MASK;
    UART_C7816_REG(ptr) |= UART_C7816_ISO_7816E_MASK;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_reset_gt_timer
 * Description   : Smart card reader interface UART reset GT timer function.
 * This function reset UART GT timer
 *
 *END**************************************************************************/
void scr_reset_gt_timer(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_C7816_REG(ptr) &= ~UART_C7816_ISO_7816E_MASK;//reset GTV timer
    UART_IS7816_REG(ptr) = UART_IS7816_GTV_MASK;
    UART_C7816_REG(ptr) |= UART_C7816_ISO_7816E_MASK;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_enable_wwt_timer
 * Description   : Smart card reader interface UART enable WT timer function.
 * This function enable UART WT timer
 *
 *END**************************************************************************/
void scr_enable_wwt(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_IE7816_REG(ptr) |= UART_IE7816_WTE_MASK;  
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_disable_wwt_timer
 * Description   : Smart card reader interface UART disable WT timer function.
 * This function disable UART WT timer
 *
 *END**************************************************************************/
void scr_disable_wwt(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_IE7816_REG(ptr) &= ~UART_IE7816_WTE_MASK; 
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_enable_adt_timer
 * Description   : Smart card reader interface UART enable ADT timer function.
 * This function enable UART ADT timer
 *
 *END**************************************************************************/
void scr_enable_adt(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_IE7816_REG(ptr) |= UART_IE7816_ADTE_MASK;
	
    /*Disable ISO-7816 mode first*/
    UART_C7816_REG(ptr) &= ~UART_C7816_ISO_7816E_MASK;
    UART_AP7816A_T0_REG(ptr) = (ATR_DURATION  & 0xFF00) >> 8;
    UART_AP7816B_T0_REG(ptr) = ATR_DURATION & 0x00ff; 
  
    UART_C7816_REG(ptr) |= UART_C7816_ISO_7816E_MASK;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_disable_adt_timer
 * Description   : Smart card reader interface UART disable ADT timer function.
 * This function disable UART ADT timer
 *
 *END**************************************************************************/
void scr_disable_adt(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_IE7816_REG(ptr) &= ~UART_IE7816_ADTE_MASK;
    UART_AP7816A_T0_REG(ptr) = 0x00;
    UART_AP7816B_T0_REG(ptr) = 0x00;  
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_enable_cwt_timer
 * Description   : Smart card reader interface UART enable CWT timer function.
 * This function enable UART CWT timer
 *
 *END**************************************************************************/
void scr_enable_cwt(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_IE7816_REG(ptr) |= UART_IE7816_CWTE_MASK;  
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_disable_cwt_timer
 * Description   : Smart card reader interface UART disable CWT timer function.
 * This function disable UART CWT timer
 *
 *END**************************************************************************/
void scr_disable_cwt(UART_MemMapPtr ptr)
{
    UART_IE7816_REG(ptr) &= ~UART_IE7816_CWTE_MASK;  
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_enable_bwt_timer
 * Description   : Smart card reader interface UART enable BWT timer function.
 * This function enable UART BWT timer
 *
 *END**************************************************************************/
void scr_enable_bwt(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_IE7816_REG(ptr) |= UART_IE7816_BWTE_MASK;  
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_disable_bwt_timer
 * Description   : Smart card reader interface UART disable BWT timer function.
 * This function disable UART BWT timer
 *
 *END**************************************************************************/
void scr_disable_bwt(smartcardreader_params_t *sCReaderParamsPtr)
{
    UART_MemMapPtr ptr = sCReaderParamsPtr->uartMemMapPtr;
    UART_IE7816_REG(ptr) &= ~UART_IE7816_BWTE_MASK;  
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_enable_init_detect
 * Description   : Smart card reader interface UART enable TS dectection function.
 * This function enable UART TS detection
 *
 *END**************************************************************************/
void scr_enable_init_detect(smartcardreader_params_t *sCReaderParamsPtr)
{
    /*clear all flag if set*/ 
    UART_IS7816_REG(sCReaderParamsPtr->uartMemMapPtr) = 0xFF;  
  
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C7816_INIT_MASK;
    if ((UART_IS7816_REG(sCReaderParamsPtr->uartMemMapPtr) & UART_IS7816_INITD_MASK) == UART_IE7816_INITDE_MASK)
    {
        /*clear flag*/
        UART_IS7816_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_IS7816_INITD_MASK;
    }
 
    /*enable TS detect interrupt*/
    UART_IE7816_REG(sCReaderParamsPtr->uartMemMapPtr)|=UART_IE7816_INITDE_MASK;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_setup_atr
 * Description   : Smart card reader interface UART ATR dectection setup function.
 * This function set UART ATR detection
 *
 *END**************************************************************************/
void scr_setup_atr(smartcardreader_params_t *sCReaderParamsPtr)
{
    uint16_t temp16;
    
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C7816_ISO_7816E_MASK;
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C7816_TTYPE_MASK;
  
    /* 
     * EMV expectation: WWT = (960 x D x WI) + (D x 480) = 480xD x (2xWI + 1)
     * UART formula: WI x 480
     */
    temp16 = sCReaderParamsPtr->Di * ((2 * sCReaderParamsPtr->WI) + 1);
    UART_WP7816A_T0_REG(sCReaderParamsPtr->uartMemMapPtr) = (temp16 & 0xFF00) >> 8;      
    UART_WP7816B_T0_REG(sCReaderParamsPtr->uartMemMapPtr) = temp16 & 0xFF;
  
    /* 
     * EMV expectation: GT = GTN not equal to 255 -> 12 + GTN
     * = GTN equal to 255 -> 12
     * UART formula: same as above
     */
    UART_WN7816_REG(sCReaderParamsPtr->uartMemMapPtr) = sCReaderParamsPtr->GTN;
  
    UART_C1_REG(sCReaderParamsPtr->uartMemMapPtr) = 0|
                                                    UART_C1_LOOPS_MASK |
                                                    UART_C1_RSRC_MASK |
                                                    UART_C1_M_MASK |                                                
                                                    UART_C1_PE_MASK;

    if (sCReaderParamsPtr->parity)
    {
        UART_C1_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C1_PT_MASK;
    }

    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) = UART_C7816_ISO_7816E_MASK | 
                                                       UART_C7816_ONACK_MASK | 
                                                       UART_C7816_ANACK_MASK;
  
    UART_C3_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C3_PEIE_MASK;
  
    /*Setting Rx threshld to 0 so that an interrupt is generated when a NACK is 
    sent either due to parity error or wrong INIT char*/
    UART_ET7816_REG(sCReaderParamsPtr->uartMemMapPtr) &= 0xF0;
  
    /*Enable Interrupt for RXT crossing*/
    UART_IE7816_REG(sCReaderParamsPtr->uartMemMapPtr)|=UART_IE7816_RXTE_MASK;
    
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C7816_ISO_7816E_MASK;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_setup_t0
 * Description   : Smart card reader interface UART initialization to T0 function.
 * This function initialize UART into T0 mode
 *
 *END**************************************************************************/
void scr_setup_t0(smartcardreader_params_t *sCReaderParamsPtr)
{
    uint16_t temp16;
    
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C7816_ISO_7816E_MASK;
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C7816_TTYPE_MASK;
  
    /* 
     * EMV expectation: WWT = (960 x D x WI) + (D x 480) = 480xD x (2xWI + 1)
     * UART formula: WI x 480
     */
    temp16 = sCReaderParamsPtr->Di * ((2 * sCReaderParamsPtr->WI) + 1);
    UART_WP7816A_T0_REG(sCReaderParamsPtr->uartMemMapPtr) = (temp16 & 0xFF00) >> 8;      
    UART_WP7816B_T0_REG(sCReaderParamsPtr->uartMemMapPtr) = temp16 & 0xFF;
  
    /* 
     * EMV expectation: GT = GTN not equal to 255 -> 12 + GTN
     * = GTN equal to 255 -> 12
     * UART formula: same as above
     */
    UART_WN7816_REG(sCReaderParamsPtr->uartMemMapPtr) = sCReaderParamsPtr->GTN;
  
    UART_C1_REG(sCReaderParamsPtr->uartMemMapPtr) = 0|
                                                    UART_C1_LOOPS_MASK |
                                                    UART_C1_RSRC_MASK |
                                                    UART_C1_M_MASK |                                                
                                                    UART_C1_PE_MASK;

    if (sCReaderParamsPtr->parity)
    {
        UART_C1_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C1_PT_MASK;
    }

    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) = UART_C7816_ISO_7816E_MASK | 
                                                       UART_C7816_ONACK_MASK | 
                                                       UART_C7816_ANACK_MASK;
  
   /*Setting Rx threshld to 4 so that an interrupt is generated when a NACK is 
    sent either due to parity error or wrong INIT char*/
    UART_ET7816_REG(sCReaderParamsPtr->uartMemMapPtr) = 0x44;

    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C7816_ISO_7816E_MASK; 
    sCReaderParamsPtr->ttype = kTType0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : scr_setup_t1
 * Description   : Smart card reader interface UART initialization to T1 function.
 * This function initialize UART into T1 mode
 *
 *END**************************************************************************/
void scr_setup_t1(smartcardreader_params_t *sCReaderParamsPtr)
{
    uint16_t bwiVal = 0;
  
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~UART_C7816_ISO_7816E_MASK;
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) |=  UART_C7816_TTYPE_MASK;
  
    /*ttype = 1*/
    /* 
    * EMV expectation: BWT = 11 + (2^BWI x 960 x D) + (D x 960)
    = 11 + (2^BWI + 1) x 960 x D
    * UART formula: BWT = (11 + (BWI ?960 ?GTFD)) * (WTX + 1)    
    */
    bwiVal = (1 << sCReaderParamsPtr->BWI) + 1;
    UART_WP7816A_T1_REG(sCReaderParamsPtr->uartMemMapPtr) = (bwiVal & 0xFF00) >> 8;          
    UART_WP7816B_T1_REG(sCReaderParamsPtr->uartMemMapPtr) = (bwiVal & 0x00FF);  
  
    /* Need to understand the use of this register */
    UART_WP7816_REG(sCReaderParamsPtr->uartMemMapPtr) = 0;
  
    /* 
    * EMV expectation: CWT = ((2^CWI + 11) + 4)
    * UART formula: CWT = 2^CWI1 + CWI2
    * EMV expectation: BGT = 22
    * UART formula: BWT = 16 + BGI
    */
    UART_WGP7816_T1_REG(sCReaderParamsPtr->uartMemMapPtr) = ((sCReaderParamsPtr->CWI)<<4) | ((sCReaderParamsPtr->BGI - 16) & 0x0F); 
    UART_WP7816C_T1_REG(sCReaderParamsPtr->uartMemMapPtr) = 15;   /* EMV = 15, ISO = 11*/
    /* 
    * EMV expectation: GT = GTN not equal to 255 -> 12 + GTN
    = GTN equal to 255 -> 11
    * UART formula: same as above
    */
    UART_WN7816_REG(sCReaderParamsPtr->uartMemMapPtr) = sCReaderParamsPtr->GTN;
  
    UART_C1_REG(sCReaderParamsPtr->uartMemMapPtr) = 0|
                                                    UART_C1_LOOPS_MASK |
                                                    UART_C1_RSRC_MASK |
                                                    UART_C1_M_MASK |                                                
                                                    UART_C1_PE_MASK;
  
    if (sCReaderParamsPtr->parity)
    {
        UART_C1_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C1_PT_MASK;
    }
    
    UART_C4_REG(sCReaderParamsPtr->uartMemMapPtr) &= ~(UART_C4_MAEN1_MASK|UART_C4_MAEN2_MASK);
    UART_C5_REG(sCReaderParamsPtr->uartMemMapPtr) = 0;
  
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) = UART_C7816_TTYPE_MASK | 
                                                       UART_C7816_ISO_7816E_MASK | 
                                                       UART_C7816_ONACK_MASK | 
                                                       UART_C7816_ANACK_MASK;
  
    UART_IE7816_REG(sCReaderParamsPtr->uartMemMapPtr) = UART_IE7816_GTVE_MASK;
  
    UART_C7816_REG(sCReaderParamsPtr->uartMemMapPtr) |= UART_C7816_ISO_7816E_MASK;
  
    sCReaderParamsPtr->ttype = kTType1;
}

/*******************************************************************************
 * EOF
 ******************************************************************************/
