/******************************************************************************
* (c) Copyright 2010-2015, Freescale Semiconductor Inc.
* ALL RIGHTS RESERVED.
***************************************************************************/
#include "common.h"
#include "metering_modules.h"
#include "meterlib_cfg.h"
#include "metering_const.h"
#include "drivers.h"
#include "fraclib.h"
#include "meterlib.h"
//#include "meterliblp.h"
#include <stdint.h>
#include <string.h>

/*
This file implements the metering functions using afe and adc
*/

/******************************************************************************
* static data definitions
******************************************************************************/
/* this variable is stored in non-initialized ram                             */
__no_init volatile tMETERING_RUNTIME_DATA metering_runtime_data; 

/* according to Chinese Standard, 400 imp/kWh, Q/GDW XXX-2012 */
static volatile const frac64 kwh_pulses[] = 
{
    METERLIB_KWH_PR(POWER_METERING_PULSE_NUM)
};

static volatile const frac64 kvarh_pulses[] = 
{
    METERLIB_KVARH_PR(POWER_METERING_PULSE_NUM)
};

static tMETERING_Calib_Data_1PH *pcalib_data_ph1 = NULL;
static tMETERING_Calib_Data_1PH *pcalib_data_ph2 = NULL;
static tMETERING_Calib_Data_1PH *pcalib_data_ph3 = NULL;

static volatile frac32 u24_samplePh1,
                       i24_samplePh1, i24_samplePh2, i24_samplePh3;

static volatile double U_RMS_Ph1, I_RMS_Ph1, P_Ph1, Q_Ph1, S_Ph1, 
                       U_RMS_Ph2, I_RMS_Ph2, P_Ph2, Q_Ph2, S_Ph2,
                       U_RMS_Ph3, I_RMS_Ph3, P_Ph3, Q_Ph3, S_Ph3,
                       wh, varh, freq;
static volatile int32_t phase_rotation;
#if POWER_METERING_USE_AFE_TRIGGER_ADC
static frac16 phase_shift[POWER_METERING_PHRASE_NUM] = { 0 };
#endif
static double cur_p = 0.0, cur_q = 0.0;

static volatile tMETERLIB3PH_DATA mlib = METERLIB3PH_DATA_HILB_REMEZ_50HZ;
static volatile tMETERLIB3PH_DATA mlib_vlpr = METERLIB3PH_DATA_HILB_REMEZ_50HZ_VLPR;
/* 0 - RUNNING, 1 - STOPPING, 2 - STOPPED */
static tMETERING_Status_Flag metering_status_flag = METERING_STATUS_STOPPED;

/******************************************************************************
* callback function definitions
******************************************************************************/
static int32_t metering_func_reset_variable(void)
{
    u24_samplePh1 = 0;
    i24_samplePh1 = i24_samplePh2 = i24_samplePh3 = 0;
    U_RMS_Ph1 = U_RMS_Ph2 = U_RMS_Ph3 = 0;
    I_RMS_Ph1 = I_RMS_Ph2 = I_RMS_Ph3 = 0;
    P_Ph1 = P_Ph2 = P_Ph3 = 0;
    Q_Ph1 = Q_Ph2 = Q_Ph3 = 0;
    S_Ph1 = S_Ph2 = S_Ph3 = 0;
    cur_p = cur_q = 0;

    return 0;
}

/* Measurements callback @ 3000 Hz, duration us, load  % */
static void metering_func_afe_ch0_callback(AFE_CH_CALLBACK_TYPE type, int32_t result)
{
    int16_t temp16;

    if (type == COC_CALLBACK) 
    {
        temp16 = ADC_Read(CHA) - 0x8000;
        u24_samplePh1 = (frac32)temp16 << 8;
        i24_samplePh1 = AFE_ChRead(CH0);
    }
}

/* Measurements callback @ 3000 Hz, duration us, load  % */
static void metering_func_afe_ch1_callback(AFE_CH_CALLBACK_TYPE type, int32_t result)
{
    if (type == COC_CALLBACK)
    {
        i24_samplePh2 = AFE_ChRead(CH1);
        i24_samplePh2 = i24_samplePh1;
    }
}

/* measurements callback @ 3000 Hz, duration us, load  %                      */
static void metering_func_afe_ch2_callback(AFE_CH_CALLBACK_TYPE type, int32_t result)
{
    if (type == COC_CALLBACK) 
    {
        i24_samplePh3 = AFE_ChRead(CH2);
        i24_samplePh3 = i24_samplePh1;

        if (CALI_FLAG_INIT == metering_cfg_data.calib_flag)
        {
            calib_data_update_offsets((tMETERING_CONFIG_FLASH_DATA *)&metering_cfg_data,
                                u24_samplePh1, i24_samplePh1, i24_samplePh2,
                                i24_samplePh3, 0);
        }
        //SWISR_HandlerCall(0);
        SWISR_HandlerFreqCall(0, 6000, 1200);
    }
}

static void metering_func_do_filter(tMETERLIB3PH_DATA *p, 
                                    frac24 u1Q, frac24 i1Q, 
                                    frac24 i2Q, 
                                    frac24 i3Q,
                                    frac32 *whCnt, frac32 *varCnt,                          
                                    frac64 whRes, frac64 varRes,
                                    frac16 *shift)
{
    tENERGY_CNT tmp_cnt;

    phase_rotation = METERLIB3PH_ProcSamples(p, u1Q, i1Q, u1Q, i2Q, u1Q, i3Q, shift);
    METERLIB3PH_CalcWattHours(p, &tmp_cnt, whRes);
    *whCnt = tmp_cnt.ph[1].ex + tmp_cnt.ph[2].ex + tmp_cnt.ph[3].ex;
    METERLIB3PH_CalcVarHours(p, &tmp_cnt, varRes);
    *varCnt = tmp_cnt.ph[1].ex + tmp_cnt.ph[2].ex + tmp_cnt.ph[3].ex;
    METERLIB3PH_CalcAuxiliary(p);
}

/* Calculation callback @1500 Hz, duration 681 us, load 81.7 %                */        
static void metering_func_auxcalc_swisr0_callback(void)
{
	frac16 *tmp_shift = NULL;

#if POWER_METERING_ENABLE_VREF_COMP
    vref_comp_update_cablib_cfg((tMETERING_CONFIG_FLASH_DATA *)&metering_cfg_data);
#endif

#if POWER_METERING_USE_AFE_TRIGGER_ADC == 1
    tmp_shift = phase_shift;
#endif

    metering_func_do_filter((tMETERLIB3PH_DATA*)&mlib,
                          -L_mul((u24_samplePh1 - pcalib_data_ph1->u_offset), pcalib_data_ph1->u_gain),
                          -L_mul((i24_samplePh1 - pcalib_data_ph1->i_offset), pcalib_data_ph1->i_gain),
                          -L_mul((i24_samplePh2 - pcalib_data_ph2->i_offset), pcalib_data_ph2->i_gain),
                          -L_mul((i24_samplePh3 - pcalib_data_ph3->i_offset), pcalib_data_ph3->i_gain),
                          (long *)&metering_runtime_data.wh_cnt, (long *)&metering_runtime_data.varh_cnt,
                          kwh_pulses[0], kvarh_pulses[0],
                          tmp_shift);

    SWISR_HandlerFreqCall(1, 1200, 4);
}

/* Display and user interface callback @ 4 Hz, duration 7654 us, load 3.1 % */
static void metering_func_read_result_swisr1_callback(void)
{
    int32_t calib_result = 0;

    if (METERING_STATUS_RUNNING == metering_status_flag)
    {
        /* Read data from metering engine - this is required for FreeMASTER         */
        METERLIB3PH_ReadResultsPh1((tMETERLIB3PH_DATA*)&mlib,
                                (double *)&U_RMS_Ph1,
                                (double *)&I_RMS_Ph1,
                                (double *)&P_Ph1,
                                (double *)&Q_Ph1,
                                (double *)&S_Ph1);
        METERLIB3PH_ReadResultsPh2((tMETERLIB3PH_DATA*)&mlib,
                                (double *)&U_RMS_Ph2,
                                (double *)&I_RMS_Ph2,
                                (double *)&P_Ph2,
                                (double *)&Q_Ph2,
                                (double *)&S_Ph2);
        METERLIB3PH_ReadResultsPh3((tMETERLIB3PH_DATA*)&mlib,
                                (double *)&U_RMS_Ph3,
                                (double *)&I_RMS_Ph3,
                                (double *)&P_Ph3,
                                (double *)&Q_Ph3,
                                (double *)&S_Ph3);

        cur_p = P_Ph1 + P_Ph2 + P_Ph3;
        cur_q = Q_Ph1 + Q_Ph2 + Q_Ph3;
        
        if (CALI_FLAG_FIN != metering_cfg_data.calib_flag)
        {
            /* Calibration - perform pre-processing conditionally and toggle user LED   */
            do
            {
                calib_result = calib_data_processing((tMETERING_CONFIG_FLASH_DATA *)&metering_cfg_data,
                                                    U_RMS_Ph1, I_RMS_Ph1,
                                                    U_RMS_Ph2, I_RMS_Ph2,
                                                    U_RMS_Ph3, I_RMS_Ph3,
                                                    0, 0,
                                                    P_Ph1, Q_Ph1,
                                                    P_Ph2, Q_Ph2,
                                                    P_Ph3, Q_Ph3,
                                                    0, 0);
            } while (CALI_FLAG_DATA_READY == calib_result);
        }
    }
    else
    {
        AFE_Disable();
        VREF_Disable();
        metering_status_flag = METERING_STATUS_STOPPED;
    }

    /* Read phase voltage frequency */
    freq_detect_get_freq((double *)&freq);

    /* Refresh watchdog Timer                                                   */
    WDOG_WriteRefreshSeq();
}

int32_t metering_func_hardware_init()
{
    tADC_CALIB adc_calib;

    metering_func_reset_variable();

    slcd_disp_set_num1_area(SLCD_DISP_KWA_KVAR_TOT);

    /* Enable PLL for AFE */
    //PLL_Enable(PLL32KREF_SRC1);
  
    /* VREF module must be initialized after SIM module                         */
    vref_comp_init((tMETERING_CONFIG_FLASH_DATA *)&metering_cfg_data);
  
    /* Software callbacks                                                       */
    SWISR_HandlerInit(0, PRI_LVL2, metering_func_auxcalc_swisr0_callback);
    SWISR_HandlerInit(1, PRI_LVL3, metering_func_read_result_swisr1_callback);

    SIM_CtrlPLLVLP(FALSE);
    /* clear LP bit, PLL not disabled,enable in bypass mode(BLPI-->FBI) */
    FLL_CtrlLP(FALSE);

    /* AFE module                                                               */
    SIM_SelAfePllClk(SIM_MCG_PLL_CLK);
    /* AFE sample rate = 6144Khz / 1024 = 6Khz(~166.7us), AFE channels delay = 2 * 70 / 6.144 = 22.768us */
    AFE_ChInit(CH0,
                AFE_CH_SWTRG_CCM_PGAOFF_CONFIG(POWER_METERING_AFE_OSR),
                (0 * POWER_METERING_SAR_CONT),
                PRI_LVL1,
                (AFE_CH_CALLBACK)metering_func_afe_ch0_callback);
    AFE_ChInit(CH1,
                AFE_CH_SWTRG_CCM_PGAOFF_CONFIG(POWER_METERING_AFE_OSR),
                (0 * POWER_METERING_SAR_CONT),
                PRI_LVL1,
                (AFE_CH_CALLBACK)metering_func_afe_ch1_callback);
    AFE_ChInit(CH2, 
                AFE_CH_SWTRG_CCM_PGAOFF_CONFIG(POWER_METERING_AFE_OSR),
                (0 * POWER_METERING_SAR_CONT),
                PRI_LVL1,
                (AFE_CH_CALLBACK)metering_func_afe_ch2_callback);
    /* PLL = 32768 Hz * 375 = 12288000 */
    AFE_Init(AFE_MODULE_RJFORMAT_CONFIG(AFE_PLL_CLK, POWER_METERING_AFE_DIV, POWER_METERING_AFE_PLL_CLK));

    /* ADC module                                                               */
    PORT_Init (PORTC, PORT_MODULE_ALT0_MODE, PIN5);
    PORT_Init (PORTC, PORT_MODULE_ALT0_MODE, PIN6);
    PORT_Init (PORTC, PORT_MODULE_ALT0_MODE, PIN7);
    /* Do ADC calibration will have some benefits */
    ADC_ExecCalib(ADC_MODULE_DIV4_16B_HWTRG_IREF_CONFIG, &adc_calib);
    ADC_SaveCalib(&adc_calib);
    ADC_Init(ADC_MODULE_DIV4_16B_HWTRG_IREF_CONFIG,
            HWAVG_OFF,
            ADC_CH_SE_POLL_CONFIG(POWER_METERING_VOLTAGE_ADC_CH_PH1),
            ADC_CH_DISABLE_CONFIG,
            ADC_CH_DISABLE_CONFIG,
#ifdef POWER_METERING_EXTERNAL_TEMP_AD
            ADC_CH_SE_POLL_CONFIG(POWER_METERING_EXTERNAL_TEMP_AD),
#else
            ADC_CH_SE_POLL_CONFIG(TEMP_SENSOR),
#endif
            PRI_LVL1,
            (ADC_CALLBACK)NULL);

#if POWER_METERING_USE_QTMR_TRIGGER_ADC
    SIM_SelTmrScs(CH1, 1);

    /* TMR to SAR connection, SAR start triggers */
    XBAR_Path(XBAR_TMR1, XBAR_ADCTRGCHA);

    /* Sensor phase error compensation, Phase 1                                 */
    XBAR_Path(XBAR_AFE0COC,XBAR_TMR1SEC);
    TMR_Init(CH1,
            TMR_PHASE_ERROR_COMPENSATION_MODE(BUS_CLK_DIV1, SEC_CNTR1_INP, LOAD_POSEDGE),
            pcalib_data_ph1->delay,
            0, 0, 0, 0, PRI_LVL1, (TMR_CH_CALLBACK)NULL);
    TMR_Enable(CH1);

#elif POWER_METERING_USE_AFE_TRIGGER_ADC
    XBAR_Path(XBAR_AFE0COC, XBAR_ADCTRGCHA);
#else
    #error "One source need to be used to trigger ADC!"
#endif

    /* Frequency measurement */
    freq_detect_init();

    metering_status_flag = METERING_STATUS_RUNNING;

    if ((CALI_FLAG_INIT == metering_cfg_data.calib_flag) || (CALI_FLAG_ITERA_INIT == metering_cfg_data.calib_flag))
    {
        /* 10s delay to wait calibration source stable */
        arch_delay(POWER_METERING_STARTUP_DELAY);
    }

    AFE_SwTrigger(CH0 | CH1 | CH2);
  
    return 0;
}

int32_t metering_func_get_cur_p_q_total(double *p, double *q)
{
    *p = cur_p;
    *q = cur_q;
  
    return 0;
}

int32_t metering_func_get_cur_v_i_ph(double *v, double *i, uint32_t ph)
{
    switch (ph)
    {
    case 0:
        *v = U_RMS_Ph1;
        *i = I_RMS_Ph1;
        break;
    case 1:
        *v = U_RMS_Ph2;
        *i = I_RMS_Ph2;
        break;
    case 2:
        *v = U_RMS_Ph3;
        *i = I_RMS_Ph3;
        break;
    case 3:
        *v = 0;
        *i = 0;
        break;
    default:
        return -1;
        break;
    }
  
    return 0;
}

int32_t metering_func_get_cur_p_q_s_ph(double *p, double *q, double *s, uint32_t ph)
{
    switch (ph)
    {
    case 0:
        *p = P_Ph1;
        *q = Q_Ph1;
        *s = S_Ph1;
        break;
    case 1:
        *p = P_Ph2;
        *q = Q_Ph2;
        *s = S_Ph2;
        break;
    case 2:
        *p = P_Ph3;
        *q = Q_Ph3;
        *s = S_Ph3;
        break;
    case 3:
        *p = 0;
        *q = 0;
        *s = 0;
        break;
    default:
        return -1;
        break;
    }
  
    return 0;
}

int32_t metering_func_stop_metering(void)
{
    metering_status_flag = METERING_STATUS_STOPPING;

    return 0;
}

tMETERING_Status_Flag metering_func_get_status(void)
{
    return metering_status_flag;
}

int32_t metering_func_set_hardfault_flag(void)
{
    metering_runtime_data.hardfault = 1;
  
    return 0;
}

int32_t metering_func_init(void)
{
    /* check validity of non-initialized variables in system ram and if data    */
    /* are not valid use default data stored in flash                           */
    if (metering_runtime_data.flag != POWER_METERING_RUNTIME_DATA_VALID_FLAG)
        memcpy((void *)&metering_runtime_data, (void *)&metering_runtime_initdata, sizeof(tMETERING_RUNTIME_DATA));
  
#if POWER_METERING_ENABLE_LOAD_CALIB_DATA
    /* 
    Read application setting from flash - if flash doesn't contain valid
    data then default data are saved into flash.
    */
    if (nvm_config_read((void *)&metering_cfg_data, sizeof(tMETERING_CONFIG_FLASH_DATA)))
    {
        metering_cfg_data.calib_flag = CALI_FLAG_INIT;
        nvm_config_write((void *)&metering_cfg_data, sizeof(tMETERING_CONFIG_FLASH_DATA));
    }
#endif
    pcalib_data_ph1 = (tMETERING_Calib_Data_1PH *)&metering_cfg_data.calib_data1;
    pcalib_data_ph2 = (tMETERING_Calib_Data_1PH *)&metering_cfg_data.calib_data2;
    pcalib_data_ph3 = (tMETERING_Calib_Data_1PH *)&metering_cfg_data.calib_data3;
#if POWER_METERING_USE_AFE_TRIGGER_ADC
    phase_shift[0] = pcalib_data_ph1->angle;
    phase_shift[1] = pcalib_data_ph2->angle;
    phase_shift[2] = pcalib_data_ph3->angle;
#endif
  
    /* Display calibration fin icon */
#if POWER_METERING_SLCD_SUPPORT
    if (CALI_FLAG_FIN == metering_cfg_data.calib_flag)
        slcd_disp_show_cali_fin();
#endif

    metering_func_hardware_init();
  
    return 0;
}

/******************************************************************************
* End of module                                                              *
******************************************************************************/
