/******************************************************************************
* (c) Copyright 2010-2015, Freescale Semiconductor Inc.
* ALL RIGHTS RESERVED.
***************************************************************************/
#include "common.h"
#include "metering_modules.h"
#include "drivers.h"
#include <string.h>
#include <stdio.h>

/*
 This file implements key handler functions
 To enhance ESD robust, we use the mechnism to check key status in 4ms iRTC to
 ensure that a key is pressed.
*/

static int8_t btn_sw3_pressed = 0;
static int8_t btn_sw4_pressed = 0;
static int8_t btn_sw5_pressed = 0;
static int8_t battery_pin_high = 0;
static int8_t battery_pin_low = 0;
static int8_t smart_card_plug_in = 0;

#define KEY_HANDLER_CHECK_KEY_PRESSING(condition, counter, rtn_val)     \
    {                                                                   \
        if (counter && (!condition()))                                  \
        {                                                               \
            counter = 0;                                                \
            rtn_val = 1;                                                \
        }                                                               \
    }

#define KEY_HANDLER_CHECK_KEY_EVT(condition, counter, evt_func)         \
    {                                                                   \
        if (counter)                                                    \
        {                                                               \
            if (condition())                                            \
            {                                                           \
                counter = 0;                                            \
                evt_func();                                             \
            }                                                           \
            else                                                        \
                counter = 0;                                            \
        }                                                               \
    }

static int32_t key_handler_sw4_press_evt(void)
{
    if (SMC_MODE_RUN == SMC_GetMode())
    {
        /* SW4 will change display item in number area 1 */
        slcd_disp_change_num1_area((tSLCD_Disp_Meaning)(slcd_disp_get_num1_area() + 1));
    }
    else
    {
        int32_t disp_item = slcd_disp_get_num1_area();
        switch (disp_item)
        {
        case SLCD_DISP_VOL_CUR_PH1:
            disp_item = SLCD_DISP_VOL_CUR_PH2;
            break;
        case SLCD_DISP_VOL_CUR_PH2:
            disp_item = SLCD_DISP_VOL_CUR_PH3;
            break;
        case SLCD_DISP_VOL_CUR_PH3:
        default:
            disp_item = SLCD_DISP_VOL_CUR_PH1;
            break;
        }
        slcd_disp_change_num1_area((tSLCD_Disp_Meaning)disp_item);
    }

    return 0;
}

static int32_t key_handler_sw5_press_evt(void)
{
#if 0 
     if (SMC_MODE_RUN == SMC_GetMode())
     {
         /* Enter VLPR */
         IRTC_EnableIsr(IRTC_MIN_MASK);
         work_mode_set_enter_vlpr_flag(1);
         work_mode_set_enter_vlps_flag(1);
     }
     else
     {
         /* Back to normal */
         IRTC_DisableIsr(IRTC_MIN_MASK);
         work_mode_set_exit_vlpr_flag(1);
     }
#endif

    return 0;
}

static int32_t key_handler_smart_card_plug_in_evt(void)
{
#if POWER_METERING_SMART_CARD_SUPPORT
    smart_card_set_activate_flag(1);
#endif
    return 0;
}

static int32_t key_handler_smart_card_plug_out_evt(void)
{
#if POWER_METERING_SMART_CARD_SUPPORT
    smart_card_deactivate();
#endif
    return 0;
}

static int32_t key_handler_sw3_press_evt(void)
{
    if (!SLCD_BACKLIGHT_STATUS())
        slcd_disp_backlight_ctrl(1);
    else
        slcd_disp_backlight_ctrl(0);

    return 0;
}

static int32_t key_handler_battery_pin_high_evt(void)
{
    /* Back to normal */
    IRTC_DisableIsr(IRTC_MIN_MASK);
    work_mode_exit_vlpr();

    return 0;
}

static int32_t key_handler_battery_pin_low_evt(void)
{
    /* Enter VLPR */
    IRTC_EnableIsr(IRTC_MIN_MASK);
    work_mode_set_enter_vlpr_flag(1);
    work_mode_set_enter_vlps_flag(1);

    return 0;
}

static void key_handler_pin_evt(PORT_CALLBACK_SRC src, uint8 pin)
{
    if (src == PORTF_CALLBACK)
    {
        if (pin == KEY_SW4_PIN_MASK)
        {
            btn_sw4_pressed = 1;
            IRTC_EnableIsr(IRTC_128HZ_MASK);
        }
        else if (pin == KEY_SW5_PIN_MASK)
        {
            btn_sw5_pressed = 1;
            IRTC_EnableIsr(IRTC_128HZ_MASK);
        }
        else if (pin == SMART_CARD_DETECT_PIN)
        {
            if (SMART_CARD_DETECTED())
            {
                smart_card_plug_in = 1;
                IRTC_EnableIsr(IRTC_128HZ_MASK);
            }
            else
            {
                key_handler_smart_card_plug_out_evt();
            }
        }
    }
    else if (src == PORTE_CALLBACK)
    {
        if (pin == KEY_SW3_PIN_MASK)
        {
            btn_sw3_pressed = 1;
            IRTC_EnableIsr(IRTC_128HZ_MASK);
        }
    }
    else if (src == PORTD_CALLBACK)
    {
        if (pin == BATTERY_PIN_MASK) 
        {
#if (0 == POWER_METERING_ENABLE_ESD_TEST_MODE)
    #if POWER_METERING_ENABLE_BATTERY_LOW_POWER
            if (BATTERY_PIN_STATUS())
                battery_pin_high = 1;
            else
                battery_pin_low = 1;
            IRTC_EnableIsr(IRTC_128HZ_MASK);
    #endif
#endif
        }
    }
}

int32_t key_handler_key_pressing_check(void)
{
    int32_t rtn = 0;

    KEY_HANDLER_CHECK_KEY_PRESSING(KEY_SW5_PRESSING, btn_sw5_pressed, rtn);

    KEY_HANDLER_CHECK_KEY_PRESSING(KEY_SW4_PRESSING, btn_sw4_pressed, rtn);

    KEY_HANDLER_CHECK_KEY_PRESSING(KEY_SW3_PRESSING, btn_sw3_pressed, rtn);

    if (rtn)
        IRTC_DisableIsr(IRTC_128HZ_MASK);

    return rtn;
}

int32_t key_handler_key_press_evt(void)
{
    KEY_HANDLER_CHECK_KEY_EVT(KEY_SW5_PRESSING, btn_sw5_pressed, key_handler_sw5_press_evt);

    KEY_HANDLER_CHECK_KEY_EVT(KEY_SW4_PRESSING, btn_sw4_pressed, key_handler_sw4_press_evt);

    KEY_HANDLER_CHECK_KEY_EVT(KEY_SW3_PRESSING, btn_sw3_pressed, key_handler_sw3_press_evt);

    return 0;
}

int32_t key_handler_key_pressed(void)
{
    return (btn_sw3_pressed + btn_sw4_pressed + btn_sw5_pressed);
}

int32_t key_handler_pin_pulling_check(void)
{
    int32_t rtn = 0;

    KEY_HANDLER_CHECK_KEY_PRESSING(!BATTERY_PIN_STATUS, battery_pin_low, rtn);

    KEY_HANDLER_CHECK_KEY_PRESSING(BATTERY_PIN_STATUS, battery_pin_high, rtn);

    KEY_HANDLER_CHECK_KEY_PRESSING(SMART_CARD_DETECTED, smart_card_plug_in, rtn);

    return rtn;
}

int32_t key_handler_pin_pull_detected(void)
{
    return (battery_pin_high + battery_pin_low + smart_card_plug_in);
}

int32_t key_handler_pin_pull_evt(void)
{
    KEY_HANDLER_CHECK_KEY_EVT(!BATTERY_PIN_STATUS, battery_pin_low, key_handler_battery_pin_low_evt);

    KEY_HANDLER_CHECK_KEY_EVT(BATTERY_PIN_STATUS, battery_pin_high, key_handler_battery_pin_high_evt);

    KEY_HANDLER_CHECK_KEY_EVT(SMART_CARD_DETECTED, smart_card_plug_in, key_handler_smart_card_plug_in_evt);

    return 0;
}

int32_t key_handler_init(void)
{
    /* SW4 */
    KEY_SW4_PIN_INIT();
    /* SW5 */
    KEY_SW5_PIN_INIT();
    /* SW3 */
    KEY_SW3_PIN_INIT();
    /* Battery */
    BATTERY_PIN_INIT();

    /* Card detector Pin, PTF0 */
    SMART_CARD_DETECT_PIN_INIT();

    PORT_InstallCallback(PRI_LVL3, key_handler_pin_evt);

    return 0;
}

/******************************************************************************
* End of module                                                              *
******************************************************************************/
