/*
 * Comms.c
 *
 *  Created on: Mar 13, 2013
 *      Author: B34981
 *      
 *  SCI Communications Driver for MagniV Bootloader 
 *  IP Versoin: SCIV5
 *  Channel: 1 (SCI1)
 *  BaudRate: 19200
 *  
 */


/**********************************************************************************************
 * Includes
 **********************************************************************************************/
#include "Comms.h"

/**********************************************************************************************
 * External objects
 **********************************************************************************************/

/**********************************************************************************************
 * Local function prototypes
 *********************************************************************************************/
void SCI_T_Byte(UINT8 u8char);


/**********************************************************************************************
 * Global variables
 **********************************************************************************************/
static UINT8 u8LIN_Byte_Counter = 0;
static UINT8 u8phrase_counter = 0;
static LIN_FRAME t_LIN_frame;
extern BootPhraseStruct BP;
extern UINT8 u8BootPhraseRcvd;

/**********************************************************************************************
 * Constants and macros
 **********************************************************************************************/
#define BAUD_RATE_VALUE		32000000/BAUD_RATE

/**********************************************************************************************
 * Local types
 **********************************************************************************************/


/**********************************************************************************************
 * Local variables
 **********************************************************************************************/


/**********************************************************************************************
 * Local functions
 **********************************************************************************************/


/**********************************************************************************************
 * Global functions
 **********************************************************************************************/

/***********************************************************************************************
 *
 * @brief    Comms_Init - Initialize the serial communications driver. Designed for SCI0 - LIN phy
 * @param    none
 * @return   none
 *
 ************************************************************************************************/  	
void Comms_Init()
{	
	/* SCI 0 initialization */
	MODRR0_S0L0RR = 0;			/* routing of LIN signal to Rx/Tx */
	SCI0BD = BAUD_RATE_VALUE;				/* baud rate setting */
	SCI0CR2_TE = 1;				/* Enable Transmitter */
	SCI0CR2_RE = 1;				/* Enable Transmitter */
	SCI0SR2_BRK13 = 1;			/* Break character is 13 bit long */

	SCI0SR2_AMAP = 1; /* alternative register are available */
	SCI0ACR2_BKDFE = 1; /* Break detect enabled */

	/* LIN physical layer */
	LP0CR_LPPUE = 1;    /* Pull up to strong signal  */
	LP0SLRM = 0x01;		/* LIN Slew Rate selection */
	/* With the LIN baudrate selected as 9600bps, 
	    the Slew Rate is set as 01 which is optimized for 10,400 Bit/s */
	LP0CR_LPE = 1;      /* Enable LIN Phy  */

	/* intialize LIN frame structure */
	t_LIN_frame = (LIN_FRAME){
		NOT_DETECTED,
		0x00,
		0x00,
		{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},
		0x00
	};



}


/***********************************************************************************************
 *
 * @brief    Comms_Tx_Ack - Reply to received Boot Phrases depending on checksum results
 * @param    ACK_OPTS ack: OK or CRC_ERROR
 * @return   none
 *
 ************************************************************************************************/  	
void Comms_Tx_Ack(enum ACK_OPTS ack)
{
	BREAK_DETECT_FLAG = 1;			/* clear break flag */
	CLEAR_SCI_STATUS_FLAG;		/* clear SCI statuts registers */
	SCI0ACR2_BKDFE = 1;			/* Enable Break circuit */
	SCI0CR2_RE = 1;				/* Enable Receiver */
	while(BREAK_DETECT_FLAG==NOT_DETECTED){}	/* wait for break detection */
	t_LIN_frame.LIN_break_detect = DETECTED;
	CLEAR_SCI_STATUS_FLAG;		/* clear SCI statuts registers */
	t_LIN_frame.LIN_sync_field = Comms_Rx_Data_Polling();
	if(t_LIN_frame.LIN_sync_field==LIN_SYNC_FIELD_VALUE){
		t_LIN_frame.LIN_PID = Comms_Rx_Data_Polling();
		ACTIVITY_LED = ~ACTIVITY_LED;
		if((t_LIN_frame.LIN_PID&LIN_PID_MASK) == BOOTLOADER_RECEIVE_FRAME_ID){
			SCI_T_Byte((UINT8)ack);
			t_LIN_frame.checksum = ~((UINT8)ack+t_LIN_frame.LIN_PID); /* calculate enhanced checksum */
			SCI_T_Byte(t_LIN_frame.checksum);	/* send CRC from slave */
		}
	}
}
//}
/***********************************************************************************************
 *
 * @brief    Comms_Reset_Reg - Reset the modified registers to its defaut values
 * @param    none
 * @return   none
 *
 ************************************************************************************************/  	
void Comms_Reset_Reg()
{
	SCI0ACR2_BKDFE = 0; /* break detect circuit disabled */
	SCI0SR2_AMAP = 0; /* alternative register are disable */
	SCI0BDH = 0x00;	/* set to reset value all control registers */
	SCI0BDL = 0x00;
	SCI0CR1 = 0x00;
	SCI0CR2 = 0x00;
	LP0CR_LPE = 0;      /* disable LIN Phy  */
}


/***********************************************************************************************
 *
 * @brief    SCI_T_Byte - Transmit a byte through the SCI interface
 * @param    UINT8 u8char - Character to transmit
 * @return   none
 *
 ************************************************************************************************/
void SCI_T_Byte(UINT8 u8char)
{
	SCI0DRL = u8char;			/*The character to be sent is stored into the SCI data register*/

	while(!SCI0SR1_TC)			/*Dummy while. Waits until Transmission is completed*/
	{
	}
}


/***********************************************************************************************
 *
 * @brief    Comms_Rx_Data - Poll or Interrupt that receives LIN frame.
 * @param    none
 * @return   none (Data must fill the BP structure, when BP is fill, update u8BootPhraseRcvd)
 *
 ************************************************************************************************/ 
void Comms_Rx_Data()
{		
	UINT8 u8counter;
	t_LIN_frame.LIN_break_detect = BREAK_DETECT_FLAG;
	if(t_LIN_frame.LIN_break_detect == DETECTED){ 		 /* break signal detected */
		SCI0ACR2_BKDFE = 0; /* Break detect disable while receiving LIN frame data */
		CLEAR_SCI_STATUS_FLAG;
		t_LIN_frame.LIN_break_detect = DETECTED;
		t_LIN_frame.LIN_sync_field = Comms_Rx_Data_Polling();
		if(t_LIN_frame.LIN_sync_field==LIN_SYNC_FIELD_VALUE){
			t_LIN_frame.LIN_PID = Comms_Rx_Data_Polling();
			if((t_LIN_frame.LIN_PID&LIN_PID_MASK) == BOOTLOADER_TRANSMIT_FRAME_ID){
				/* data reception */
				if(u8LIN_Byte_Counter == 0)			/* If first array byte */				
				{
					u8phrase_counter =  Comms_Rx_Data_Polling(); /* first byte indicates the lenght of the S-record file phrase */
					t_LIN_frame.LIN_data[0] = u8phrase_counter;
					BP.Byte[0] = u8phrase_counter;
					u8phrase_counter++;				/* Advance S-record phrase counter */
					u8LIN_Byte_Counter++;			/* Advance boot data the array pointer by one */	
				}
				if(u8phrase_counter>LIN_MAX_DATA_BYTES){				/* if phrase have more than 8 bytes (maximum number of data bytes in LIN frame). Fill whole frame */
					if(u8LIN_Byte_Counter==1){ /* In first LIN frame first byte has already been received */
						for(u8counter=1;u8counter<LIN_MAX_DATA_BYTES;u8counter++){	
							t_LIN_frame.LIN_data[u8counter]= Comms_Rx_Data_Polling(); /* fill control LIN frame structure */
							BP.Byte[u8LIN_Byte_Counter++] = t_LIN_frame.LIN_data[u8counter];	/* fill flash control structure */
						}
					}
					else{ /* fill whole LIN frame */
						for(u8counter=0;u8counter<LIN_MAX_DATA_BYTES;u8counter++){
							t_LIN_frame.LIN_data[u8counter]= Comms_Rx_Data_Polling();	/* fill control LIN frame structure */
							BP.Byte[u8LIN_Byte_Counter++] = t_LIN_frame.LIN_data[u8counter];	/* fill flash control structure */
						}
					}
					u8phrase_counter-=LIN_MAX_DATA_BYTES;	/* 8 bytes of data where received */
				}
				else{ 	/* S-record phrase is no longer bigger than 8 bytes */
					for(u8counter=0;u8counter<u8phrase_counter;u8counter++){
						t_LIN_frame.LIN_data[u8counter]= Comms_Rx_Data_Polling();	/* fill control LIN frame structure */
						BP.Byte[u8LIN_Byte_Counter++] = t_LIN_frame.LIN_data[u8counter];	/* fill flash control structure */
					}
					u8phrase_counter=0;					/* Clear S-record phrase byte's lenght counter */
					u8LIN_Byte_Counter = 0;				/* Reset byte counter */
					u8BootPhraseRcvd = 1;				/* Indicate main application that a Boot Phrase was received */

				}
			}
		}

	}
	SCI0ACR2_BKDFE = 1; /* Break detect enabled */
}



/***********************************************************************************************
 *
 * @brief    SCI_T_Byte - Transmit a byte through the SCI interface
 * @param    UINT8 u8char - Character to transmit
 * @return   none
 *
 ************************************************************************************************/

unsigned char Comms_Rx_Data_Polling(void){
	while(SCI0SR1_RDRF==0){}
	(void)SCI0SR1;						/* Clear Rx Flag */	
	return SCI0DRL;	
}
