/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GMCLIB_SvmStd.c
*
* @author   B04459
*
* @version  1.0.13.0
*
* @date     Apr-26-2010
*
* @brief    Source file for #GMCLIB_SvmStd function.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GMCLIB_GROUP
    @addtogroup GMCLIB_GROUP
@else
    @defgroup GMCLIB_GROUP   GMCLIB
@endif
*/

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GMCLIB_SvmStd.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-exported)
-----------------------------------------------------------------------------*/

/**************************************************************************//*!
@brief      Function alias #GMCLIB_SvmStd.
            Fucntion for calculation of duty-cycle ratios using Standard Space
            Vector Modulation technique.

@param[in]  *pIn Pointer to structure containing direct \f$ U_{\alpha}
             \f$ and quadrature \f$ U_{\beta}\f$ components of the stator
             voltage vector
@param[out] *pOut Pointer to structure containing calculated duty-cycle ratios
            of the 3-Phase system

@return     The function returns 32-bit value in format INT, representing the
			actual space sector which contains the stator reference vector U_s.

@details    The #GMCLIB_SvmStdANSIC function, denoting ANSI-C compatible
            implementation, can be called via function alias
            #GMCLIB_SvmStd.

            \anchor fig1_GMCLIB_SvmStd
            \image latex SvmFigure1.eps "Power Stage Schematic Diagram" width=14cm
            \image html SvmFigure1.jpg "Power Stage Schematic Diagram"

            \par
            The SVMstd function for calculating duty-cycle ratios is
            widely-used in the modern electric drives. This function calculates
            appropriate duty-cycle ratios, which are needed for generating the
            given stator reference voltage vector using a special Space Vector
            Modulation technique, termed Standard Space Vector Modulation.
            The basic principle of the Standard Space Vector Modulation
            Technique can be explained with the help of the power stage diagram
            in Fig. \ref fig1_GMCLIB_SvmStd.

            Top and bottom switches are working in a complementary mode; i.e.,
            if the top switch, \f$ S_{At} \f$, is ON, then the corresponding
            bottom switch, \f$ S_{Ab} \f$, is OFF and vice versa. Considering
            that value 1 is assigned to the ON state of the top switch and
            value 0 is assigned to the ON state of the bottom switch, the
            switching vector, \f$ [a, b, c]^T \f$, can be defined. Creating such
            a vector allows numerical definition of all possible switching
            states. In a three-phase power stage configuration (as shown in Fig.
            \ref fig1_GMCLIB_SvmStd), eight possible switching states (detailed
            in Fig. \ref fig2_GMCLIB_SvmStd) are feasible.

            \anchor fig2_GMCLIB_SvmStd
            \image latex SvmFigure2.eps "Basic Space Vectors" width=10cm
            \image html SvmFigure2.jpg "Basic Space Vectors"

            These states, together
            with the resulting instantaneous output line-to-line and phase
            voltages, are listed in Table \ref tab1_GMCLIB_SvmStd.

            \anchor tab1_GMCLIB_SvmStd
            <table ALIGN=CENTER border="1">
            <CAPTION ALIGN=TOP>Switching Patterns</CAPTION>
            <tr>
              <th>\f$a\f$</th>
              <th>\f$b\f$</th>
              <th>\f$c\f$</th>
              <th>\f$U_a\f$</th>
              <th>\f$U_b\f$</th>
              <th>\f$U_c\f$</th>
              <th>\f$U_{AB}\f$</th>
              <th>\f$U_{BC}\f$</th>
              <th>\f$U_{CA}\f$</th>
              <th>Vector</th>
            </tr>
            <tr>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$O_{000}\f$</td>
            </tr>
            <tr>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$\frac{2}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$-U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$U_{0}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-\frac{2}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$U_{60}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$-\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$\frac{2}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$U_{120}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$-\frac{2}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$U_{240}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$-\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$\frac{2}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$-U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$U_{300}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-\frac{2}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$\frac{1}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$-U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$U_{360}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td>\f$1\f$</td>
              <td>\f$1\f$</td>
              <td>\f$1\f$</td>
              <td>\f$0\f$</td>
              <td>\f$0\f$</td>
              <td>\f$0\f$</td>
              <td>\f$0\f$</td>
              <td>\f$0\f$</td>
              <td>\f$0\f$</td>
              <td>\f$O_{111}\f$</td>
            </tr>
            </table>

            The quantities of the direct-\f$U_\alpha\f$ and the
            quadrature-\f$U_\beta\f$ components of the two-phase orthogonal
            coordinate system, describing the three-phase stator voltages, are
            expressed by the Clarke Transformation.

            \anchor eq1_SvmStd
            \f[
            U_{\alpha} = \frac{2}{3} \left(U_a-\frac{U_b}{2}-\frac{U_c}{2}\right)
            \f]

            \anchor eq2_SvmStd
            \f[
            U_{\beta} = \frac{2}{3} \left(0 +\frac{\sqrt{3}U_b}{2}-
            \frac{\sqrt{3}U_c}{2}\right)
            \f]

            The three-phase stator voltages,\f$ U_a \f$, \f$ U_b \f$, and
            \f$ U_c\f$, are transformed using Clarke Transformation into the
            \f$ U_\alpha \f$ and the \f$ U_\beta \f$ components of the
            two-phase orthogonal coordinate system. The transformation results
            are listed in Table \ref tab2_GMCLIB_SvmStd.

            \anchor tab2_GMCLIB_SvmStd
            <table ALIGN=CENTER border="1">
            <CAPTION ALIGN=TOP>Switching Patterns and Space Vectors</CAPTION>
            <tr>
              <th>\f$a\f$</th>
              <th>\f$b\f$</th>
              <th>\f$c\f$</th>
              <th>\f$U_\alpha\f$</th>
              <th>\f$U_\beta\f$</th>
              <th>Vector</th>
            </tr>
            <tr>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$O_{000}\f$</td>
            </tr>
            <tr>
              <td ALIGN=CENTER>\f$1\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$\frac{2}{3}U_{DCBus}\f$</td>
              <td ALIGN=CENTER>\f$0\f$</td>
              <td ALIGN=CENTER>\f$U_0\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td>\f$1\f$</td>
              <td>\f$1\f$</td>
              <td>\f$0\f$</td>
              <td>\f$\frac{1}{3}U_{DCBus}\f$</td>
              <td>\f$\frac{1}{\sqrt{3}}U_{DCBus}\f$</td>
              <td>\f$U_{60}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td>\f$0\f$</td>
              <td>\f$1\f$</td>
              <td>\f$0\f$</td>
              <td>\f$-\frac{1}{3}U_{DCBus}\f$</td>
              <td>\f$\frac{1}{\sqrt{3}}U_{DCBus}\f$</td>
              <td>\f$U_{120}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td>\f$0\f$</td>
              <td>\f$1\f$</td>
              <td>\f$1\f$</td>
              <td>\f$-\frac{2}{3}U_{DCBus}\f$</td>
              <td>\f$0\f$</td>
              <td>\f$U_{240}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td>\f$0\f$</td>
              <td>\f$0\f$</td>
              <td>\f$1\f$</td>
              <td>\f$-\frac{1}{3}U_{DCBus}\f$</td>
              <td>\f$-\frac{1}{\sqrt{3}}U_{DCBus}\f$</td>
              <td>\f$U_{300}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td>\f$1\f$</td>
              <td>\f$0\f$</td>
              <td>\f$1\f$</td>
              <td>\f$\frac{1}{3}U_{DCBus}\f$</td>
              <td>\f$-\frac{1}{\sqrt{3}}U_{DCBus}\f$</td>
              <td>\f$U_{360}\f$</td>
            </tr>
            <tr ALIGN=CENTER>
              <td>\f$1\f$</td>
              <td>\f$1\f$</td>
              <td>\f$1\f$</td>
              <td>\f$0\f$</td>
              <td>\f$0\f$</td>
              <td>\f$O_{111}\f$</td>
            </tr>
            </table>

            Fig. \ref fig2_GMCLIB_SvmStd graphically depicts some feasible basic
            switching states (vectors). It is clear that there are six non-zero
            vectors \f$ U_0, U_{60}, U_{120}, U_{180}, U_{240}, U_{300}\f$
            and two zero vectors \f$ O_{111}, O_{000}\f$, usable for switching.
            Therefore, the principle of the Standard Space Vector Modulation
            resides in applying appropriate switching states for a certain time
            and thus generating a voltage vector identical to the reference one.

            \anchor fig3_GMCLIB_SvmStd
            \image latex SvmFigure3.eps "Projection of Reference Voltage Vector in Sector I" width=8cm
            \image html SvmFigure3.jpg "Projection of Reference Voltage Vector in Sector I"

            Referring to that principle, an objective of the Standard Space
            Vector Modulation is an approximation of the reference stator
            voltage vector \f$ U_S \f$ with an appropriate combination of the
            switching patterns composed of basic space vectors. The graphical
            explanation of this objective is shown in \ref fig3_GMCLIB_SvmStd
            and \ref fig4_GMCLIB_SvmStd.

            \anchor fig4_GMCLIB_SvmStd
            \image latex SvmFigure4.eps "Detail of the Voltage Vector Projection in Sector I" width=8cm
            \image html SvmFigure4.jpg "Detail of the Voltage Vector Projection in Sector I"

            The stator reference voltage vector \f$ U_S \f$ is phase-advanced by
            \f$ 30^\circ \f$ from the axis-\f$ \alpha \f$ and thus might be
            generated with an appropriate combination of the adjacent basic
            switching states \f$ U_0 \f$ and \f$ U_{60}\f$. These figures also
            indicate resultant \f$ U_\alpha \f$ and \f$ U_\beta \f$
            components for space vectors \f$ U_0 \f$ and \f$ U_{60} \f$

            In this case, the reference stator voltage vector \f$ U_S \f$ is
            located in Sector I and, as previously mentioned, can be generated
            with the appropriate duty-cycle ratios of the basic switching states
            \f$ U_{60} \f$ and \f$ U_0 \f$. The principal equations concerning
            this vector location are:

            \anchor eq3_SvmStd
            \f[
            T = T_{60} + T_0 + T_{null}
            \f]

            \anchor eq4_SvmStd
            \f[
            U_s = \frac{T_{60}}{T} \cdot U_{60}+\frac{T_0}{T} \cdot U_0
            \f]

            where \f$ T_{60} \f$ and \f$ T_0 \f$ are the respective duty-cycle
            ratios for which the basic space vectors \f$ U_{60} \f$ and
            \f$ U_0 \f$ should be applied within the time period T. T \f$_{null}
            \f$ is the course of time for which the null vectors \f$ O_{000} \f$
            and \f$ O_{111} \f$ are applied. Those duty-cycle ratios can be
            calculated using equations:

            \anchor eq5_SvmStd
            \f[
            u_{\beta}=\frac{T_{60}}{T} \ cdot \left| U_{60} \right| \cdot
            \sin 60^\circ
            \f]

            \anchor eq6_SvmStd
            \f[
            u_\alpha = \frac{T_0}{T} \cdot \left| U_0 \right| + \frac{u_\beta}
            {\tan 60^\circ}
            \f]

            Considering that normalized magnitudes of basic space vectors are
            \f$\left| U_{60}\right| = \left| U_0 \right|= \frac {2}{\sqrt{3}}\f$
            and by substitution of the trigonometric expressions
            \f$ \sin 60^\circ \f$ and \f$ \tan 60^\circ \f$ by their quantities
            \f$ \frac{2}{\sqrt{3}}\f$ and \f$ \sqrt{3}\f$, respectively, eq.
            \ref eq5_SvmStd and eq.\ref eq6_SvmStd can be rearranged for the
            unknown duty-cycle ratios \f$ \frac{T_{60}}{T} \f$ and \f$
            \frac{T_0}{T} \f$:

            \anchor eq7_SvmStd
            \f[
            \frac{T_{60}}{T} = u_\beta
            \f]

            \anchor eq8_SvmStd
            \f[
            U_S = \frac{T_{120}}{T} \cdot U_{120}+\frac{T_{60}}{T} \cdot U_{60}
            \f]

            \anchor fig5_GMCLIB_SvmStd
            \image latex SvmFigure5.eps "Projection of the Reference Voltage Vector in Sector II" width=8cm
            \image html SvmFigure5.jpg "Projection of the Reference Voltage Vector in Sector II"

            Sector II is depicted in \ref fig5_GMCLIB_SvmStd. In this particular
            case, the reference stator voltage vector \f$ U_S \f$ is generated
            by the appropriate duty-cycle ratios of the basic switching states
            \f$ U_{60} \f$ and \f$ U_{120} \f$. The basic equations describing
            this sector are:

            \anchor eq9_SvmStd
            \f[
            T = T_{120} + T_{60} + T{null}
            \f]

            \anchor eq10_SvmStd
            \f[
            U_s = \frac{T_{120}}{T}*U_{120}+\frac{T_{60}}{T}*U_{60}
            \f]

            where \f$ T_{120} \f$ and \f$ T_{60} \f$ are the respective
            duty-cycle ratios for which the basic space vectors \f$ U_{120} \f$
            and \f$ U_{60} \f$ should be applied within the time period T.
            These resultant duty-cycle ratios are formed from the auxiliary
            components termed A and B. The graphical representation of the
            auxiliary components is shown in \ref fig6_GMCLIB_SvmStd.

            \anchor fig6_GMCLIB_SvmStd
            \image latex SvmFigure6.eps "Detail of the Voltage Vector Projection in Sector II" width=8cm
            \image html SvmFigure6.jpg "Detail of the Voltage Vector Projection in Sector II"

            The equations describing those auxiliary time-duration components
            are:

            \anchor eq11_SvmStd
            \f[
            \frac{\sin30^\circ}{\sin120^\circ} = \frac{A}{u_\beta}
            \f]

            \anchor eq12_SvmStd
            \f[
            \frac{\sin60^\circ}{\sin60^\circ} = \frac{B}{u_\alpha}
            \f]

            Eq.\ref eq10_SvmStd and eq.\ref eq11_SvmStd have been formed using the
            sine rule. These equations can be rearranged for the calculation of
            the auxiliary time-duration components A and B. This is done simply
            by substitution of the trigonometric terms
            \f$ \sin 30^\circ, \sin 120^\circ \f$ and \f$ \sin 60^\circ \f$
            by their numerical representations \f$ \frac{1}{2} \f$,
            \f$ \frac{\sqrt{3}}{2}\f$ and \f$\frac{1}{\sqrt{3}}\f$, respectively.

            \anchor eq13_SvmStd
            \f[
            A = \frac{1}{\sqrt{3}} \cdot u_\beta
            \f]

            \anchor eq14_SvmStd
            \f[
            B = u_\alpha
            \f]

            The resultant duty-cycle ratios, \f$ \frac{T_{120}}{T} \f$ and
            \f$ \frac{T_{60}}{T} \f$, are then expressed in terms of the
            auxiliary time-duration components defined by eq.\ref eq14_SvmStd and
            eq.\ref eq15_SvmStd, as follows:

            \anchor eq15_SvmStd
            \f[
            \frac{T_{120}}{T} \cdot \left| U_{120} \right| = A - B
            \f]

            \anchor eq16_SvmStd
            \f[
            \frac{T_{60}}{T} \cdot \left| U_{60} \right| = A + B
            \f]

            With the help of these equations and also considering the normalized
            magnitudes of the basic space vectors to be
            \f$ \left| U_{120} \right| = \left| U_{60} \right| = \frac{2}{\sqrt{3}} \f$,
            the equations expressed for the unknown duty-cycle ratios of basic
            space vectors \f$ \frac{T_{120}}{T} \f$ and
            \f$ \frac{T_{60}}{T} \f$ can be written:

            \anchor eq17_SvmStd
            \f[
            \frac{T_{120}}{T} = \frac{1}{2} \left( u_\beta - \sqrt{3} \cdot
            u_\alpha \right)
            \f]

            \anchor eq18_SvmStd
            \f[
            \frac{T_{60}}{T} = \frac{1}{2} \left( u_\beta + \sqrt{3} \cdot
            u_\alpha \right)
            \f]

            The duty-cycle ratios in remaining sectors can be derived using the
            same approach. The resulting equations will be similar to those
            derived for Sector I and Sector II.

            To depict duty-cycle ratios of the basic space vectors for all
            sectors, we define:

            - Three auxiliary variables:

            \anchor eq19_SvmStd
            \f[
            X = u_\beta
            \f]

            \anchor eq20_SvmStd
            \f[
            Y = \frac{1}{2} \cdot \left( u_\beta + \sqrt{3} \cdot u_\alpha \right)
            \f]

            \anchor eq21_SvmStd
            \f[
            Z = \frac{1}{2} \cdot \left( u_\beta - \sqrt{3} \cdot u_\alpha \right)
            \f]

            Two expressions t_1 and t_2
            which generally represent duty-cycle ratios of the basic space
            vectors in the respective sector; e.g., for the first sector, t_1
            and t_2 represent duty-cycle ratios of the basic space vectors
            \f$ U_{60} \f$ and \f$ U_0 \f$; for the second sector, t_1 and t_2
            represent duty-cycle ratios of the basic space vectors \f$U_{120}\f$
            and \f$ U_{60} \f$, etc.

            For each sector, the expressions t_1 and t_2, in terms of auxiliary
            variables X, Y and Z, are listed in Table \ref tab3_GMCLIB_SvmStd.

            \anchor tab3_GMCLIB_SvmStd
            <table ALIGN=CENTER border="1" WIDTH="50%">
            <CAPTION ALIGN=TOP>Determination of t_1 and t_2 expressions</CAPTION>
            <tr>
              <th>Sector</th>
              <th>\f$U_{0}, U_{60}\f$</th>
              <th>\f$U_{60}, U_{120}\f$</th>
              <th>\f$U_{120}, U_{180}\f$</th>
              <th>\f$U_{180}, U_{240}\f$</th>
              <th>\f$U_{240}, U_{300}\f$</th>
              <th>\f$U_{300}, U_{0}\f$</th>
            </tr>
            <tr>
              <td ALIGN=CENTER>t_1</td>
              <td ALIGN=CENTER>X</td>
              <td ALIGN=CENTER>Y</td>
              <td ALIGN=CENTER>-Y</td>
              <td ALIGN=CENTER>Z</td>
              <td ALIGN=CENTER>-Z</td>
              <td ALIGN=CENTER>-X</td>
            </tr>
            <tr>
              <td ALIGN=CENTER>t_2</td>
              <td ALIGN=CENTER>-Z</td>
              <td ALIGN=CENTER>Z</td>
              <td ALIGN=CENTER>X</td>
              <td ALIGN=CENTER>-X</td>
              <td ALIGN=CENTER>-Y</td>
              <td ALIGN=CENTER>Y</td>
            </tr>
            </table>

            For the determination of auxiliary variables X eq.\ref eq19_SvmStd,
            Y eq.\ref eq20_SvmStd and Z eq.\ref eq21_SvmStd, the sector
            number is required. This information can be obtained by several
            approaches. One approach discussed here requires the use of modified
            Inverse Clark Transformation to transform the direct-\f$ \alpha \f$
            and quadrature-\f$ \beta \f$ components into a balanced three-phase
            quantity \f$ u_{ref1} \f$, \f$ u_{ref2} \f$ and \f$ u_{ref3} \f$,
            used for saightforward calculation of the sector number, to be shown
            later.

            \anchor eq22_SvmStd
            \f[
            u_{ref1} = u_\beta
            \f]

            \anchor eq23_SvmStd
            \f[
            u_{ref2} = \frac{1}{2} \cdot \left( -u_\beta + \sqrt{3} \cdot
            u_\alpha \right)
            \f]

            \anchor eq24_SvmStd
            \f[
            u_{ref3} = \frac{1}{2} \cdot \left( -u_\beta - \sqrt{3} * u_\alpha
            \right)
            \f]

            The modified Inverse Clark Transformation projects the
            quadrature-\f$ u_\beta \f$ component into \f$ u_{ref1} \f$, as shown
            in \ref fig7_GMCLIB_SvmStd and \ref fig8_GMCLIB_SvmStd, whereas
            voltages generated by the conventional Inverse Clark Transformation
            project the \f$ u_\alpha \f$ component into \f$ u_{ref1} \f$.

            \anchor fig7_GMCLIB_SvmStd
            \image latex SvmFigure7.eps "Direct-ua and Quadrature-ub Components of Stator Reference Voltage" width=10cm
            \image html SvmFigure7.jpg "Direct-ua and Quadrature-ub Components of Stator Reference Voltage"

            \ref fig7_GMCLIB_SvmStd depicts the \f$ u_\alpha \f$ and
            \f$ u_\beta \f$ components of the stator reference
            voltage vector \f$ U_S \f$ that were calculated by the equations
            \f$ u_\alpha = \cos \vartheta  \f$ and \f$ u_\beta  = \sin \vartheta \f$,
            respectively.

            \anchor fig8_GMCLIB_SvmStd
            \image latex SvmFigure8.eps "Reference Voltages u_ref1, u_ref2 and u_ref3" width=10cm
            \image html SvmFigure8.jpg "Reference Voltages u_ref1, u_ref2 and u_ref3"

            The Sector Identification Tree, shown in \ref fig9_GMCLIB_SvmStd,
            can be a numerical solution of the approach shown in
            \ref fig8_GMCLIB_SvmStd.

            \anchor fig9_GMCLIB_SvmStd
            \image latex SvmFigure9.eps "Identification of the Sector Number" width=10cm
            \image html SvmFigure9.jpg "Identification of the Sector Number"

            It should be pointed out that, in the worst case, three simple
            comparisons are required to precisely identify the sector of the
            stator reference voltage vector. For example, if the stator
            reference voltage vector resides according to the one shown in
            \ref fig3_GMCLIB_SvmStd, the stator reference voltage vector is
            phase-advanced by \f$ 30^\circ \f$ from the \f$ \alpha \f$-axis,
            which results in the positive quantities of \f$ u_{ref1} \f$ and
            \f$ u_{ref2} \f$ and the negative quantity of \f$ u_{ref3} \f$;
            refer to \ref fig8_GMCLIB_SvmStd. If these quantities are used as
            the inputs to the Sector Identification Tree, the product of those
            comparisons will be Sector I. Using the same approach identifies
            Sector II, if the stator reference voltage vector is located
            according to the one shown in \ref fig6_GMCLIB_SvmStd. The
            variables \f$ t_1 \f$, \f$ t_2 \f$ and \f$ t_3 \f$, representing
            switching duty-cycle ratios of the respective three-phase system,
            are given by the following equations:

            \anchor eq25_SvmStd
            \f[
            t_1 = \frac{T-t\_1-t\_2}{2}
            \f]

            \anchor eq26_SvmStd
            \f[
            t_2 = t_1 + t\_1
            \f]

            \anchor eq27_SvmStd
            \f[
            t_3 = t_2 + t\_2
            \f]

            where T is the switching period, t_1 and t_2 are the duty-cycle
            ratios (see table.3) of the basic space vectors, given for the
            respective sector eq.\ref eq21_SvmStd, eq.\ref eq22_SvmStd and
            eq.\ref eq23_SvmStd are specific solely to the Standard Space Vector
            Modulation technique; consequently, other Space Vector Modulation
            techniques discussed later will require deriving different equations.

            The next step is to assign the correct duty-cycle ratios, \f$t_1\f$,
            \f$ t_2 \f$ and \f$ t3 \f$, to the respective motor phases. This is
            a simple task, accomplished in view of the position of the stator
            reference voltage vector as shown in Table \ref tab4_GMCLIB_SvmStd.

            \anchor tab4_GMCLIB_SvmStd
            <table ALIGN=CENTER border="1" WIDTH="50%">
            <CAPTION ALIGN=TOP>Assignment of the Duty-Cycle Ratios to Motor Phases</CAPTION>
            <tr>
              <th>Sector</th>
              <th>\f$U_{0}, U_{60}\f$</th>
              <th>\f$U_{60}, U_{120}\f$</th>
              <th>\f$U_{120}, U_{180}\f$</th>
              <th>\f$U_{180}, U_{240}\f$</th>
              <th>\f$U_{240}, U_{300}\f$</th>
              <th>\f$U_{300}, U_{0}\f$</th>
            </tr>
            <tr>
              <td ALIGN=CENTER>pwm_a</td>
              <td ALIGN=CENTER>\f$t_3\f$</td>
              <td ALIGN=CENTER>\f$t_2\f$</td>
              <td ALIGN=CENTER>\f$t_1\f$</td>
              <td ALIGN=CENTER>\f$t_1\f$</td>
              <td ALIGN=CENTER>\f$t_2\f$</td>
              <td ALIGN=CENTER>\f$t_3\f$</td>
            </tr>
            <tr>
              <td ALIGN=CENTER>pwm_b</td>
              <td ALIGN=CENTER>\f$t_2\f$</td>
              <td ALIGN=CENTER>\f$t_3\f$</td>
              <td ALIGN=CENTER>\f$t_3\f$</td>
              <td ALIGN=CENTER>\f$t_2\f$</td>
              <td ALIGN=CENTER>\f$t_1\f$</td>
              <td ALIGN=CENTER>\f$t_1\f$</td>
            </tr>
            <tr>
              <td ALIGN=CENTER>pwm_b</td>
              <td ALIGN=CENTER>\f$t_1\f$</td>
              <td ALIGN=CENTER>\f$t_1\f$</td>
              <td ALIGN=CENTER>\f$t_2\f$</td>
              <td ALIGN=CENTER>\f$t_3\f$</td>
              <td ALIGN=CENTER>\f$t_3\f$</td>
              <td ALIGN=CENTER>\f$t_2\f$</td>
            </tr>
            </table>

            The principle of the Space Vector Modulation technique consists in
            applying the basic voltage vectors \f$ U_{XXX} \f$ and \f$O_{XXX}\f$
            for the certain time in such a way that the mean vector, generated
            by the Pulse Width Modulation approach for the period T, is equal to
            the original stator reference voltage vector \f$ U_S \f$. This
            provides a great variability of the arrangement of the basic vectors
            during the PWM period T. Those vectors might be arranged either to
            lower switching losses or to achieve diverse results, such as
            center-aligned PWM, edge-aligned PWM or a minimal number of
            switching states. A brief discussion of the widely-used
            center-aligned PWM follows.
            Generating the center-aligned PWM pattern is accomplished
            practically by comparing the threshold levels, pwm_a, pwm_b and
            pwm_c with a free-running up-down counter. The timer counts to 1
            (0x7FFF) and then down to 0 (0x0000). It is supposed that when a
            threshold level is larger than the timer value, the respective PWM
            output is active. Otherwise, it is inactive; see \ref fig10_GMCLIB_SvmStd

            \anchor fig10_GMCLIB_SvmStd
            \image latex SvmFigure10.eps "Standard Space Vector Modulation Technique - Center-Aligned PWM" width=10cm
            \image html SvmFigure10.jpg "Standard Space Vector Modulation Technique - Center-Aligned PWM"

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gmclib.h"
#define U_MAX 15

SWLIBS_2Syst tr32InVoltage;
SWLIBS_3Syst tr32PwmABC;
tU32         u32SvmSector;

void main(void)
{
    // Input voltage vector 15V @ angle 30deg
    // alpha component of input voltage vector = 12.99[V]
    // beta component of input voltage vector  = 7.5[V]
    tr32InVoltage.s32Arg1       = FRAC32(12.99/U_MAX);
    tr32InVoltage.s32Arg2       = FRAC32(7.5/U_MAX);

    // output pwm dutycycles stored in structure referenced by tr32PwmABC
    // pwmA dutycycle   = 0x7FFF A2C9 = FRAC32(0.9999888... )
    // pwmb dutycycle   = 0x4000 5D35 = FRAC32(0.5000111... )
    // pwmc dutycycle   = 0x0000 5D35 = FRAC32(0.0000111... )
    // svmSector        = 0x1 [sector]
    u32SvmSector = GMCLIB_SvmStd(&tr32PwmABC,&tr32InVoltage);
}
\endcode

@par Performance:
            \anchor tab5_GMCLIB_SvmStd
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GMCLIB_SvmStd function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>924/990</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>101/117</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>85/101</td>
            </tr>
            </table>

******************************************************************************/
tU32 GMCLIB_SvmStdANSIC(SWLIBS_3Syst *pOut,const SWLIBS_2Syst *const pIn)
{
    register tFrac32 s32X;
    register tFrac32 s32Y;
    register tFrac32 s32Z;
    register tFrac32 s32XSat;
    register tFrac32 s32YSat;
    register tFrac32 s32ZSat;
    register tFrac32 s32Save;
    register tFrac32 s32Alfa;


    // Value s32Alfa is for late use
    // (Sqrt(3) * u(alfa))/2
    s32Alfa = (0x6EDA * (pIn->s32Arg1>>16))<<1;

    // Definition Uref1 of the Invert Clark Transformation

    // Definition Uref2 of the Invert Clark Transformation

    // Definition Uref2 of the Invert Clark Transformation


    s32X = pIn->s32Arg2;
    s32XSat = pIn->s32Arg2;

    s32Y = (pIn->s32Arg2 >> 2) + (s32Alfa>>1);
    s32YSat = (s32Y > (tFrac32)0x3FFFFFFF)?(tFrac32)0x3FFFFFFF:(s32Y);
    s32YSat = (s32YSat < (tFrac32)(0xC0000000))?(tFrac32)0xC0000000:(s32YSat);
    s32YSat = s32YSat << 1;

    s32Z = (pIn->s32Arg2 >> 2) - (s32Alfa>>1);
    s32ZSat = (s32Z < (tFrac32)(0xC0000000))?(tFrac32)0xC0000000:(s32Z);
    s32ZSat = (s32ZSat > (tFrac32)0x3FFFFFFF)?(tFrac32)0x3FFFFFFF:(s32ZSat);
    s32ZSat = s32ZSat<<1;

    // Sector identification
    if(s32YSat < 0)
    {
        if(s32ZSat < 0)
        {
            //Sector = 5
        	s32Save = F32AddSat(0x3FFFFFFF,s32X>>1);
            pOut->s32Arg2 = (s32Save < 0)?0x0:s32Save;
        	//pOut->s32Arg2 = (0x7FFFE000+s32Z + s32Y)>>1;
            pOut->s32Arg1 = F32SubSat(s32Save,s32ZSat);
            pOut->s32Arg3 = F32SubSat(pOut->s32Arg1,s32YSat);

            return(0x5);
        }
        else
        {
            if(s32XSat > 0)
            {
                //Sector = 3

            	s32Save = F32AddSat(0x3FFFFFFF,F32SubSat(s32Y,s32X>>1));
                pOut->s32Arg1 = (s32Save < 0)?0x0:s32Save;
            	//pOut->s32Arg1 = (0x7FFF0000 + s32Y - s32X)>>1;
                pOut->s32Arg3 = F32SubSat(s32Save,s32YSat);
                pOut->s32Arg2 = F32AddSat(pOut->s32Arg3,s32X);

                return(0x3);
            }
            else
            {
                //Sector = 4

            	s32Save = F32AddSat(0x3FFFFFFF,F32SubSat(s32X>>1,s32Z));
                pOut->s32Arg1 = (s32Save < 0)?0x0:s32Save;
            	//pOut->s32Arg1 =(0x7FFF0000 - s32Z + s32X)>>1;
                pOut->s32Arg2 = F32AddSat(s32Save,s32ZSat);
                pOut->s32Arg3 = F32SubSat(pOut->s32Arg2,s32X);

                return(0x4);
            }
        }
    }
    else
    {
        if(s32ZSat < 0)
        {
            if(s32XSat > 0)
            {
                //Sector = 1

            	s32Save =(F32SubSat(0x3FFFFFFF,s32Y));
                pOut->s32Arg3 = (s32Save < 0)?0x0:s32Save;
            	//pOut->s32Arg3 = (0x7FFF0000 - s32X + s32Z)>>1;
                pOut->s32Arg2 = F32AddSat(s32Save,s32X);
                pOut->s32Arg1 = F32SubSat(pOut->s32Arg2,s32Z<<1);

                return(0x1);
            }
            else
            {
                //Sector = 6

            	s32Save = F32AddSat(0x3FFFFFFF,F32SubSat(s32X>>1,s32Y));
                pOut->s32Arg2 = (s32Save < 0)?0x0:s32Save;
            	//pOut->s32Arg2 = (0x7FFF0000 + s32X - s32Y)>>1;
                pOut->s32Arg3 = F32SubSat(s32Save,s32X);
                pOut->s32Arg1 = F32AddSat(pOut->s32Arg3,s32YSat);

                return(0x6);
            }
        }
        else
        {
            //Sector = 2

        	s32Save = F32SubSat(0x3FFFFFFF,F32AddSat(s32Y,s32Z));
            pOut->s32Arg3 = (s32Save < 0)?0x0:s32Save;
        	//pOut->s32Arg3 = (0x7FFFE000 - s32Y - s32Z)>>1;
            pOut->s32Arg1 = F32AddSat(s32Save,s32YSat);
            pOut->s32Arg2 = F32AddSat(pOut->s32Arg1,s32ZSat);

            return(0x2);
        }
    }
}

#ifdef __cplusplus
}
#endif

/* End of file */
