/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file      GMCLIB_ClarkInv.c
*
* @author    Roman Filka
*
* @version   1.0.12.0
*
* @date      Apr-26-2010
*
* @brief     Source file for Clarke Inverse Transformation algorithm.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GMCLIB_GROUP
	@addtogroup GMCLIB_GROUP
@else
	@defgroup GMCLIB_GROUP	GMCLIB
@endif
*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GMCLIB_ClarkInv.h"

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| external declarations
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs and structures (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| global variable definitions (scope: module-exported)
|----------------------------------------------------------------------------*/

/******************************************************************************
| global variable definitions (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototypes (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementations (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementations (scope: module-exported)
|----------------------------------------------------------------------------*/
/**************************************************************************//*!
@brief          Inverse Clarke Trasformation algorithm implementation.

@param[out]     *pOut   Pointer to structure containing data of three-phase
                        stationary system
                        \f$ \left(sA-sB-sC\right) \f$.

@param[in]      *pIn    Pointer to structure containing data of two-phase
                        stationary orthogonal system
                        \f$ \left(\alpha-\beta\right) \f$.

@return         void

@details        The #GMCLIB_ClarkInvANSIC function, denoting ANSI-C compatible
                implementation, can be called via function alias
                #GMCLIB_ClarkInv.

                The #GMCLIB_ClarkInv function calculates the Inverse Clarke
                transformation, which is used to transform values from the
                two-phase \f$ \left( \alpha-\beta \right) \f$ orthogonal
                coordinate system to the three-phase \f$\left(sA-sB-sC\right)\f$
                coordinate system according to these equations:
                \anchor eq1_GMCLIB_ClarkInv
                \f[
                i_{sA} = i_{\alpha}
                \f]

                \anchor eq2_GMCLIB_ClarkInv
                \f[
                i_{sB} = - \frac{1}{2} \cdot i_{\alpha} + \frac{\sqrt{3}}{2} \cdot i_{\beta}
                \f]

                \anchor eq3_GMCLIB_ClarkInv
                \f[
                i_{sC} = - \frac{1}{2} \cdot i_{\alpha} - \frac{\sqrt{3}}{2} \cdot i_{\beta}
                \f]

@note           The inputs and the outputs are normalized to fit in range
                \f$\left[-1,1\right)\f$.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gmclib.h"

SWLIBS_2Syst tr32AlBe;
SWLIBS_3Syst tr32Abc;

void main(void)
{
    // input phase alpha = sin(45) = 0.707106781
    // input phase beta  = cos(45) = 0.707106781
    tr32AlBe.s32Arg1  = FRAC32(0.707106781);
    tr32AlBe.s32Arg2  = FRAC32(0.707106781);

    // output should be
    // tr32Abc.s32Arg1 ~ phA = 0x5A827999
    // tr32Abc.s32Arg2 ~ phB = 0x2120FB83
    // tr32Abc.s32Arg3 ~ phC = 0x845C8AE5
    GMCLIB_ClarkInv(&tr32Abc,&tr32AlBe);
}
\endcode

@par Performance:
            \anchor tab1_GMCLIB_ClarkInv
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GMCLIB_ClarkInv function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>142/156</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>68/76</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>60/64</td>
            </tr>
            </table>


****************************************************************************/
void GMCLIB_ClarkInvANSIC (SWLIBS_3Syst *pOut,
                           const SWLIBS_2Syst *const pIn)
{
#ifdef USE_FRAC32_ARITHMETIC
    register tFrac32    f32M1;
    register tFrac32    f32M2;

    f32M1          = F32Neg(pIn->s32Arg1>>1);
    f32M2          = F32Mul(F32_SQRT3_DIVBY_2,pIn->s32Arg2);

    pOut->s32Arg1   = pIn->s32Arg1;
    pOut->s32Arg2   = F32AddSat(f32M1,f32M2);
    pOut->s32Arg3   = F32SubSat(f32M1,f32M2);
#else
     register tFrac16   f16M1;
     register tFrac16   f16M2;

     f16M1          = F16Neg(((tFrac16)(pIn->s32Arg1>>16))>>1);
     f16M2          = F16Mul(F16_SQRT3_DIVBY_2,(tFrac16)((pIn->s32Arg2)>>16));

     pOut->s32Arg1  = pIn->s32Arg1;
     pOut->s32Arg2  = ((tFrac32)(F16AddSat(f16M1,f16M2)))<<16;
     pOut->s32Arg3  = ((tFrac32)(F16SubSat(f16M1,f16M2)))<<16;
#endif
}
#ifdef __cplusplus
}
#endif

/* End of file */
