/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file      GMCLIB_Clark.c
*
* @author    Roman Filka
*
* @version   1.0.14.0
*
* @date      Apr-26-2010
*
* @brief     Source file for Clarke Transformation algorithm.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GMCLIB_GROUP
	@addtogroup GMCLIB_GROUP
@else
	@defgroup GMCLIB_GROUP	GMCLIB
@endif
*/
#ifdef __cplusplus
extern "C" {
#endif
/******************************************************************************
| Includes
------------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GMCLIB_Clark.h"

/******************************************************************************
| external declarations
------------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| typedefs and structures (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| global variable definitions (scope: module-exported)
------------------------------------------------------------------------------*/

/******************************************************************************
| global variable definitions (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| function prototypes (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| function implementations (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| function implementations (scope: module-exported)
------------------------------------------------------------------------------*/
/**************************************************************************//*!
@brief          Clarke Trasformation algorithm implementation.

@param[out]     *pOut  Pointer to structure containing data of two-phase
                       stationary orthogonal system
                       \f$ \left( \alpha-\beta \right) \f$.

@param[in]      *pIn   Pointer to structure containing data of three-phase
                       stationary system
                       \f$ \left(sA-sB-sC \right) \f$.

@return         void

@details        The #GMCLIB_ClarkANSIC function, denoting ANSI-C compatible
                implementation, can be called via function alias
                #GMCLIB_Clark.

                The Clarke Trasformation is used to transform values from the
                three-phase (A-B-C) coordinate system to the two-phase
                (alpha-beta) orthogonal coordinate system according to the
                following equations:
                \anchor eq1_GMCLIB_Clark
                \f[
                i_{\alpha} = i_{sA}
                \f]

                \anchor eq2_GMCLIB_Clark
                \f[
                i_{\beta} = \left(i_{sB} - i_{sC}\right) \cdot \frac{1}{\sqrt{3}}
                \f]

                where assuming that the axis \f$sA\f$ (axis of the first phase) and
                the axis \f$ \alpha \f$ are in the same direction.

@note           The inputs and the outputs are normalized to fit in range
                \f$\left[-1,1\right)\f$.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gmclib.h"

SWLIBS_3Syst tr32Abc;
SWLIBS_2Syst tr32AlBe;

void main(void)
{
    // input phase A = sin(45) = 0.707106781
    // input phase B = sin(45 + 120) = 0.258819045
    // input phase C = sin(45 - 120) = -0.965925826
    tr32Abc.s32Arg1  = FRAC32(0.707106781);
    tr32Abc.s32Arg2  = FRAC32(0.258819045);
    tr32Abc.s32Arg3  = FRAC32(-0.965925826);

    // output should be
    // tr32AlBe.s32Arg1 ~ alpha = 0x5A827999
    // tr32AlBe.s32Arg2 ~ beta  = 0x5A827999
    GMCLIB_Clark(&tr32AlBe,&tr32Abc);
}
\endcode

@par Performance:
            \anchor tab1_GMCLIB_Clark
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GMCLIB_Clark function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>132/136</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>76/90</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>60/70</td>
            </tr>
            </table>

******************************************************************************/
void GMCLIB_ClarkANSIC (SWLIBS_2Syst *pOut,
                        const SWLIBS_3Syst *const pIn)
{
#ifdef USE_FRAC32_ARITHMETIC
    // iAlpha = iA
    pOut->s32Arg1 = pIn->s32Arg1;

    // iBeta = (iB - iC) * 1/sqrt(3)
    pOut->s32Arg2 = F32SubSat(F32Mul(pIn->s32Arg2,F32_1_DIVBY_SQRT3),
                              F32Mul(pIn->s32Arg3,F32_1_DIVBY_SQRT3));
#else
     register tFrac16 s16M1;
     register tFrac16 s16M2;

     // iAlpha = iA
     pOut->s32Arg1  = pIn->s32Arg1;

     // iBeta = (iB - iC) * 1/sqrt(3)
     s16M1          = F16Mul((tFrac16)((pIn->s32Arg2)>>16),F16_1_DIVBY_SQRT3);
     s16M2          = F16Mul((tFrac16)((pIn->s32Arg3)>>16),F16_1_DIVBY_SQRT3);
     pOut->s32Arg2  = ((tFrac32)(F16SubSat(s16M1,s16M2)))<<16;
#endif
}
#ifdef __cplusplus
}
#endif

/* End of file */
