/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GFLIB_Ramp.c
*
* @author   Roman Filka
*
* @version  1.0.8.0
*
* @date     Apr-26-2010
*
* @brief    Source file containing routines for calculation of a ramp function.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90. 
*
******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB
@endif
*/ 

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GFLIB_Ramp.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-exported)
-----------------------------------------------------------------------------*/

/**************************************************************************//*!
\nosubgrouping
@brief      The function calculates the up/down ramp
            with the step increment/decrement defined in the pParam structure.

@param[in,out]  *pParam     Pointer to ramp parameters structure
@param[in]      s32In       Input argument representing the desired output value.
  
@return     The function returns 32-bit value in format Q1.31, which represents
            the actual ramp output value. This, in time, is approaching
            the desired (input) value by step increments defined in pParam
            structure.

@details    The #GFLIB_RampANSIC function, denoting ANSI-C compatible
            source code implementation, can be called via function alias
            #GFLIB_Ramp.

            \par
            If the desired (input) value is greater than the ramp output value,
            the function adds the \c s32RampUp value to the actual output value.
            The output cannot be greater than the desired value.

            \par
            If the desired value is lower than the actual value, the function
            subtracts the \c s32RampDown value from the actual value.
            The output cannot be lower than the desired value.

            \par
            Functionality of implemented ramp algorithm can be explained with
            use of Fig. \ref fig1_GFLIB_Ramp

            \anchor fig1_GFLIB_Ramp
            \image latex GFLIB_Ramp_Figure1.eps "GFLIB_Ramp functionality" width=14cm
            \image html GFLIB_Ramp_Figure1.jpg "GFLIB_Ramp functionality"

@note       All parameters and states, used by the function, can be reset during
            declaration using #GFLIB_RAMP_DEFAULT macro.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32In;
tFrac32 s32Out;

// Definition of one ramp instance
GFLIB_RAMP_T trMyRamp = GFLIB_RAMP_DEFAULT;

void main(void)
{
    // Setting parameters for hysteresis
    trMyRamp.s32RampUp      = 214748364;
    trMyRamp.s32RampDown    = 71582788;

    // input value = 0.5
    s32In  = FRAC32(0.5);

    // output should be 0xCCCCCCC
    s32Out = GFLIB_Ramp(s32In,&trMyRamp);
}
\endcode

@par Performance:
            \anchor tab1_GFLIB_Ramp
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_Ramp function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>122/142</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>51/67</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>46/63</td>
            </tr>
            </table>

******************************************************************************/
tFrac32 GFLIB_RampANSIC(tFrac32 s32In, GFLIB_RAMP_T *pParam)
{
    register tFrac32 s32TempInc;
    register tFrac32 s32TempDec;
    register tFrac32 s32TempState;

    s32TempInc = F32AddSat(pParam->s32State,pParam->s32RampUp);
    s32TempDec = F32SubSat(pParam->s32State,pParam->s32RampDown);

    s32TempState = (pParam->s32State<=s32In)? s32TempInc : s32TempDec;

    pParam->s32State = (((s32TempState>=s32In)&&(pParam->s32State<=s32In))||
                       ((s32TempState<=s32In)&&(pParam->s32State>s32In)))?
                       s32In : s32TempState;

    // Returning the ramp state
    return(pParam->s32State);
}

#ifdef __cplusplus
}
#endif

/* End of file */
