/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GFLIB_Lut1D.c
*
* @author   Andrzej Lara
*
* @version   1.0.14.0
*
* @date     Apr-26-2010
* 
* @brief  Source file for the #GFLIB_Lut1D function.  
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90. 
*
******************************************************************************/

/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB   
@endif
*/ 

#ifdef __cplusplus
extern "C" {
#endif
    
/*******************************************************************************
 * Includes
 ******************************************************************************/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GFLIB_Lut1D.h"

/*******************************************************************************
| External declarations
------------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
------------------------------------------------------------------------------*/

/*******************************************************************************
| Typedefs and structures       (scope: module-local)
------------------------------------------------------------------------------*/

/*******************************************************************************
| Global variable definitions   (scope: module-exported)
------------------------------------------------------------------------------*/

/*******************************************************************************
| Global variable definitions   (scope: module-local)
------------------------------------------------------------------------------*/

/*******************************************************************************
| Function prototypes           (scope: module-local)
------------------------------------------------------------------------------*/

/*******************************************************************************
| Function implementations      (scope: module-local)
------------------------------------------------------------------------------*/

/*******************************************************************************
| Function implementations      (scope: module-exported)
------------------------------------------------------------------------------*/

/**************************************************************************//*!

@brief      The function performs the linear interpolation over an arbitrary
            data set.

@param[in]  s32In The abscissa for which 1D interpolation is performed.
     
@param[in]  pParam  Pointer to the parameters structure.
  
@return     The interpolated value from the table with 16-bit accuracy. 

@details    The #GFLIB_Lut1D function performs one dimensional
            linear interpolation over a table of data.  The data are assumed to
            represent a one dimensional function sampled at equidistant points.
            The following interpolation formula is used:
            \f[
              y = y_1 + \frac{y_2 - y_1}{x_2 - x_1} \cdot (x - x_{1})
            \f]
            where:
            - \f$y\f$ is the interpolated value
            - \f$y_1\f$ and \f$y_2\f$ are the ordinate values at, respectively,
              the beginning and the end of the interpolating interval
            - \f$x_1\f$ and \f$x_2\f$ are the abscissa values at, respectively,
              the beginning and the end of the interpolating interval
            - the \f$x\f$ is the input value provided to the function in 
              the \c s32In argument

            The interpolating intervals are defined in the table provided
            by the \c ps32Table member of the parameters structure.  The
            table contains ordinate values consecutively over the whole
            interpolating range.  The abscissa values are assumed to be defined
            implicitly by a single interpolating interval length and a table
            index while the interpolating index zero is the table element
            pointed to by the \c ps32Table parameter.  The abscissa value is
            equal to the multiplication of the interpolating index and the
            interpolating interval length.  For example let's consider the
            following interpolating table:

            \anchor tab1_GFLIB_Lut1D
            <table border="1" CELLPADDING="5" align = "center">
                <caption>#GFLIB_Lut1D example table</caption>
                <tr>
                  <th>ordinate (y) <p></th> <th>interpolating index</th> <th>abscissa (x)</th>
                </tr>
                <tr>
                  <td>\f$-0.5\f$</td> <td>\f$-1\f$</td> <td>\f$-1*(2^{-1})\f$</td>
                </tr>
                <tr>
                  <td>\c ps32Table \f$\rightarrow 0.0\f$</td> <td>\f$0\f$</td> <td>\f$0*(2^{-1})\f$</td>
                </tr>
                <tr>
                  <td>\f$0.25\f$</td> <td>\f$1\f$</td> <td>\f$1*(2^{-1})\f$</td>
                </tr>
                <tr>
                  <td>\f$0.5\f$</td> <td>N/A</td> <td>\f$2*(2^{-1})\f$</td>
                </tr>
            </table>

            The table \ref tab1_GFLIB_Lut1D contains 4 interpolating points
            (note four rows).  Interpolating interval length in this example
            is equal to \f$ 2^{-1} \f$.
            The \c ps32Table parameter points to the second row, defining also
            the interpolating index 0.  The x-coordinates of the interpolating
            points are calculated in the right column.

            It should be noticed that the \c ps32Table pointer does not have to
            point to the start of the memory area with ordinate values.
            Therefore the interpolating index can be positive or negative or,
            even, does not have to have zero in its range.

            Special algorithm is used to make the computation efficient, 
            however under some additional assumptions as provided below:
              - the values of the interpolated function are 32-bit long
              - the length of each interpolating interval is equal to
                \f$ 2^{-n}\f$, where \f$n\f$ is an integer in the range of
                1, 2, ... 29
              - the provided abscissa for interpolation is 32-bit long

            The algorithm  performs the following steps:

              -# Compute the index representing the interval, in which the
                 linear interpolation will be performed:
                 \f[
                    I = x>>s_{Interval}
                 \f]
                 where \f$I\f$ is the interval index and the
                 \f$s_{Interval}\f$ the shift amount provided in the parameters
                 structure as the member \c s32ShamIntvl.  The operator
                 \f$>>\f$ represents the binary arithmetic right shift.
              -# Compute the abscissa offset within an interpolating interval:
                 \f[
                    \Delta{x} = x<<s_{Offset} \texttt{ \& 0x7fffffff}
                 \f]
                 where \f$\Delta{x}\f$ is the abscissa offset within an
                 interval and the \f$s_{Offset}\f$ is the shift amount provided
                 in the parameters structure.  The operators \f$<<\f$ and
                 \f$\&\f$ represent, respectively, the binary left shift and
                 the bitwise logical conjunction.  It should be noted that the
                 computation represents extraction of some least significant
                 bits of the \f$x\f$ with the sign bit cleared.
              -# Compute the interpolated value by the linear interpolation
                 between the the ordinates read from the table at the start
                 and the end of the computed interval.  The computed abscissa
                 offset is used for the linear interpolation.
                 \f[
                 \begin{array}{l}
                   y_1 = (\texttt{32-bit data at address pTable}) + 4 \cdot I \\
                   y_2 = (\texttt{32-bit data at address pTable}) + 4 \cdot (I + 1) \\
                   y = y_1 + (y_2 - y_1)\cdot\Delta{x}
                 \end{array}
                 \f]
                 where \f$y\f$, \f$y_1\f$ and  \f$y_2\f$ are, respectively, the
                 interpolated value, the ordinate at the start of the
                 interpolating interval, the ordinate at the end of the
                 interpolating interval.  The \f$\texttt{pTable}\f$ is the
                 address provided in the parameters structure \c
                 pParam->as32Table.  It should be noted that due to assumption
                 of the equidistant data points, the division by the interval
                 length is avoided.
                 
            It should be noted that the computations are performed with the
            16-bit accuracy.  In particular the 16 least significant bits are
            ignored in all multiplications.

            The shift amounts shall be provided in the parameters structure (\c
            pParam->s32ShamOffset, \c pParam->s32ShamIntvl).  The address of
            the table with the data, the \f$\texttt{pTable}\f$, shall be defined
            by the parameter structure member \c pParam->ps32Table.
            
            The shift amounts, the \f$s_{Interval}\f$ and \f$s_{Offset}\f$,
            can be computed with the following formulas:
              \f[
                \begin{array}{l}
                  s_{Interval} = 31 - |n| \\
                  s_{Offset} = |n|
                \end{array}
              \f]
            where \f$n\f$ is the integer defining the length of the
            interpolating interval in the range of -1, -2, ... -29.

            The computation of the abscissa offset and the interval index can
            be viewed also in the following way.  The input abscissa value can
            be divided into two parts.  The first \f$n\f$ most significant bits
            of the 32-bit word, after the sign bit, compose the interval
            index, in which interpolation is performed.  The rest of the bits
            form the abscissa offset within the interpolating interval.  This
            simple way to calculate the interpolating interval index and the
            abscissa offset is the consequence of assumption of all
            interpolating interval lengths equal \f$2^{-n}\f$.

            It should be noted that the input abscissa value can be positive or
            negative.  If it is positive then the ordinate values are read as
            in the ordinary data array, that is at or after the data pointer
            provided in the parameters structure (\c pParam->ps32Table).
            However, if it is negative, then the ordinate values are read from
            the memory, which is located behind the \c pParam->ps32Table
            pointer.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

GFLIB_LUT1D_T Param;

// The interpolating interval length is 2^-1
// The interpolating table pointer is defined to point to
// the element 1 (pTable1[1]).  The interpolating index can be
// then a value of -1, 0, 1.
tFrac32 pTable1[4] = {
    1073741824,     // interpolating index = -1, abscissa = -2^-1
    1342177280,     // interpolating index =  0, abscissa =  0.0
    1610612736,     // interpolating index =  1, abscissa =  2^-1
    1879048192      // interpolating index N/A,  abscissa =  1.0
                    // N/A = Not Applicable
};

volatile tFrac32 x;
volatile tFrac32 y;

int
main(int argc, char *argv[])
{
    Param.ps32Table = &(pTable1[1]);
    Param.s32ShamOffset = 1;
    Param.s32ShamIntvl = 31 - 1;

    x = 0;
    y = GFLIB_Lut1D(x, &Param);    // y = 0x50000000

    x = 536870912;
    y = GFLIB_Lut1D(x, &Param);    // y = 0x58000000
    
    x = 1610612736;
    y = GFLIB_Lut1D(x, &Param);    // y = 0x68000000

    x = 3758096384;
    y = GFLIB_Lut1D(x, &Param);    // y = 0x48000000
    
    return 0;
}
\endcode

@par Performance:
            \anchor tab2_GFLIB_Lut1D
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_Lut1D function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>66/64</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>35/38</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>32/33</td>
            </tr>
            </table>

@note       The function performs the linear interpolation with 16-bit
            accuracy.

@warning    The function does not check whether the input abscissa value is
            within a range allowed by the interpolating data table
            (\c pParam->ps32Table).  If the computed interval index points to
            data outside the provided data table then the interpolation will be
            computed with invalid data.

******************************************************************************/
tFrac32 GFLIB_Lut1DANSIC (tFrac32 s32In, 
                          const GFLIB_LUT1D_T *const pParam)
{
    register tFrac32 s32DX;
    register tFrac16 s16DX;
    register tS32 s32Intvl;
    register tFrac16 s16Y1;
    register tFrac32 s32Y1;
    register tFrac16 s16Y2;
    register tFrac32 s32Y;
    register const tFrac32 *ps32Intvl;

    /* Extract the fractional part of the input abscissa */
    s32DX = s32In<<(pParam->s32ShamOffset);
    s32DX &= 0x7fffffff;
    s16DX = (tS16) (s32DX>>16);
    
    s32Intvl = s32In>>(pParam->s32ShamIntvl);
    ps32Intvl = (pParam->ps32Table) + s32Intvl;
    s32Y1 = (*ps32Intvl);

    /* Read interpolated data and ignore 16 less significant 
     * bits by rejecting them
     */
    s16Y1 = (tFrac16) (s32Y1>>16);
    s16Y2 = (tFrac16) ((*(ps32Intvl + 1))>>16);

    /* Perform the actual interpolation with 16-bit accuracy */
    s32Y = s32Y1;
    s32Y = F32MacF16F16(s32Y, s16Y1, (tS16) -s16DX);
    s32Y = F32MacF16F16(s32Y, s16Y2, s16DX);

    return s32Y;
}

#ifdef __cplusplus
}
#endif

/* End of file */
