/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GFLIB_IntegratorTR.c
*
* @author   Roman Filka
*
* @version  1.0.9.0
*
* @date     Apr-26-2010
*
* @brief    Source file containing routines for calculation of a discrete
*           integrator using trapezoidal (Bilinear) transformation.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90. 
*
******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB
@endif
*/ 

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GFLIB_IntegratorTR.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-exported)
-----------------------------------------------------------------------------*/

/**************************************************************************//*!
\nosubgrouping
@brief      The function calculates a discrete implementation of integrator
            (sum), discretized using trapezoidal (Bilinear) transformation.

@param[in]      s32In       Input argument to be integrated.
@param[in,out]  *pParam     Pointer to integrator parameters structure
  
@return     The function returns 32-bit value in format Q1.31, which represents
            the actual integrated value of input signal.

@details    The #GFLIB_IntegratorTRANSIC function, denoting ANSI-C compatible
            source code implementation, can be called via function alias
            #GFLIB_IntegratorTR.

            \par
            The function #GFLIB_IntegratorTR implements a discrete integrator
            using trapezoidal (Bilinear) transformation.

            The continuous time domain representation of the integrator is
            defined as:
            \anchor eq1_GFLIB_IntegratorTR
            \f[
                u(t)=\int^t_0 e(t) \, dt
            \f]

            The transfer function for this integrator, in continuous
            time domain, is described using the Laplace transformation as
            follows:
            \anchor eq2_GFLIB_IntegratorTR
            \f[
                H(s)=\frac{U(s)}{E(s)}=\frac{1}{s}
            \f]

            Transforming equation \ref eq2_GFLIB_IntegratorTR into a digital
            time domain using Bilinear transformation, leads to a following
            transfer function:
            \anchor eq3_GFLIB_IntegratorTR
            \f[
                \mathcal{Z} \{H(s)\} =\frac{U(z)}{E(z)}=\frac{T_s + T_s z^{-1}}{2 - 2 z^{-1}}
            \f]

            where \f$T_s\f$ is sampling period of the system. The discrete
            implementation of digital transfer function \ref eq3_GFLIB_IntegratorTR
            is as follows:
            \anchor eq4_GFLIB_IntegratorTR
            \f[
                u(k) = u(k-1) + e(k)\cdot \frac{T_s}{2} + e(k-1)\cdot \frac{T_s}{2}
            \f]

            Considering fractional math implementation, the integrator input and
            output maximal values (scales) must be known. Then the discrete
            implementation is given as follows:
            \anchor eq5_GFLIB_IntegratorTR
            \f[
                u(k) = u(k-1) + e(k)\cdot \frac{T_s}{2} \frac{E_{MAX}}{U_{MAX}} + e(k-1)\cdot \frac{T_s}{2} \frac{E_{MAX}}{U_{MAX}}
            \f]

            where \f$ E_{MAX}\f$ is input scale and \f$ U_{MAX}\f$ is output
            scale. Then integrator constant C1 is defined as:
            \anchor eq6_GFLIB_IntegratorTR
            \f[
                C1_f = \frac{T_s}{2} \frac{E_{MAX}}{U_{MAX}}
            \f]

            In order to implement the discrete form integrator as in \ref eq5_GFLIB_IntegratorTR
            on fixed point platform, value of \f$ C1_f\f$ coefficient must
            reside in a fractional range \f$\left[-1,1\right)\f$. Therefore
            scaling must be introduced as follows:
            \anchor eq7_GFLIB_IntegratorTR
            \f[
                 s32C1   = C1_f \cdot 2^{-u16NShift}
            \f]

            Introduced scaling is chosen such that coefficient \f$s32C1\f$
            fits into fractional range \f$\left[-1,1\right)\f$.
            To simplify the implementation, this scaling is chosen to
            be a power of 2, so the final scaling is a simple shift operation
            using \c u16NShift variable. Hence the shift is calculated as:
            \anchor eq8_GFLIB_IntegratorTR
            \f[
                u16NShift =
                \mbox{ceil} \left( \frac{\log(C1_f)}{\log(2)} \right)
            \f]

@note       All parameters and states, used by the function, can be reset during
            declaration using #GFLIB_INTEGRATOR_TR_DEFAULT macro.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32In;
tFrac32 s32Out;

// Definition of one integrator instance
GFLIB_INTEGRATOR_TR_T trMyIntegrator = GFLIB_INTEGRATOR_TR_DEFAULT;

void main(void)
{
    // Setting parameters for integrator, Ts = 100e-6, E_MAX=U_MAX=1
    trMyIntegrator.s32C1         = FRAC32(100e-6/2);
    trMyIntegrator.u16NShift     = 0;

    // input value = 0.5
    s32In  = FRAC32(0.5);

    // output should be 0x0000D1B7
    s32Out = GFLIB_IntegratorTR(s32In,&trMyIntegrator);
}
\endcode

@par Performance:
            \anchor tab1_GFLIB_IntegratorTR
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_IntegratorTR function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>184/142</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>82/68</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>66/56</td>
            </tr>
            </table>

******************************************************************************/
tFrac32 GFLIB_IntegratorTRANSIC(tFrac32 s32In, GFLIB_INTEGRATOR_TR_T *pParam)
{
#ifdef USE_FRAC32_ARITHMETIC
    register tFrac32 s32M1;
    register tFrac32 s32M2;
    register tFrac32 s32A1;
   
    /* 
     * Implemented equation:
     * out(k) = out(k-1) + in(k)*C1 + in(k-1)*C1
     * 
     * Calculation steps:
     * M1   = in(k)*C1
     * M2   = in(k-1)*C1
     * A1   = M1+M2
     * Acc  = out(k-1)+A1
     */

    // M1   = in(k)*C1
    s32M1   = F32MulSat(s32In, pParam->s32C1);

    // M2   = in(k-1)*C1
    s32M2   = F32MulSat(pParam->s32InK1, pParam->s32C1);

    // A1 = M1+M2
    s32A1   = F32Add(s32M1,s32M2);

    // A2 = out(k-1)+A1 
    pParam->s32State  = F32Add(pParam->s32State,s32A1);

    // Storing input value in the buffer
    pParam->s32InK1  = s32In;

    // Returning de-scaled value
    return(F32ShlSat(pParam->s32State,(tS32)pParam->u16NShift));
#else
    register tFrac32 s32A1;

    /*
     * Implemented equation:
     * out(k) = out(k-1) + in(k)*C1 + in(k-1)*C1
     *
     * Calculation steps:
     * M1   = in(k)*C1
     * M2   = in(k-1)*C1
     * A1   = M1+M2
     * Acc  = out(k-1)+A1
     */

    // A1   = out(k-1) + in(k)*C1
    s32A1   = F32MacF16F16(pParam->s32State,
                             (tFrac16)(s32In>>16),
                             (tFrac16)((pParam->s32C1)>>16));

    // out(k) = A1 + in(k-1)*C1
    pParam->s32State  = F32MacF16F16(s32A1,
                             (tFrac16)((pParam->s32InK1)>>16),
                             (tFrac16)((pParam->s32C1)>>16));

    // Storing input value in the buffer
    pParam->s32InK1  = s32In;

    // Returning de-scaled value
    return(F32ShlSat(pParam->s32State,(tS32)pParam->u16NShift));

#endif
}

#ifdef __cplusplus
}
#endif

/* End of file */
