/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GFLIB_Hyst.c
*
* @author   Roman Filka
*
* @version  1.0.11.0
*
* @date     Apr-26-2010
*
* @brief    Source file containing routines for calculation of the hysteresis
*           function.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GFLIB_GROUP
	@addtogroup GFLIB_GROUP
@else
	@defgroup GFLIB_GROUP	GFLIB
@endif
*/

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
* Includes
******************************************************************************/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GFLIB_Hyst.h"

/******************************************************************************
* External declarations
******************************************************************************/

/******************************************************************************
* Defines and macros            (scope: module-local)
******************************************************************************/

/******************************************************************************
* Typedefs and structures       (scope: module-local)
******************************************************************************/

/******************************************************************************
* Global variable definitions   (scope: module-exported)
******************************************************************************/

/******************************************************************************
* Global variable definitions   (scope: module-local)
******************************************************************************/

/******************************************************************************
* Function prototypes           (scope: module-local)
******************************************************************************/

/******************************************************************************
* Function implementations      (scope: module-local)
******************************************************************************/

/******************************************************************************
* Function implementations      (scope: module-exported)
******************************************************************************/

/**************************************************************************//*!
@brief      Hysteresis function.

@param[in]  s32In       Input signal in form of a 32bit fixed point fractional
                        number, normalized between \f$\left[-1,1\right)\f$.

@param[in]  *pParam     Pointer to structure with parameters and states of the
                        hysteresis function.

@return     The function returns the value of hysteresis output, which is equal
            to either \e s32OutValOn or \e s32OutValOff depending on the value
            of input and state of the function output in previous calculation
            step. The output value is interpreted as fixed point
            32-bit number, normalized between \f$\left[-1,1 \right)\f$.

@details    The #GFLIB_HystANSIC function, denoting ANSI-C compatible
            source code implementation, can be called via function alias
            #GFLIB_Hyst.

            \par
            The #GFLIB_Hyst function provides a computational method for
            calculation of a hysteresis(relay) function. The function switches
            the output between the two predefined values stored in \e s32OutValOn
            and \e s32OutValOff members of structure #GFLIB_HYST_T.
            When the value of input is higher than the upper threshold
            \e s32OutValOn then the output value is equal to \e s32OutValOn.
            On the other hand, when the input value is lower than the lower
            threshold \e s32OutValOff then the output value is equal to
            \e s32OutValOff.
            When the input value is between the two threshold values then
            the output retains its value.
            \anchor eq1_GFLIB_Hyst
            \f[
            s32OutState(k) = \left\{\begin{array}{lll}
                       s32OutValOn & \mbox{if} & s32In \geq s32HystOn \\
                       s32OutValOff & \mbox{if} & s32In \leq s32HystOff \\
                       s32OutState(k-1) & \mbox{otherwise} &
                     \end{array}\right.
            \f]

            \par
            Graphical description of #GFLIB_Hyst functionality is shown on Fig.
            \ref fig1_GFLIB_Hyst.

            \anchor fig1_GFLIB_Hyst
            \image latex GFLIB_Hyst_Figure1.eps "Hysteresis function" width=7cm
            \image html GFLIB_Hyst_Figure1.jpg "Hysteresis function"

@warning    For correct functionality, s32HystOn must be greater than s32HystOff.

@note       All parameters and states, used by the function, can be reset during
            declaration using #GFLIB_HYST_DEFAULT macro.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32In;
tFrac32 s32Out;

// Definition of one hysteresis instance
GFLIB_HYST_T trMyHyst = GFLIB_HYST_DEFAULT;

void main(void)
{
    // Setting parameters for hysteresis
    trMyHyst.s32HystOn      = FRAC32(0.3);
    trMyHyst.s32HystOff     = FRAC32(-0.3);
    trMyHyst.s32OutValOn    = FRAC32(0.5);
    trMyHyst.s32OutValOff   = FRAC32(-0.5);

    // input value = 0.5
    s32In  = FRAC32(0.5);

    // output should be 0x40000000
    s32Out = GFLIB_Hyst(s32In,&trMyHyst);
}
\endcode

@par Performance:
            \anchor tab1_GFLIB_Hyst
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_Hyst function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>44/26</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>27/26</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>22/21</td>
            </tr>
            </table>
******************************************************************************/
tFrac32 GFLIB_HystANSIC(tFrac32 s32In, GFLIB_HYST_T *pParam)
{
    register tFrac32 s32TempVal;

    s32TempVal          = (s32In < pParam->s32HystOn)? pParam->s32OutState :
                                                        pParam->s32OutValOn;

    pParam->s32OutState = (s32In <= pParam->s32HystOff)? pParam->s32OutValOff :
                                                         s32TempVal;

    return (pParam->s32OutState);
}

#ifdef __cplusplus
}
#endif

/* End of file */
