/******************************************************************************
 *
 * (c) Copyright 2009, Freescale & STMicroelectronics
 *
 ***************************************************************************//*!
 *
 * @file     GFLIB_AtanYXShifted.c
 *
 * @author   r29302
 * 
 * @version  1.0.10.0
 * 
 * @date     Apr-26-2010
 * 
 * @brief    Function performs calculation the AranYXShifted.  
 *                       
 *
 *******************************************************************************
 *
 * Function implemented as ANSIC ISO/IEC 9899:1990, C90. 
 *
 ******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB   
@endif
*/ 

#ifdef __cplusplus
extern "C" {
#endif
    
/******************************************************************************
 * Includes
 ******************************************************************************/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "GFLIB_AtanYX.h"
#include "GFLIB_AtanYXShifted.h"

/******************************************************************************
 * External declarations
 ******************************************************************************/

/******************************************************************************
 * Defines and macros            (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Typedefs and structures       (scope: module-local)
 ******************************************************************************/


/******************************************************************************
 * Global variable definitions   (scope: module-exported)
 ******************************************************************************/

/******************************************************************************
 * Global variable definitions   (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Function prototypes           (scope: module-local)
 ******************************************************************************/

/****************************************************************************** * Function implementations      (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Function implementations      (scope: module-exported)
 ******************************************************************************/

/**************************************************************************//*!
@brief      The function calculates the angle of two sine waves shifted
            in phase one to each other.

@param[in]  s32InY  The value of the first signal, assumed to be \f$sin(\theta)\f$.
@param[in]  s32InX  The value of the second signal, assumed to be
                    \f$sin(\theta + \Delta\theta)\f$.
@param[in]  pParam  The parameters for the functions.

@return     The function returns the angle of two sine waves shifted
            in phase one to each other.
        
@details    The function calculates the angle of two sinusoidal signals, one
            shifted in phase to the other.  The phase shift between sinusoidal
            signals does not have to be \f$\pi/2\f$ and can be any value.

            It is assumed that the arguments of the function are as follows:
            \f[
                \begin{array}{lr}
                y = sin(\theta)\\
                x = sin(\theta + \Delta\theta)\\
                \end{array}
            \f]

            where:
            - \f$x, y\f$ are respectively, the \c s32InX, and \c s32InY
              arguments
            - \f$\theta\f$ is the angle to be computed by the function
            - \f$\Delta\theta\f$ is the phase difference between the 
              \f$x, y\f$ signals

            At the end of computations, an angle offset \f$\theta_{Offset}\f$
            is added to the computed angle \f$\theta\f$.  The angle offset is
            an additional parameter, which can be used to set the zero of the
            \f$\theta\f$ axis.  If \f$\theta_{Offset}\f$ is zero, then the
            angle computed by the function will be exactly \f$\theta\f$.

            The #GFLIB_AtanYXShifted function does not use directly the angle
            offset \f$\theta_{Offset}\f$ and the phase difference \f$\theta\f$.
            The function's parameters, contained in the function parameters
            structure \c GFLIB_ATANYXSHIFTED_T, need to be computed by means of
            provided Matlab function (see below).

            If \f$\Delta\theta=\pi/2\f$ or \f$\Delta\theta=-\pi/2\f$ then the
            function is similar to the #GFLIB_AtanYX function, however the
            #GFLIB_AtanYX function in this case is more effective with regard
            to execution time and accuracy.

            In order to use the function the following steps are necessary to
            be completed:
            - define \f$\Delta\theta\f$ and \f$\theta_{Offset}\f$, the
              \f$\Delta\theta\f$ shall be known from the input sinusoidal
              signals, the \f$\theta_{Offset}\f$ needs to be set arbitrarily
            - compute values for the function parameters structure by means
              of provided Matlab function
            - convert the computed values into integer format and insert them
              into C code (see also the C code example)

            The function uses the following algorithm for computing the angle:
            \f[
                \begin{array}{lr}
                b = \frac{\displaystyle S}{\displaystyle 2cos\frac{\Delta\theta}{2}}(y + x)\\
                a = \frac{\displaystyle S}{\displaystyle 2sin\frac{\Delta\theta}{2}}(x - y)\\
                \theta = \mbox{AtanYX}(b, a) - (\Delta\theta/2 - \theta_{Offset})\\
                \end{array}
            \f]

            where:
            - \f$x\f$, \f$y\f$ are respectively, the \c s32InX, and \c s32InY
            - \f$\theta\f$ is the angle to be computed by the function,
              see the previous equation
            - \f$\Delta\theta\f$ is the phase difference between the 
              \f$x, y\f$ signals, see the previous equation
            - \f$S\f$ is a scaling coefficient, \f$S\f$ is almost 1, 
              (\f$S < 1\f$), see also explanation below
            - \f$a\f$, \f$b\f$ intermediate variables
            - \f$\theta_{Offset}\f$ is the additional phase shift, the computed
              angle will be \f$\theta + \theta_{Offset}\f$

            The scale coefficient \f$S\f$ is used to prevent overflow and to
            assure symmetry around 0 for the whole fractional range.  \f$S\f$
            shall be less than \f$1.0\f$, but as large as possible.  The
            algorithm implemented in this function uses the value of
            \f$1 - 2^{-15}\f$.

            The algorithm can be easily justified by proving the
            trigonometric identity:
            \f[
                \mbox{tan}(\theta + \Delta\theta)
                    = \frac
                        {\displaystyle (y + x)cos\frac{\Delta\theta}{2}}
                        {\displaystyle (x - y)sin\frac{\Delta\theta}{2}}
            \f]

            For the purposes of fractional arithmetic the algorithm is
            implemented such that additional values are used as shown in the
            equation below:
            \f[
                \begin{array}{lr}
                \frac{\displaystyle S}{\displaystyle 2cos\frac{\Delta\theta}{2}} = C_y = K_{y}2^{N_y}\\
                \frac{\displaystyle S}{\displaystyle 2sin\frac{\Delta\theta}{2}} = C_x = K_{x}2^{N_x}\\
                \theta_{adj} = \Delta\theta/2 - \theta_{offset}\\
                \end{array}
            \f]
            where:
            - \f$C_y\f$, \f$C_x\f$ are the algorithm coefficients for 
              \f$y\f$ and \f$x\f$ signals
            - \f$K_y\f$ is multiplication coefficient of the \f$y\f$ signal
              represented the parameters structure member \c pParam->s32Ky
            - \f$K_x\f$ is multiplication coefficient of the \f$x\f$ signal
              represented the parameters structure member \c pParam->s32Kx
            - \f$N_y\f$ is scaling coefficient of the \f$y\f$ signal
              represented the parameters structure member \c pParam->s32Ny
            - \f$N_x\f$ is scaling coefficient of the \f$x\f$ signal
              represented the parameters structure member \c pParam->s32Nx
            - \f$\theta_{adj}\f$ is an adjusting angle represented by the
              parameters structure member \c pParam->s32ThetaAdj

            The multiplication and scaling coefficients and the adjusting
            angle shall be defined in a parameters structure provided as
            the function input parameter.

            The function uses 16-bit fractional arithmetic for multiplication,
            therefore the 16 least significant bits of the input values and
            \f$K_y\f$, \f$K_x\f$ multiplication coefficients are ignored.

            The function initialization parameters can be calculated as
            shown in the following Matlab code:

            \par
\code
function [KY, KX, NY, NX, THETAADJ] = atanyxshiftedpar(dthdeg, thoffsetdeg)
// ATANYXSHIFTEDPAR calculation of parameters for atanyxshifted() function
//
// [KY, KX, NY, NX, THETAADJ] = atanyxshiftedpar(dthdeg, thoffsetdeg)
//
// dthdeg = phase shift (delta theta) between sine waves in degrees
// thoffsetdeg = angle offset (theta offset) in degrees
// NY - scaling coefficient of y signal
// NX - scaling coefficient of x signal
// KY - multiplication coefficient of y signal
// KX - multiplication coefficient of x signal
// THETAADJ - adjusting angle in radians, scaled from [-pi, pi) to [-1, 1)

    if (dthdeg < -180) || (dthdeg >= 180)
        error('atanyxshiftedpar: dthdeg out of range');
    end
    if (thoffsetdeg < -180) || (thoffsetdeg >= 180)
        error('atanyxshiftedpar: thoffsetdeg out of range');
    end

    dth2 = ((dthdeg/2)/180*pi);
    thoffset = (thoffsetdeg/180*pi);
    CY = (1 - 2^-15)/(2*cos(dth2));
    CX = (1 - 2^-15)/(2*sin(dth2));
    if(abs(CY) >= 1) NY = ceil(log2(abs(CY)));
    else NY = 0;
    end
    if(abs(CX) >= 1) NX = ceil(log2(abs(CX)));
    else NX = 0;
    end
    KY = CY/2^NY;
    KX = CX/2^NX;
    THETAADJ = dthdeg/2 - thoffsetdeg;

    if THETAADJ >= 180
        THETAADJ = THETAADJ - 360;
    elseif THETAADJ < -180
        THETAADJ = THETAADJ + 360;
    end

    THETAADJ = THETAADJ/180;

return;
\endcode

            \par
            While applying the function some general guidelines should be
            considered as stated below.

            At some values of the phase shift, and particularly at phase shift
            approaching -180, 0 or 180 degrees, the algorithm may become
            numerically unstable causing that any error, contributed by input
            signal imperfections or finite precision arithmetic, is magnified
            significantly.  Therefore some care should be taken to avoid error
            where possible.  The detailed error analysis of the algorithm is
            beyond the scope of this documentation, however general guidelines
            are provided.
            
            There are several sources of error in the function:
            - error of the supplied signals values due to the finite
              resolution of the AD conversion
            - error contributed by higher order harmonics appearing in the
              input signals
            - computational error of the multiplication due to finite length of
              registers
            - error of the phase shift \f$\Delta\theta\f$ representation in 
              finite precision arithmetic and in value
            - error due to differences in signal amplitudes

            It should be noticed that the function requires both signals to have
            the same amplitude. To minimize the output error the amplitude
            of both signals should be close to 1.0 as much as possible.

            The function has been tested to be reliable at phase shift in the
            range of [-165, -15] and [15, 165] degrees for perfect sinusoidal
            input signals.  Beyond this range the function operates correctly,
            however the output error can be beyond the guaranteed value.  In
            real application the error, contributed by an AD conversion and by
            higher order harmonics of the input signals, should be also taken
            into account.

@note       The function calls the #GFLIB_AtanYX function.  The function
            may become numerically unstable for the phase shift approaching
            -180, 0 or 180 degrees.  The function accuracy is guaranteed for the
            phase shift in the range of [-165, -15] and [15, 165] degrees at
            perfect input signals.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32InY;
tFrac32 s32InX;
tFrac32 s32Ang;
GFLIB_ATANYXSHIFTED_T Param;

void main(void)
{
    //dtheta = 69.33deg, thetaoffset = 10deg
    //CY = (1 - 2^-15)/(2*cos((69.33/2)/180*pi))= 0.60789036201452440
    //CX = (1 - 2^-15)/(2*sin((69.33/2)/180*pi))= 0.87905201358520957
    //NY = 0 (abs(CY) < 1)
    //NX = 0 (abs(CX) < 1)
    //KY = 0.60789/2^0 = 0.60789036201452440
    //KX = 0.87905/2^0 = 0.87905201358520957
    //THETAADJ = 10/180 = 0.137027777777778

    Param.s32Ky = FRAC32(0.60789036201452440);
    Param.s32Kx = FRAC32(0.87905201358520957);
    Param.s32Ny = 0;
    Param.s32Nx = 0;
    Param.s32ThetaAdj = FRAC32(0.137027777777778);
    
    // theta = 15 deg
    // Y = sin(theta) = 0.2588190
    // X = sin(theta + dtheta) = 0.9951074
    s32InY = FRAC32(0.2588190);
    s32InX = FRAC32(0.9951074);    
    s32Ang = GFLIB_AtanYXShifted(s32InY, s32InX, &Param);

    // s32Ang contains 0x11c6cdfc, the theoretical value is 0x11c71c72
    
    return;
}
\endcode

@par Performance:
            \anchor tab1_GFLIB_ControllerPIp
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_AtanYXShifted function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>220/248</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>295/321</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>100/134</td>
            </tr>
            </table>

******************************************************************************/
tFrac32 GFLIB_AtanYXShiftedANSIC(tFrac32 s32InY, tFrac32 s32InX,
                const GFLIB_ATANYXSHIFTED_T * const pParam)
{
    register tFrac32 s32Aux;       /* 32-bit auxiliary var, many purposes */
    register tFrac32 s32AccA;      /* 32-bit accumulator */
    register tFrac32 s32AccB;      /* 32-bit accumulator */
    register tFrac32 s32AccShMax;  /* Max. 32-bit fract. shifted right */
    register tFrac32 s32AccShMin;  /* Min. 32-bit fract. shifted right */
    register tFrac16 s16Ky;        /* Ky coeff. in 16-bit fract. format */
    register tFrac16 s16Kx;        /* Kx coeff. in 16-bit fract. format */
    register tFrac16 s16InY;       /* s32Y in 16-bit fract. format */
    register tFrac16 s16InX;       /* s32X in 16-bit fract. format */
    register tS32 s32Ny;           /* pParam->Ny, auxiliary */
    register tS32 s32Nx;           /* pParam->Nx, auxiliary */
    register tS32 s32ThetaAdj;     /* pParam->ThetaAdj, auxiliary */
  
    /* Load some values from memory.
     * Convert 32-bit fractional number to 16-bit fractional format. */
    s16InY = (tS16) (s32InY>>16);
    s16InX = (tS16) (s32InX>>16);
    s16Ky = (tS16) (pParam->s32Ky>>16);
    s16Kx = (tS16) (pParam->s32Kx>>16);
    s32Ny = pParam->s32Ny;
    s32Nx = pParam->s32Nx;
    s32ThetaAdj = pParam->s32ThetaAdj;
    
    /* Compute: AccB = Ky*(Y + X)
     * Note that 16 less significant bits of multiplicand and multiplier
     * are ignorred.
     * The saturation at mac operation is required due to error on
     * input y, x values.  The error may cause an overflow, which is
     * theoretically not possible. */
    s32AccB = F32MulF16F16(s16InX, s16Ky);
    s32AccB = F32MacSatF16F16(s32AccB, s16Ky, s16InY);
 
    /* Compute: AccA = Kx*(X - Y)
     * Note that 16 less significant bits of multiplicand and multiplier
     * are ignorred.
     * The saturation at mac operation is required due to error on
     * input y, x values.  The error may cause an overflow, which is
     * theoretically not possible.
     * The explicit type cast in fron of negation is placed to order
     * to avoid a warning by some compilers about conversion from
     * int to short dur to integer promotion. */
    s32AccA = F32MulF16F16(s16Kx, s16InX);
    s32AccA = F32MacSatF16F16(s32AccA, (tFrac16) -s16Kx, s16InY);
 
    /* Compute: AccB = Ky*(Y + X)<<Ny
     * Saturate if necessary */
    s32Aux = s32AccB;
    s32AccB = s32AccB << s32Ny;
    s32AccShMax = ((tFrac32) INT32_MAX) >> s32Ny;
    s32AccShMin = ((tFrac32) INT32_MIN) >> s32Ny;
    s32AccB = (s32Aux > s32AccShMax) ? INT32_MAX:s32AccB;
    s32AccB = (s32Aux <= s32AccShMin) ? INT32_MIN:s32AccB;
 
    /* Compute: AccA = Kx*(X - Y)<<Nx
     * Saturate if necessary */
    s32Aux = s32AccA;
    s32AccA = s32AccA << s32Nx;
    s32AccShMax = ((tFrac32) INT32_MAX) >> s32Nx;
    s32AccShMin = ((tFrac32) INT32_MIN) >> s32Nx;
    s32AccA = (s32Aux > s32AccShMax) ? INT32_MAX:s32AccA;
    s32AccA = (s32Aux <= s32AccShMin) ? INT32_MIN:s32AccA;
 
    /* Call AtanYX routine */
    s32Aux = GFLIB_AtanYX(s32AccB, s32AccA);

    /* Make final adjustemnt */
    s32Aux = F32Sub(s32Aux, s32ThetaAdj);
 
    return s32Aux;
}

#ifdef __cplusplus
}
#endif

/* End of file */
