/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file      GFLIB_Acos.c
*
* @author    B04459 
* 
* @version    1.0.15.0
* 
* @date      Apr-26-2010
* 
* @brief     Source file for Arccosine function
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB   
@endif
*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Defines.h"
#include "SWLIBS_Inlines.h"

#include "GFLIB_Acos.h"
#include "GFLIB_Sqrt.h"

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************************
| external declarations
|-----------------------------------------------------------------------*/

/************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------*/
    
/************************************************************************
| typedefs and structures (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| global variable definitions (scope: module-exported)
|-----------------------------------------------------------------------*/

/************************************************************************
| global variable definitions (scope: module-local)
|-----------------------------------------------------------------------*/

const GFLIB_ACOS_TAYLOR_T gflibAcosCoef = 
{
    91918582,  66340080,  9729967, 682829947,      12751,  // Coef. <0;1/2)
    -52453538, -36708911,-15136243,-964576326, 1073652175  // Coef.  <1/2;1)
};
/************************************************************************
| function prototypes (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| function implementations (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| function implementations (scope: module-exported)
|-----------------------------------------------------------------------*/
/**************************************************************************//*!
@brief     Arccosine function - Piece-wise polynomial approximation.

@param[in]  s32In       Input argument is a 32bit number that contains the
                        value between \f$\left[-1,1\right)\f$.
                        
@param[in]  *pParam     Pointer to array of Taylor coefficients. The
                        function alias #GFLIB_Acos uses the default coefficients.

@return     The function returns \f$\frac{\arccos(s32In)}{\pi} \f$ as fixed point
            32-bit number, normalized between \f$\left[0,1\right)\f$.

@details    The #GFLIB_AcosANSIC function, denoting ANSI-C compatible
            source code implementation, can be called via function alias
            #GFLIB_Acos.

            \par
            The #GFLIB_Acos function provides a computational method for
            calculation of the standard inverse trigonometric \e arccosine function
            \f$\arccos(x)\f$, using the piece-wise polynomial approximation.
            Function \f$\arccos(x)\f$ takes a ratio of the length of the adjacent
            side to the length of the hypotenuse and returns the angle. 
            
            \anchor fig1_GFLIB_Acos
            \image latex GFLIB_Acos_Figure1.eps "Course of the functionGFLIB_Acos" width=10cm
            \image html GFLIB_Acos_Figure1.jpg "Course of the functionGFLIB_Acos"

            The computational algorithm uses the symmetry of the \f$ arccos(x) \f$ function
            around the point \f$ \left(0, \pi/2 \right) \f$, which allows to compute the function values
            just in the interval \f$ \left[0, 1 \right) \f$ and to compute the function values in the
            interval \f$ \left[-1, 0 \right) \f$ by the simple formula:
            
            \anchor eq1_GFLIB_Acos
            \f[
                y_{\left[-1, 0 \right)} = \pi/2 + y_{\left[ 0, 1 \right)},
            \f]
            
            where: 
            - \f$ y_{\left[-1, 0 \right)} \f$ is the \f$arccos(x)\f$ function value in the interval
              \f$ \left[-1, 0 \right) \f$
            - \f$ y_{\left[0, 1 \right)} \f$ is the \f$arccos(x)\f$ function value in the interval
              \f$ \left[0, 1 \right) \f$
             
            Additionally, because the \f$arccos(x)\f$ function is difficult for
            polynomial approximation for \f$ x \f$ approaching 1 (or -1 by symmetry)
            due to its derivatives approaching infinity, special transformation 
            is used to transform the range of \f$ x \f$ from \f$ \left[0.5, 1 \right)\f$ to
            \f$ \left(0, 0.5 \right] \f$:
            
            \anchor eq2_GFLIB_Acos
            \f[
                \arccos(\sqrt{1-x}) = \frac{\pi}{2} - \arccos(\sqrt{x})
            \f]
             
            In this way the computation of the \f$ arccos(x) \f$ function in the range 
            \f$ \left[0.5, 1 \right) \f$ can be replaced by the computation in the range 
            \f$ \left(0, 0.5 \right] \f$, in which approximation is easier.
            
            For the interval \f$ \left(0, 0.5 \right]\f$ the algorithm uses polynomial approximation
            as follows. 

            \anchor eq3_GFLIB_Acos
            \f[
                \mbox{s32Dump} = a_1 \cdot s32In^4 + a_2 \cdot s32In^3 + a_3 \cdot s32In^2 + a_4 \cdot s32In + a_5
            \f]

            \anchor eq4_GFLIB_Acos
            \f[
                \arccos(s32In) =
                    \left\{\begin{array}{lclcl}
                       \mbox{-s32Dump}  & \mbox{if} & -1 \leq \mbox{s32In} < 0 \\
                       \\
                       \mbox{s32Dump} & \mbox{if} & 0 \leq \mbox{s32In} < 1
                    \end{array}\right.
            \f]

            Division of \f$\left[0,1\right)\f$ interval into two
            sub-intervals with polynomial coefficients calculated for each
            sub-interval, are noted in table \ref tab1_GFLIB_Acos. 

            \anchor tab1_GFLIB_Acos
            <table border="1" CELLPADDING="5" align = "center">
            <caption>Integer Polynomial coefficients for each interval:</caption>
            <tr>
              <th>Interval</th>
              <th>\f$a_1\f$</th>
              <th>\f$a_2\f$</th>
              <th>\f$a_3\f$</th>
              <th>\f$a_4\f$</th>
              <th>\f$a_5\f$</th>
            </tr>
            <tr align="center">
              <td style="font-weight:bold"> \f$ \left< 0 , \frac{1}{2} \right) \f$ </td>
              <td>91918582</td> <td>66340080</td> <td>9729967</td> <td>682829947</td><td>12751</td>
            </tr>
            <tr align="center">
              <td style="font-weight:bold">\f$ \left< \frac{1}{2}  , 1 \right) \f$</td>
              <td>-52453538</td><td>-36708911</td><td>-15136243</td><td>-964576326</td><td>1073652175</td>
            </tr>
            </table>

            The implementation of the #GFLIB_Acos is almost the same as in the 
            function #GFLIB_Asin. However the output of the #GFLIB_Acos is 
            corrected as follows:
            \anchor eq5_GFLIB_Acos
            \f[
                \mbox{s32Dump} = FRAC32(0.5) - s32Dump 
            \f]
            
            The polynomial coefficients were obtained using Matlab fitting
            function, where polynomial of 5th order was used for fitting of
            each respective sub-interval. The functions \e arcsine and \e
            arccosine are similar therefore #GFLIB_Acos function uses same
            polynomial coefficients as #GFLIB_Asin function.

            \anchor fig2_GFLIB_Acos
            \image latex GFLIB_Acos_Figure2.eps "acos(x) vs. GFLIB_Acos(s32In)" width=14cm
            \image html GFLIB_Acos_Figure2.jpg "acos(x) vs. GFLIB_Acos(s32In)"

            \par
            Fig.\ref fig2_GFLIB_Acos depicts the floating
            point \e arccosine function generated from Matlab, the approximated
            value of \e arccosine function obtained from the #GFLIB_Acos
            function and their difference.
            The achieved accuracy is better than 1.3LSB on upper 16-bit of the 
            32bit result.

@note       The output angle is normalized into the range of 
            \f$\left[0,1\right)\f$.  The function calls The function uses data
            stored in the gflibAcosCoef structure, which is supplied to the
            #GFLIB_Acos function as the argument, which is masked in the
            function alias #GFLIB_Acos.  The polynomial coefficients can be
            calculated by the user and in such case the #GFLIB_AcosANSIC
            function call with pointer to the newly defined coefficients shall
            be used instead of the function alias.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32Input;
tFrac32 s32Angle;

void main(void)
{
    // input s32Input = 0
    s32Input  = FRAC32(0);

    // output should be 0x400031EF = 0.5 => pi/2
    s32Angle = GFLIB_Acos(s32Input);
}
\endcode

@par Performance:
            \anchor tab2_GFLIB_Acos
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_Acos function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>178/158</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>40/40</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>182/180</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>77/74</td>
            </tr>
            </table>

****************************************************************************/
tFrac32 GFLIB_AcosANSIC(tFrac32 s32In, const GFLIB_ACOS_TAYLOR_T *const pParam) 
{
    tFrac32 s32Sign,s32Dump;
    tS32    nSector = 0;
    
    s32Sign = s32In;
    s32In = F32AbsSat(s32In);
    if (s32In >= INT32_MAX/2)
    {
        s32In = F32NegSat(s32In);
        s32In = (s32In ^ INT32_MIN);
        s32In = GFLIB_Sqrt(s32In);
        nSector = 1;
    }
    
    s32In = s32In >> 15;
        
    //Polynom
    // (a1 * x) + a2
    s32Dump = ((pParam->GFLIB_ACOS_SECTOR[nSector].s32a[0] >> 16) * (s32In)) + pParam->GFLIB_ACOS_SECTOR[nSector].s32a[1];
    // ((a1 * x) + a2) * x + a3
    s32Dump = ((s32Dump >> 14) * (s32In>>2)) + pParam->GFLIB_ACOS_SECTOR[nSector].s32a[2];
    // (((a1 * x) + a2) * x + a3) * x + a4
    s32Dump = ((s32Dump >> 14) *(s32In>>2)) + pParam->GFLIB_ACOS_SECTOR[nSector].s32a[3];
    // ((((a1 * x) + a2) * x + a3) * x + a4) * x + a5
    s32Dump = ((s32Dump >> 14) *(s32In>>2)) + pParam->GFLIB_ACOS_SECTOR[nSector].s32a[4];
    
    // Sign is added to the result value 
    s32Dump = (s32Sign > 0)?s32Dump:-s32Dump;
    return (INT32_MAX/2 - s32Dump);
}

#ifdef __cplusplus
}
#endif

/* End of file */
