/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GDFLIB_FilterMA.c
*
* @author   Roman Filka
*
* @version  1.0.8.0
*
* @date     Apr-26-2010
*
* @brief    Moving average filter - recursive form.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GDFLIB_GROUP
    @addtogroup GDFLIB_GROUP
@else
    @defgroup GDFLIB_GROUP   GDFLIB
@endif
*/

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GDFLIB_FilterMA.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-exported)
-----------------------------------------------------------------------------*/

/**************************************************************************//*!
@ingroup    GDFLIB_GROUP
@brief      This function clears internal filter accumulator.

@param[in,out]  *pParam     Pointer to filter structure with filter accumulator
                            and filter parameters

@return     void 

@details    This function clears internal accumulator of moving average filter.
            It shall be called after filter parameter initialization and whenever
            the filter initialization is required.

            \par
            The size of the filter window (number of filtered points)
            shall be defined prior to this function call. The number of the
            filtered points is defined by assigning a value to the u16NSamples
            variable stored within the filter structure. This number represents
            the number of filtered points as a power of 2 as follows:
            \f[
                n_p = 2^{u16NSamples}    \qquad  0 \leq u16NSamples \leq 31
            \f]

@note       This function shall not be called together with #GDFLIB_FilterMA
            unless periodical clearing of filter buffers is required.


@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gdflib.h"

GDFLIB_FILTER_MA_T trMyMA = GDFLIB_FILTER_MA_DEFAULT;

void main(void)
{
    GDFLIB_FilterMAInit(&trMyMA);
}
\endcode

@par Performance:
            \anchor tab1_GDFLIB_FilterMAInit
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GDFLIB_FilterMAInit function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>2/6</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>10/10</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>9/8</td>
            </tr>
            </table>

******************************************************************************/
void GDFLIB_FilterMAInitANSIC(GDFLIB_FILTER_MA_T *pParam)
{
    pParam->s32Acc  = 0;
}

/**************************************************************************//*!
@ingroup    GDFLIB_GROUP

@brief      This function implements moving average recursive filter.

@param[in]      s32In       Value of input signal to be filtered in step (k).
                            The value is 32 bit number in Q1.31 format.

@param[in,out]  *pParam     Pointer to filter structure with filter accumulator and
                            filter parameters

@return     The function returns 32-bit value in format Q1.31, representing
            the filtered value of the input signal in step (k).

@details    The #GDFLIB_FilterMAANSIC function, denoting ANSI-C compatible
            implementation, can be called via function alias #GDFLIB_FilterMA.

            \par
            This function calculates a recursive form of an average
            filter. The filter calculation consist of the following equations:
            \f[
                acc(k) = acc(k-1) + x(k)
            \f]
            
            \f[
                y(k) = \frac{acc(k)}{n_p}
            \f]

            \f[
                acc(k) \leftarrow acc(k)-y(k)
            \f]

            where x(k) is the actual value of the input signal, acc(k) is the
            internal filter accumulator, y(k) is the actual filter output and
            \f$n_p\f$ is the number of points in the filtered window. The size
            of the filter window (number of filtered points) shall be
            defined prior to this function call. The number of the
            filtered points is defined by assigning a value to the u16NSamples
            variable stored within the filter structure. This number represents
            the number of filtered points as a power of 2 as follows:
            \f[
                n_p = 2^{u16NSamples}    \qquad  0 \leq u16NSamples \leq 31
            \f]

@note       The size of the filter window (number of filtered points) must be
            defined prior to this function call and must be equal to or greater
            than 0 and equal or smaller than 31 \f$(0 \leq u16NSamples \leq 31)\f$.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gdflib.h"

tFrac32 s32Input;
tFrac32 s32Output;

GDFLIB_FILTER_MA_T trMyMA = GDFLIB_FILTER_MA_DEFAULT;

void main(void)
{
    // input value = 0.25
    s32Input  = FRAC32(0.25);

    // filter window = 2^5 = 32 samples
    trMyMA.u16NSamples   = 5;
    GDFLIB_FilterMAInit(&trMyMA);

    // output should be 0x1000000 = FRAC32(0.0078125)
    s32Output = GDFLIB_FilterMA(s32Input,&trMyMA);
}
\endcode

@par Performance:
            \anchor tab1_GDFLIB_FilterMA
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GDFLIB_FilterMA function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>130/118</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>48/51</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>47/48</td>
            </tr>
            </table>
******************************************************************************/
tFrac32 GDFLIB_FilterMAANSIC(tFrac32 s32In, GDFLIB_FILTER_MA_T *pParam)
{
    register tFrac32 s32temp;

    // Input scaled such as to avoid saturation of the algorithm
    pParam->s32Acc      = F32AddSat(pParam->s32Acc,s32In>>12);
    s32temp             = (pParam->s32Acc)>>(pParam->u16NSamples);
    pParam->s32Acc      = F32SubSat(pParam->s32Acc,s32temp);

    return(F32ShlSat(s32temp,12));
}

#ifdef __cplusplus
}
#endif

/* End of file */
