/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDEDRRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTA BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>
#include "fsl_device_registers.h"

#include "fsl_mcglite_hal.h"
#include "fsl_sim_hal.h"
#include "fsl_flexio_hal.h"
#include "fsl_port_hal.h"

#define FLEXIO_CLK (2000000)
#define MAX_FREQ   (FLEXIO_CLK/2)   // 1MHz
#define MIN_FREQ   (FLEXIO_CLK/256) // 8KHz

static void flexio_init(void)
{
    // enable FlexIO clock gating
    // 48MHz IRC as clock source
    SIM_HAL_EnableClock(SIM, kSimClockGateFlexio0);
    CLOCK_HAL_SetFlexioSrc(SIM, 0, kClockFlexioSrcMcgIrClk);

    // Reset the FlexIO.
    FLEXIO_HAL_SetFlexioEnableCmd(FLEXIO, false); // Disable the Flexio first.
    FLEXIO_HAL_SetSoftwareResetCmd(FLEXIO, true);
    FLEXIO_HAL_SetSoftwareResetCmd(FLEXIO, false);
    // Enable FlexIO work in debug mode, Fase access.
    FLEXIO_HAL_SetDebugModeCmd(FLEXIO, true);
    FLEXIO_HAL_SetFastAccessCmd(FLEXIO, true);
    FLEXIO_HAL_SetFlexioEnableCmd(FLEXIO, true);
}

static void flexio_pwm_init(uint32_t freq, uint8_t duty)
{
    flexio_timer_config_t fxioTimerCfg = {0};
    uint32_t low, high, i;

    // check param
    if (freq > MAX_FREQ || freq < MIN_FREQ)
    {
        return;
    }
    if (duty > 99 || duty == 0)
    {
        duty = 50;
    }

    // Configure the timer 0 for generating PWM
    fxioTimerCfg.trgsel = FLEXIO_HAL_TIMER_TRIGGER_SEL_SHIFTnSTAT(0);
    fxioTimerCfg.trgpol = kFlexioTimerTriggerPolarityActiveLow;
    fxioTimerCfg.trgsrc = kFlexioTimerTriggerSourceInternal;
    fxioTimerCfg.pincfg = kFlexioPinConfigOutput;
    fxioTimerCfg.pinsel = 0; // FXIO_D0
    fxioTimerCfg.pinpol = kFlexioPinActiveHigh;
    //fxioTimerCfg.timod  = kFlexioTimerModeDual8BitPWM; /* To generate PWM. */
    fxioTimerCfg.timod  = kFlexioTimerModeDisabled; /* To generate PWM. */
    fxioTimerCfg.timout = kFlexioTimerOutputOneNotAffectedByReset;
    fxioTimerCfg.timdec = kFlexioTimerDecSrcOnFlexIOClockShiftTimerOutput;
    fxioTimerCfg.timrst = kFlexioTimerResetNever;
    fxioTimerCfg.timdis = kFlexioTimerDisableNever;
    fxioTimerCfg.timena = kFlexioTimerEnabledAlways;
    fxioTimerCfg.tstop  = kFlexioTimerStopBitDisabled;
    fxioTimerCfg.tstart = kFlexioTimerStartBitDisabled;

    /*
     * Set TIMCMP[15:8] = low output period - 1
     * Set TIMCMP[7:0] = high output period - 1
     */
    high = (FLEXIO_CLK / freq) * duty / 100;
    low  = (FLEXIO_CLK / freq) * (100 - duty) / 100;
    fxioTimerCfg.timcmp = ((low - 1) << 8) | (high - 1);
    FLEXIO_HAL_ConfigureTimer(FLEXIO, 0, &fxioTimerCfg);
}

static void flexio_pwm_start(void)
{
    FLEXIO_BWR_TIMCTL_TIMOD(FLEXIO, 0, kFlexioTimerModeDual8BitPWM);
}

static void flexio_pwm_stop(void)
{
    FLEXIO_BWR_TIMCTL_TIMOD(FLEXIO, 0, kFlexioTimerModeDisabled);
}

/*!
 * @brief example of FlexIO simulate PWM
 *
 */
int main(void)
{
    uint32_t duty = 100, i;

    // enable the PORTD clock
    SIM_HAL_EnableClock(SIM, kSimClockGatePortD);
    // PTD0, FXIO_D0 as PWM output pin
    PORT_HAL_SetMuxMode(PORTD, 0u, kPortMuxAlt6);

    // init flexio IP module with clock
    flexio_init();

    while (1)
    {
        flexio_pwm_init(8000, --duty);
        flexio_pwm_start();
        // delay for a while to check PWM
        for (i = 0; i < 100000; i++)
        {
            asm("nop");
        }
        if (duty == 0)
        {
            duty = 100;
        }
    }
}
