#include "typedefs.h"
#include "mc9s12zvml128.h"
#include "S12ZVM_devconfig.h"
#include "mlib.h"
#include "gflib.h"
#include "SWLIBS_Config.h"
#include "freemaster.h"
#include "lin.h"
#if _TL_FRAME_SUPPORT_ == _TL_MULTI_FRAME_
#include "lin_lin21tl_api.h"
#endif



/*****************************************************************************
*
* Definitions
*
*****************************************************************************/
#define APP_INIT                        0     /* Application states */
#define APP_ALIGNMENT                   1
#define APP_START                       2
#define APP_RUN                         3
#define APP_STOP                        4
#define APP_ERROR                       5

#define ALIGNMENT_TIME					200		// 200 ms Alignment time
#define START_FIRST_PERIOD				(781*40)        // max 781 * 41.9 (TimerCh0 uses 1.28us clock period -> 781 = 1ms)
#define START_CMT_CNT					8
#define START_CMT_ACCELERATION	        FRAC16(0.7711)  // used in open loop START UP as accelerator constant (commutation_period = commutation_period * 0.9)
#define ADVANCE_ANGLE					FRAC16(0.3815)   // should be 0.5 (next_cmt = TimerCNTwhileZCfound + ZC_period * ADVANCE_ANGLE)
#define TIME_TOFF						220		          // 220 * 1.28us = 300us -> time after CMT where Back-EMF is not sensed
#define TIME_TOFF_FRAC					FRAC16(0.28)
#define BEMF_SENSING_POINT_FRAC         FRAC16(0.41)	// Back-EMF sensing point = duty_cycle * BEMF_SENSING_POINT_FRAC * 2   [PMF clock = PTU clock * 2]

#define PTU_TOT_TRIGGERs_GEN_NO         2
#define PTU_TOT_LISTS_NO                2

#define DEBOUNCE_VALUE					20	// push buttons and start/stop switch timing
#define LED_TIMER_MS				 	50

#define STALLCHECK_MAX_ERRORS           6

#define RUN_PI_UPPER_LIMIT				0.95
#define RUN_PI_LOWER_LIMIT				0.175
#define RUN_PI_SPEED_PARAM_CC1			0.01
#define RUN_PI_CURRENT_PARAM_CC1		0.05

#define ALIGN_CURRENT_SCALED			8000	// 5A
#define ALIGN_PI_UPPER_LIMIT			0.4		// 40% duty cycle
#define ALIGN_PI_LOWER_LIMIT			0.02	// 1 us duty cycle (2%)
#define ALIGN_PI_PARAM_CC1				0.02


#define DC_BUS_CURRENT_LIMIT			8000	// 5A

#define DOWN_BUTTON						PTP_PTP2
#define UP_BUTTON						PTP_PTP1
#define TOGGLE_SWITCH					PTT_PTT1
#define CURRENT_LIMIT_LED				PTS_PTS4
#define FAULT_LED						PTS_PTS5


//#define _LOW_SPEED_SCALING
#define _HIGH_SPEED_SCALING

// PLL settings
/*
// Internal clock 1MHz, 25/12.5 MHz CPU/Bus clock, 6.25 MHz ADC clock
#define _INTERNAL_CLOCK						// 1 MHz internal clock is used
#define	CPMU_REFDIV		0
#define	CPMU_SYNDIV		24
#define	CPMU_POSTDIV	1
#define	CPMU_REFFRQ		0
#define	CPMU_VCOFRQ		1
#define ADC_TIM			0
#define	MIN_ADC_TRIGGER_FIRST	8
#define	MIN_ADC_TRIGGER_SECOND	48
#define	PWM_MODULO		1250
#define	PWM_DEADTIME	13
#define TIM_PRESCALER	4 		// Timer prescaler 16; 12.5 MHz/16 = 1.28 us
#define TIMER_1MS		781
#define	SCI_BAUDRATE	1302
*/
/*
// Internal clock 1MHz, 50/25 MHz CPU/Bus clock, 6.25 MHz ADC clock
#define _INTERNAL_CLOCK						// 1 MHz internal clock is used
#define	CPMU_REFDIV		0
#define	CPMU_SYNDIV		24
#define	CPMU_POSTDIV	0
#define	CPMU_REFFRQ		0
#define	CPMU_VCOFRQ		1
#define ADC_TIM			1
#define	MIN_ADC_TRIGGER_FIRST	16
#define	MIN_ADC_TRIGGER_SECOND	96
#define	PWM_MODULO		2500
#define	PWM_DEADTIME	25
#define TIM_PRESCALER	5 		// Timer prescaler 32; 25 MHz/32 = 1.28 us
#define TIMER_1MS		781
#define	SCI_BAUDRATE	2604
*/
/*
// Internal clock 1MHz, 100/50 MHz CPU/Bus clock, 8.33 MHz ADC clock
#define _INTERNAL_CLOCK						// 1 MHz internal clock is used
#define	CPMU_REFDIV		0
#define	CPMU_SYNDIV		49
#define	CPMU_POSTDIV	0
#define	CPMU_REFFRQ		0
#define	CPMU_VCOFRQ		3
#define ADC_TIM			2
#define	MIN_ADC_TRIGGER_FIRST	24
#define	MIN_ADC_TRIGGER_SECOND	144
#define	PWM_MODULO		5000
#define	PWM_DEADTIME	50
#define TIM_PRESCALER	6 		// Timer prescaler 64; 50 MHz/64 = 1.28 us
#define TIMER_1MS		781
#define	SCI_BAUDRATE	5208
*/

// External clock 4MHz, 25/12.5 MHz CPU/Bus clock, 6.25 MHz ADC clock
#define _EXTERNAL_CLOCK
#define	CPMU_REFDIV		3
#define	CPMU_SYNDIV		24
#define	CPMU_POSTDIV	1
#define	CPMU_REFFRQ		0
#define	CPMU_VCOFRQ		1
#define ADC_TIM			0
#define	MIN_ADC_TRIGGER_FIRST	8
#define	MIN_ADC_TRIGGER_SECOND	48
#define	PWM_MODULO		1250
#define	PWM_DEADTIME	13
#define TIM_PRESCALER	4 		// Timer prescaler 16; 12.5MHz/16 = 1.28 us
#define TIMER_1MS		781
#define	SCI_BAUDRATE	1302

/*
// External clock 4MHz, 50/25 MHz CPU/Bus clock, 6.25 MHz ADC clock
#define _EXTERNAL_CLOCK
#define	CPMU_REFDIV		3
#define	CPMU_SYNDIV		24
#define	CPMU_POSTDIV	0
#define	CPMU_REFFRQ		0
#define	CPMU_VCOFRQ		1
#define ADC_TIM			1
#define	MIN_ADC_TRIGGER_FIRST	16
#define	MIN_ADC_TRIGGER_SECOND	96
#define	PWM_MODULO		2500
#define	PWM_DEADTIME	25
#define TIM_PRESCALER	5 		// Timer prescaler 32; 25 MHz/32 = 1.28 us
#define TIMER_1MS		781
#define	SCI_BAUDRATE	2604
*/
/*
// External clock 4MHz, 100/50 MHz CPU/Bus clock, 8.33 MHz ADC clock
#define _EXTERNAL_CLOCK
#define	CPMU_REFDIV		1
#define	CPMU_SYNDIV		24
#define	CPMU_POSTDIV	0
#define	CPMU_REFFRQ		0
#define	CPMU_VCOFRQ		3
#define ADC_TIM			2
#define	MIN_ADC_TRIGGER_FIRST	24
#define	MIN_ADC_TRIGGER_SECOND	144
#define	PWM_MODULO		5000
#define	PWM_DEADTIME	50
#define TIM_PRESCALER	6 		// Timer prescaler 64; 50 MHz/64 = 1.28 us
#define TIMER_1MS		781
#define	SCI_BAUDRATE	5208
*/

#ifdef _HIGH_SPEED_SCALING
/***** High speed scaling *********************
 * For 4 pole-pairs motors
 * Boundary values for 1.28 us timer period:
 * - Minimal speed:        60 rpm
 * - Minimal speed scaled: 86
 * - Maximal speed:        22887 rpm
 * - Maximal speed scaled: 32767
 *********************************************/
#define SPEED_CALC_NUMERATOR            12788281
#define STALLCHECK_MIN_CMT_PERIOD		97			// 97 = 125 us (20 krpm on 4 pole-pairs motor), min value = 85 (22.89 krpm)
#define REQUIRED_RUN_SPEED				852		// 1431.655765 = 1000 rpm
#define MIN_SPEED						(852/2)	// 500 rpm minimal speed for Down button control (should be min 10% of nominal motor speed)
#define MAX_SPEED						(852*15)	// 13 krpm maximum speed for Up button control
#define SPEED_STEP						(852/2)	// 500 rpm, Up/Down step for button controls
//*****
#endif

#ifdef _LOW_SPEED_SCALING
/****** Low speed scaling *********************
 * For 2 pole-pairs motors
 * Boundary values for 1.28 us timer period:
 * - Minimal speed:        120 rpm
 * - Minimal speed scaled: 341
 * - Maximal speed:        11444 rpm
 * - Maximal speed scaled: 32767
 *********************************************/
#define SPEED_CALC_NUMERATOR            67108864
#define STALLCHECK_MIN_CMT_PERIOD		391			// 391 = 500 us (~10k rpm on 2 pole-pairs motor), min value = 341 (11.44 krpm)
#define REQUIRED_RUN_SPEED				2863		// 2863.311531 = 1000 rpm
#define MIN_SPEED						(2863/2)	// 500 rpm minimal speed for Down button control (should be min 10% of nominal motor speed)
#define MAX_SPEED						(2863*5)	// 12 krpm maximum speed for Up button control
#define SPEED_STEP						(2863/4)	// 250 rpm, Up/Down step for button controls
//*****
#endif

/*****************************************************************************
*
* Types Definition
*
*****************************************************************************/
typedef void (*tPointerFcn)(void); /* pointer to function */

typedef union uDriveStatus {
  unsigned int word;
  struct
  {
    unsigned int Alignment        :1;
    unsigned int Sensorless       :1;
    unsigned int DisableAdc	   :1;
    unsigned int StallCheckReq    :1;
    unsigned int EnableCMT        :1;
    unsigned int AfterCMT	   :1;
    unsigned int CloseLoop        :1;
    unsigned int NewZC		   :1;
    unsigned int AdcSaved			:1;
    unsigned int CurrentLimiting  :1;
    unsigned int Fault             :1;
  }bit;
}tDriveStatus;

typedef union uInterfaceStatus {
  unsigned char  byte;
  struct
    {
    unsigned char Switch           :1;
    unsigned char SwitchChange     :1;
    unsigned char Up               :1;
    unsigned char Down             :1;
    unsigned char bit4             :1;
    unsigned char bit5             :1;
    unsigned char bit6             :1;
    unsigned char bit7             :1;
  }bit;
}tInterfaceStatus;

typedef union uDriveDemand
  {
  unsigned char  byte;
  struct
    {
    unsigned char StopRun          :1;
    unsigned char bit1             :1;
    unsigned char bit2             :1;
    unsigned char bit3             :1;
    unsigned char bit4             :1;
    unsigned char bit5             :1;
    unsigned char bit6             :1;
    unsigned char bit7             :1;
    }bit;
}tDriveDemand;

/*****************************************************************************
*
* Constants Definition
*
*****************************************************************************/
const char MaskVal[6]   = {0x34,0x1c, 0x13,0x31, 0x0d,0x07}; // {0x30,0x0c, 0x03,0x30, 0x0c,0x03};
const char OutCtl[6]    = {0x0c,0x30, 0x30,0x03, 0x03,0x0c};
const char BemfPhase[6] = {0x03,0x02,0x01,0x03,0x02,0x01};

/*
Phase Voltage sensing select bits:
00 Pin HD selected , VHD / 12 connected to ADC channel
01 Pin HS0 selected , VHS0 / 6 connected to ADC channel
10 Pin HS1 selected , VHS1 / 6 connected to ADC channel
11 Pin HS2 selected, VHS2 / 6 connected to ADC channel
*/

/*****************************************************************************
*
* Variables Definition
*
*****************************************************************************/
unsigned int speed = 0;
volatile tU16 NextCmtPeriod, duty_cycle;
volatile tU16 NextCmtSector, ActualCmtSector;
volatile tU16 alignmentTimer, startCMTcounter;

volatile unsigned char appState;
volatile tDriveStatus driveStatus;

volatile unsigned char phaseStatus;

volatile tFrac16 DCBusCurrentOffset, DCBusCurrent, DCBusCurrentZC, DCBusCurrentFiltered;
volatile tU16 phaseVoltage, DCBusVoltageHalf;
volatile short bemfVoltage, bemfVoltageOld;
volatile tU16 timeBEMF, timeOldBEMF, timeCommutation, timeZCToff;
volatile tU16 timeZC, lastTimeZC;
volatile tU16 actualPeriodZC;
volatile tU16 periodZC_F_PhA, periodZC_R_PhA;
volatile tU16 periodZC_F_PhB, periodZC_R_PhB;
volatile tU16 periodZC_F_PhC, periodZC_R_PhC;
volatile unsigned long period6ZC, periodZcAvrg;
volatile tU16 advanceAngle;

volatile tU16 debugPeriodMin, debugPeriodMax, stallCheckCounter;
volatile unsigned char StallError;

volatile tFrac16 speedErr, requiredSpeed, currentErr, DCBusCurrentLimit;
volatile tFrac32 actualSpeed;
volatile tFrac16 speedPIOut, currentPIOut;
volatile GFLIB_CONTROLLER_PIAW_R_T_F16 speedPIPrms, currentPIPrms, AlignCurrentPIPrms;

volatile int AdcErrorLDOK, AdcErrorRSTAR, AdcErrorTRIG, AdcErrorEOL, AdcErrorCMD, AdcErrorIA;

volatile tInterfaceStatus interfaceStatus;
volatile tDriveDemand driveDemand, driveDemandRemote;

volatile unsigned char GduFlags;

void initCPMU(void);
void initGPIO(void);
void initTIM(void);
void initPMF(void);
void initGDU(void);
void initPTU(void);
void initADC(void);
void initSCI(void);
void initInt(void);
void AdcCalibration(void);

void UpdateDutycycle(void);
void StallCheck(void);
void CheckManualInterface(void);
void ZCdetectPhCfallingAdc(void);
void ZCdetectPhBraisingAdc(void);
void ZCdetectPhAfallingAdc(void);
void ZCdetectPhCraisingAdc(void);
void ZCdetectPhBfallingAdc(void);
void ZCdetectPhAraisingAdc(void);
void AppStartToRun(void);
void AppRunToStop(void);
void AppAlignmentToStart(void);
void AppStopToAlignment(void);
void AppError(void);
void AppStop(void);
void AppRun(void);
void AppStart(void);
void AppAlignment(void);
void AppInit(void);
void UpdateCurrentLimitingLed(void);
void CheckFaults(void);
void UpdateDutycycle(void);

// Back-EMF sensing based on ADC
void ZCdetectPhCfallingAdc(void);
void ZCdetectPhBraisingAdc(void);
void ZCdetectPhAfallingAdc(void);
void ZCdetectPhCraisingAdc(void);
void ZCdetectPhBfallingAdc(void);
void ZCdetectPhAraisingAdc(void);
/*
INTERRUPT void TIMchan0_ISR(void);
INTERRUPT void TIMchan3_ISR(void);
INTERRUPT void ADC1done_ISR(void);
INTERRUPT void PMFfault_ISR(void);
INTERRUPT void PMFreloadA_ISR(void);
INTERRUPT void PMFreloadA_ISR(void);
INTERRUPT void PTUTrigger0Done_ISR(void);
INTERRUPT void PMFreloadoverrun_ISR(void);
INTERRUPT void PTUTrigger1Done_ISR(void);
INTERRUPT void ADC0error_ISR(void);
INTERRUPT void ADC0done_ISR(void);
INTERRUPT void ADC1error_ISR(void);
*/
static tPointerFcn AppStateMachine[] = {
                                     AppInit,
                                     AppAlignment,
                                     AppStart,
                                     AppRun,
                                     AppStop,
                                     AppError
                                   };

/****************************************************************************
 *  Trigger event list
 *
 *  +-------------------+ - (int)PTUTriggerEventList)
 *  +===================+ - Trigger 0 space
 *  +*******************+ - List 0 Offset from PTUTriggerEventList = 0 = (int)&PTUTriggerEventList[0][0][0] - (int)PTUTriggerEventList
 *  | DelayT0           |
 *  +-------------------+
 *  | DelayT1           |
 *  +-------------------+
 *  | 0x0000            | - End of delay List 0
 *  +*******************+ - List 1 Offset from PTUTriggerEventList = 3 = (int)&PTUTriggerEventList[0][1][0] - (int)PTUTriggerEventList
 *  | DelayT0           |
 *  +-------------------+
 *  | DelayT1           |
 *  +-------------------+
 *  | DelayT2           |
 *  +-------------------+
 *  | 0x0000            | - End of delay List 1
 *  +===================+ - Trigger 1 space
 *  +*******************+ - List 0 Offset from PTUTriggerEventList = 8 = (int)&PTUTriggerEventList[1][0][0] - (int)PTUTriggerEventList
 *  | DelayT0           |
 *  +-------------------+
 *  | DelayT1           |
 *  +-------------------+
 *  | 0x0000            | - End of delay List 0
 *  +*******************+ - List 1 Offset from PTUTriggerEventList = 12 = (int)&PTUTriggerEventList[1][1][0] - (int)PTUTriggerEventList
 *  | DelayT0           |
 *  +-------------------+
 *  | DelayT1           |
 *  +-------------------+
 *  | DelayT2           |
 *  +-------------------+
 *  | 0x0000            | - End of delay List 1
 *  +===================+
 *
 *
 ***************************************************************************/
PR_SECTION(ptuTrigE)
  volatile short PTUTriggerEventList[PTU_TOT_TRIGGERs_GEN_NO][PTU_TOT_LISTS_NO][3] = {
                                              {   /*DelayT0 */
                                                  {0x0100,0x0200,0x0000},{0x0000,0x0000,0x0000}
                                              },
                                              {
                                            	  {0x0100,0x0000,0x0000},{0x0000,0x0000,0x0000}
                                              }
                                            };

/****
 * CMD_SEL[1:0]:
 * 00 normal conversion
 * 40 end of sequence
 * 80 end of list, wrap to top, continue
 * C0 end of list, stop
 *
 */
PR_SECTION(adcLists)
  volatile char ADC0CommandList[6][4] = {
	  {0x40,0xD0,0x00,0x00},	// current sense channel, end of sequence [D0]
	  {0xC0,0xCB,0x00,0x00},	// end of list + no int [C0], HD voltage [CB], 4 clock cycles sample time [00], reserved [00]
	  {0x00,0x00,0x00,0x00},
	  {0x00,0x00,0x00,0x00},
	  {0x00,0x00,0x00,0x00},
      {0x00,0x00,0x00,0x00}
  };
PR_SECTION(adcLists)
  volatile char ADC1CommandList[6][4] = {
	  {0xC0,0xCA,0x00,0x00},	// [end of list] + no int [C0], phase voltage [CA], 4 clock cycles sample time [00], reserved [00]
	  {0x00,0x00,0x00,0x00},
	  {0x00,0x00,0x00,0x00},
	  {0x00,0x00,0x00,0x00},
	  {0x00,0x00,0x00,0x00},
      {0x00,0x00,0x00,0x00}
  };

  volatile unsigned short ADC0ResultList[6] = {0, 0, 0, 0, 0, 0};
  volatile unsigned short ADC1ResultList[6] = {0, 0, 0, 0, 0, 0};
PR_SECTION(DEFAULT_SEC)

static tPointerFcn ZCdetectionAdc[] = {
                                      ZCdetectPhCfallingAdc,
                                      ZCdetectPhBraisingAdc,
                                      ZCdetectPhAfallingAdc,
                                      ZCdetectPhCraisingAdc,
                                      ZCdetectPhBfallingAdc,
                                      ZCdetectPhAraisingAdc
                                   };


/*****************************************************************************
*
* Function: void initCPMU(void)
*
* Description: Clock, Reset and Power Management Unit configuration
*
*****************************************************************************/
void initCPMU(void)
{
	// Wait for stable supply after power up
	while (GDUF_GLVLSF)
		GDUF_GLVLSF = 1;

	CPMUREFDIV_REFDIV = CPMU_REFDIV;
	CPMUREFDIV_REFFRQ = CPMU_REFFRQ;
	CPMUSYNR_SYNDIV = CPMU_SYNDIV;
	CPMUSYNR_VCOFRQ = CPMU_VCOFRQ;
	CPMUPOSTDIV_POSTDIV = CPMU_POSTDIV;

#ifdef _EXTERNAL_CLOCK
	CPMUOSC_OSCE = 1;
	while (CPMUIFLG_UPOSC == 0) {}; // Wait for oscillator to start up (UPOSC=1) and PLL to lock (LOCK=1)
#endif

	while (CPMUIFLG_LOCK == 0) {};
	CPMURFLG  = 0x60; 	//Clear PORF and LVRF
}


void Init_LIN(void){
	
    LP0CR_LPPUE = 1;    /* Pull up to strong signal  */
    LP0SLRM = 0x01;		/* LIN Slew Rate selection */
    					/* With the LIN baudrate selected as 9600bps, 
                           the Slew Rate is set as 01 which is optimized for 10,400 Bit/s*/
    
    LP0CR_LPE = 1;      /* Enable LIN Phy  */

}

void Init_PIM(void){	

	MODRR0 = (MODRR0_S0LRR_MASK && 0b000) | (MODRR0_SCI1RR_MASK); 
}
/*****************************************************************************
*
* Function: void initGPIO(void)
*
* Description: I/O pins configuration
*
*****************************************************************************/
void initGPIO(void)
{
  MODRR0_SCI1RR = 1; // serial port

  DDRS_DDRS4 = 1;	// DC Bus current limit LED
  DDRS_DDRS5 = 1;	// Fault LED
}

/*****************************************************************************
*
* Function: void initTIM(void)
*
* Description: Timer module configuration
*
*****************************************************************************/
void initTIM(void){

  TIM0TIOS_IOS0 = 1;
  TIM0TIOS_IOS3 = 1;      // Set channel 1 as output compare

  TIM0TSCR1_PRNT = 0; // NON Precision timer

  TIM0TCTL2_OL0 = 1;  // Toggle output of tim chan 0
  TIM0TCTL2_OM0 = 0;
  TIM0TCTL2_OL3 = 0;  // No action on output of tim chan 3
  TIM0TCTL2_OM3 = 0;

  TIM0TIE_C0I = 1;    // Enable Interrupts
  TIM0TIE_C3I = 1;    // Enable Interrupts

  TIM0TSCR2_PR = TIM_PRESCALER;

  TIM0TC3 = TIMER_1MS;   // Interrupt to be every ~ 1[ms]

  TIM0OCPD = 0xff;        // disconnect all output compare pins

  TIM0TSCR1_TEN = 1;
}

/*****************************************************************************
*
* Function: void initPMF(void)
*
* Description: PMF module configuration
*
*****************************************************************************/
void initPMF(void){

  PMFCFG0_EDGEA = 1;
  PMFCFG0_EDGEB = 1;
  PMFCFG0_EDGEC = 1;

  PMFCFG2_REV0 = 1;   // 01 PWM generator A generates reload event.
  PMFCFG2_REV1 = 0;

  PMFOUTB = 0x2A;		//  Low MOSFETs ON while SW control (Unipolar PWM)

  PMFCFG3_VLMODE = 0x01;	// Writing to value register zero also writes to value registers one to five

  //PMFFEN = 0 // Fault disabled
  // .... (other fault registers)

  PMFFQCA = 0;              // Reload every PWM, Half-cycle disabled, fcore / 1

  PMFMODA = PWM_MODULO;
  PMFDTMA = PWM_DEADTIME;

  PMFCFG2 |= 0x3f;			// mask all PWM outputs
  PMFOUTC_OUTCTL = 0x3f;		// all outputs in Software mode
  PMFENCA_LDOKA = 1;			// apply PMF Modulo value

  PMFENCA_RSTRTA = 1;         // 1 = PWM restart at commutation event

  PMFENCA_PWMENA = 1;

  //PMFENCA_PWMRIEA = 1;		// Reload Interrupt - Used only for debugging

  PMFCFG1_ENCE = 1;	// buffered PMFOUTC, PMFOUTB, SWAPx and MSKx (change on commutation event)

  PMFENCA_GLDOKA = 1;         // 0 = Local LDOKA controls buffered registers / 1 = external Load OK controls buffered registers
}


/*****************************************************************************
*
* Function: void initGDU(void)
*
* Description: GDU module configuration
*
*****************************************************************************/
void initGDU(){

	  GDUE_GCPE = 1; // charge pump

	  GDUE_GCSE0 = 1; // enable Current Sense Amplifier 0

	  GDUF = 0xff;	// Flag Register - clear all error flags

	  //GDUCLK1 = ... // Boost option clock
	  //GDUE_GBOE = 1; // enable Boost Converter

	  //GDUBCL = ... // Boost Current Limit Register

	  GDUCLK2_GCPCD = 2;  // Fbus / 32

	  GDUCTR = 0x13;	// blanking time

	  GDUDSLVL = 0x77;	// desat. level

	  GDUE_GFDE = 1; // enable FET pre-driver

	  GDUDSE = 0x77;	// Clear Desaturation Error Flags
}

/*****************************************************************************
*
* Function: void initPTU(void)
*
* Description: Programmable Trigger Unit module configuration
*
*****************************************************************************/
void initPTU(void){
	//PTUIEL_TG0DIE = 1; 	// Trigger Generator 0 Done Interrupt Enable - Used only for debugging
	PTUIEL_TG1DIE = 1; 	// Trigger Generator 1 Done Interrupt Enable - store Adc measurement time

	PTUPTRL = (uint8_t)((long)PTUTriggerEventList);
	PTUPTRM = (uint8_t)(((long)PTUTriggerEventList) >> 8);
	PTUPTRH = (uint8_t)(((long)PTUTriggerEventList) >> 16);

	TG0L1IDX = (uint8_t)(((long)&PTUTriggerEventList[0][0][0] - (long)PTUTriggerEventList) >> 1); // same as TG0L0IDX
    TG1L0IDX = (uint8_t)(((long)&PTUTriggerEventList[1][0][0] - (long)PTUTriggerEventList) >> 1);
    TG1L1IDX = (uint8_t)(((long)&PTUTriggerEventList[1][0][0] - (long)PTUTriggerEventList) >> 1); // same as TG1L0IDX

	PTUE_TG0EN = 1;       // Enable Trigger Generation 0
	PTUE_TG1EN = 1;       // Enable Trigger Generation 1

	PTUC_PTULDOK = 1;     // Switch list when next reload event
}

/*****************************************************************************
*
* Function: void initAdc(void)
*
* Description: ADC module configuration
*
*****************************************************************************/
void initADC(void){

  /****************************************
   * ADC0
   ****************************************/

  ADC0CTL_0_ACC_CFG = 3;      // Dual access mode
  ADC0CTL_0_STR_SEQA = 1;     // Store result at abort/restart

  //ADC0CTL_1 = 0;
  ADC0TIM = ADC_TIM;          // clock: clk = fbus / (2x(reg.value + 1)) [0.25 - 8MHz]

  ADC0FMT_DJM = 0;            // left justified result data
  ADC0FMT_SRES = 4;           // 12-bit result

  ADC0CONIE_1_EOL_IE = 1;     // End of list interrupt enable

  // ADC0 Command Base Pointer
  ADC0CBP_0 = (uint8_t)(((long)ADC0CommandList) >> 16);
  ADC0CBP_1 = (uint8_t)(((long)ADC0CommandList) >> 8);
  ADC0CBP_2 = (uint8_t)((long)ADC0CommandList);

  // ADC0 Result Base Pointer
  ADC0RBP_0 = (uint8_t)(((long)ADC0ResultList) >> 16);
  ADC0RBP_1 = (uint8_t)(((long)ADC0ResultList) >> 8);
  ADC0RBP_2 = (uint8_t)((long)ADC0ResultList);

  // ADC0 Command/Result Offset registers
  ADC0CROFF1 = 0;

  ADC0CTL_0_ADC_EN = 1;       // enable ADC0
  ADC0EIE = 0xEE;     // enable all errors interrupts
  //ADC0EIE = 0x08;     // Enable Trigger Error ISR only


  /****************************************
   * ADC1
   ****************************************/

  ADC1CTL_0_ACC_CFG = 3;      // Dual access mode
  ADC1CTL_0_STR_SEQA = 1;     // Store result at abort/restart

  //ADC1CTL_1 = 0;
  ADC1TIM = ADC_TIM;          // clock: clk = fbus / (2x(reg.value + 1)) [0.25 - 8MHz]

  ADC1FMT_DJM = 0;            // left justified result data
  ADC1FMT_SRES = 4;           // 12-bit result

  //ADC1CONIE_1_EOL_IE = 1;     // End of list interrupt enable - Used only for debugging

  // ADC1 Command Base Pointer
  ADC1CBP_0 = (uint8_t)((((long)ADC1CommandList) >> 16) & 0x0000FF);
  ADC1CBP_1 = (uint8_t)((((long)ADC1CommandList) >> 8) & 0x0000FF);
  ADC1CBP_2 = (uint8_t)((long)ADC1CommandList & 0x0000FF);

  // ADC1 Result Base Pointer
  ADC1RBP_0 = (uint8_t)((((long)ADC1ResultList) >> 16) & 0x0000FF);
  ADC1RBP_1 = (uint8_t)((((long)ADC1ResultList) >> 8) & 0x0000FF);
  ADC1RBP_2 = (uint8_t)((long)ADC1ResultList & 0x0000FF);

  // ADC1 Command/Result Offset registers
  ADC1CROFF1 = 0;

  ADC1CTL_0_ADC_EN = 1;       // enable ADC1
  //ADC1EIE = 0xEE;     // enable all error interrupts
  ADC1EIE = 0x08;     // Enable Trigger Error ISR only

}

/*****************************************************************************
*
* Function: void initSCI(void)
*
* Description: SCI module configuration
*
*****************************************************************************/
void initSCI(void){
  SCI1BD	 = SCI_BAUDRATE;
  SCI1CR2_TE = 1;
  SCI1CR2_RE = 1;
  //SCICR2_RIE = 1;
}


/*****************************************************************************
*
* Function: void initInt(void)
*
* Description: Interrupt priorities configuration
*
*****************************************************************************/
void initInt(void){
  // set PTU Trigger 0 ISR (0x0E4) priority to 6
  // 0xE4 / 4 = 0x39 => reg. value = 0x38 + offset 1
  //INT_CFADDR = 0x38;
  //INT_CFDATA1 = 6;

  // set PTU Trigger 1 ISR (0x0E0) priority to 6
  // 0xE0 / 4 = 0x38 => reg. value = 0x38 + offset 0
  INT_CFADDR = 0x38;
  INT_CFDATA0 = 6;

  // set ADC 0 done ISR (0x184) priority to 6
  // 0x184 / 4 = 0x61 => reg. 0x60, offset 1
  INT_CFADDR = 0x60;
  INT_CFDATA1 = 6;

  // set Commutation ISR (0x1CC) priority to 5
  // 0x1CC / 4 = 0x73 => reg. value = 0x70 + offset 3
  INT_CFADDR = 0x70;
  INT_CFDATA3 = 5;

  // set Speed control loop ISR (0x1C0) to 4
  // 0x1C0 / 4 = 0x70 => reg. value = 0x70 + offset 0
  INT_CFADDR = 0x70;
  INT_CFDATA0 = 4;
}

/*****************************************************************************
*
* Function: void TIMchan3_ISR(void)
*
* Description: Timer channel 3 Output Compare Interrupt Service Routine
*              used to calculate speed, speed control loop and current control loop
*
*****************************************************************************/
interrupt VectorNumber_Vtim0ch3 void TIMchan3_ISR(void)
{
tFrac16 PIOut;

  EnableInterrupts;

  TIM0TC3 = TIM0TC3 + TIMER_1MS;

  if (driveStatus.bit.StallCheckReq == 1) {
      driveStatus.bit.StallCheckReq = 0;
      StallCheck();
  }

  if (driveStatus.bit.Sensorless == 1)
  {
	period6ZC = periodZC_F_PhA + periodZC_R_PhA + periodZC_F_PhB + periodZC_R_PhB + periodZC_F_PhC + periodZC_R_PhC;
	actualSpeed = SPEED_CALC_NUMERATOR / period6ZC;

	speedErr = requiredSpeed - (tFrac16) actualSpeed;
    speedPIOut = GFLIB_ControllerPIrAW(speedErr, (GFLIB_CONTROLLER_PIAW_R_T_F16 *)&speedPIPrms, F16);

    currentErr = DCBusCurrentLimit - DCBusCurrentFiltered;
    currentPIOut = GFLIB_ControllerPIrAW(currentErr, (GFLIB_CONTROLLER_PIAW_R_T_F16 *) &currentPIPrms, F16);

   
	if (currentPIOut >= speedPIOut) {
		currentPIPrms.f32Acc = ((tFrac32) speedPIOut) << 16;
		currentPIPrms.f16InErrK1 = 0;
		PIOut = speedPIOut;
		driveStatus.bit.CurrentLimiting = 0;
	} else {
		speedPIPrms.f32Acc = ((tFrac32) currentPIOut) << 16;
		speedPIPrms.f16InErrK1 = 0;
		PIOut = currentPIOut;
		driveStatus.bit.CurrentLimiting = 1;
	}

    duty_cycle = MLIB_Mul(PIOut, PWM_MODULO, F16);    // duty cycle 0-1 -> 0-PWM_MODULO
    UpdateDutycycle();
  }
  else if ((appState == APP_START) || (appState == APP_ALIGNMENT))
  {
	currentErr = ALIGN_CURRENT_SCALED - DCBusCurrent;
	PIOut = GFLIB_ControllerPIrAW(currentErr, (GFLIB_CONTROLLER_PIAW_R_T_F16 *) &AlignCurrentPIPrms, F16);

    duty_cycle = MLIB_Mul(PIOut, PWM_MODULO, F16);    // duty cycle 0-1 -> 0-PWM_MODULO
    UpdateDutycycle();
  }

  CheckManualInterface();
  UpdateCurrentLimitingLed();

  TIM0TFLG1 = TIM0TFLG1_C3F_MASK;
}



/*****************************************************************************
*
* Function: void UpdateDutycycle(void)
*
* Description: - Update of duty cycle to PMF
*              - Update of trigger delays
*
*****************************************************************************/
void UpdateDutycycle(void)
{
tU16 delay1, delay2;

  PMFVAL0 = duty_cycle;

  delay1 = duty_cycle >> 2;												// middle of the pulse - DC Bus current
  if (delay1 < MIN_ADC_TRIGGER_FIRST) delay1 = MIN_ADC_TRIGGER_FIRST;
  PTUTriggerEventList[0][0][0] = delay1;

  delay2 = MLIB_Mul(duty_cycle,BEMF_SENSING_POINT_FRAC,F16);			// end of the pulse
  PTUTriggerEventList[1][0][0] = delay2;								// ADC1 - phase voltage

  if (delay2 < (delay1 + MIN_ADC_TRIGGER_SECOND)) delay2 = delay1 + MIN_ADC_TRIGGER_SECOND;
  PTUTriggerEventList[0][0][1] = delay2;								// ADC0 second sample - DC Bus Voltage

  PTUC_PTULDOK = 1;
}

/*****************************************************************************
*
* Function: void TIMchan0_ISR(void)
*
* Description: Timer channel 0 Output Compare Interrupt Service Routine
*              - Commutation ISR
*
*****************************************************************************/
interrupt VectorNumber_Vtim0ch0 void TIMchan0_ISR(void){

  EnableInterrupts;

  timeCommutation = TIM0TCNT;

  if (driveStatus.bit.Sensorless == 1)
  {
	  if (driveStatus.bit.NewZC == 0) { // Zero-cross not found in previous commutation period
		  timeZC = timeCommutation - (actualPeriodZC >> 1);		// Middle between two commutations
		  GDUPHMUX_GPHMX = BemfPhase[NextCmtSector];	// will be applied at the next PMF Reload
		  PTUC_PTULDOK = 1;
	  }
	  TIM0TC0 = timeCommutation + (actualPeriodZC << 1);
	  timeZCToff = MLIB_Mul(actualPeriodZC, TIME_TOFF_FRAC,F16);
	  driveStatus.bit.StallCheckReq = 1;
  }
  else
  {
	  TIM0TC0 = timeCommutation + NextCmtPeriod;	// alignment + open loop
  }

  ActualCmtSector = NextCmtSector;
  if (driveStatus.bit.EnableCMT) {
	  NextCmtSector++;
	  if (NextCmtSector > 5) NextCmtSector = 0;

	  PMFCFG2 = 0x40 + MaskVal[NextCmtSector];		// will be applied at the next commutation
	  PMFOUTC_OUTCTL = OutCtl[NextCmtSector];		// will be applied at the next commutation
  }

  if (driveStatus.bit.Alignment) {	// in high speeds applications move this code to control loop timer
	  if (alignmentTimer > 0) {
		  alignmentTimer--;
	  }
  }
  	driveStatus.bit.AfterCMT = 1;
	driveStatus.bit.DisableAdc = 1;
	driveStatus.bit.NewZC = 0;
	driveStatus.bit.AdcSaved = 0;
	TIM0TFLG1 = TIM0TFLG1_C0F_MASK;
}

/*****************************************************************************
*
* Function: void PMFreloadA_ISR(void) - Used only for debugging
*
* Description: PMF Reload A Interrupt Service Routine
*
*****************************************************************************/
interrupt VectorNumber_Vpmfra void PMFreloadA_ISR(void){
volatile char tmp;
  tmp = PMFFQCA;
  PMFFQCA_PWMRFA = 1;		// Clear flag
}

/*****************************************************************************
*
* Function: void PMFfault_ISR(void)
*
* Description: PMF Fault Interrupt Service Routine
*
*****************************************************************************/
interrupt VectorNumber_Vpmffault void PMFfault_ISR(void){

	PMFFIF = PMFFIF_FIF0_MASK;		// Clear flag Fault 0, ...
}

/*****************************************************************************
*
* Function: void PMFreloadoverrun_ISR(void)
*
* Description: PMF Reload Overrun Interrupt Service Routine
*
*****************************************************************************/
interrupt VectorNumber_Vpmfro void PMFreloadoverrun_ISR(void){

	PMFROIF = PMFROIF_PMFROIFA_MASK;		// Clear flag
}

/*INTERRUPT void SCI1_ISR(void){
  if (SCI1SR1_RDRF)
  {
    volatile uint8_t tmp1;
    tmp1 = SCI1SR1;
    tmp1 = SCI1DRL;
  }
}*/



/*****************************************************************************
*
* Function: void PTUTrigger0Done_ISR(void) - Used only for debugging
*
* Description: PTU Trigger0 Done Interrupt Service Routine
*
*****************************************************************************/
interrupt VectorNumber_Vptutg0dn void PTUTrigger0Done_ISR(void)
{
	PTUIFL = PTUIFL_TG0DIF_MASK;		// Clear flag
}


/*****************************************************************************
*
* Function: void PTUTrigger1Done_ISR(void)
*
* Description: PTU Trigger1 Done Interrupt Service Routine
*
*****************************************************************************/
interrupt VectorNumber_Vptutg1dn void PTUTrigger1Done_ISR(void)
{
	timeOldBEMF = timeBEMF;
	timeBEMF = TIM0TCNT;

	PTUIFL = PTUIFL_TG1DIF_MASK;		// Clear flag
}

/*****************************************************************************
*
* Function: void ADC0error_ISR(void)
*
* Description: ADC0 Error Interrupt Service Routine
*
*****************************************************************************/
interrupt VectorNumber_Vadc0err void ADC0error_ISR(void)
{
char tmpAdcEIF;

  tmpAdcEIF = ADC0EIF;

  if (tmpAdcEIF & ADC0EIF_LDOK_EIF_MASK) {       // Load Ok Error
      AdcErrorLDOK++;
      ADC0EIF = ADC0EIF_LDOK_EIF_MASK;
  }
  if (tmpAdcEIF & ADC0EIF_RSTAR_EIF_MASK) {       // Restart Request Error
      AdcErrorRSTAR++;
      ADC0EIF = ADC0EIF_RSTAR_EIF_MASK;
  }
  if (tmpAdcEIF & ADC0EIF_TRIG_EIF_MASK) {       // Trigger Error => Soft Reset
      AdcErrorTRIG++;
      ADC0CTL_0_ADC_SR = 1;
  }
  if (tmpAdcEIF & ADC0EIF_EOL_EIF_MASK) {       // End Of List Error => Soft Reset
      AdcErrorEOL++;
      ADC0CTL_0_ADC_SR = 1;
  }
  if (tmpAdcEIF & ADC0EIF_CMD_EIF_MASK) {       // Command Value Error => Soft Reset
      AdcErrorCMD++;
      ADC0CTL_0_ADC_SR = 1;
  }
  if (tmpAdcEIF & ADC0EIF_IA_EIF_MASK) {       // Illegal Access Error => Soft Reset
      AdcErrorIA++;
      ADC0CTL_0_ADC_SR = 1;
  }
}

/*****************************************************************************
*
* Function: void ADC0done_ISR(void)
*
* Description: ADC0 Conversion Done Interrupt Service Routine
*
*****************************************************************************
*
* Benchmark:
* 1) No ADC Approximation:
*    - looking for zero-cross, not found:          4.6 us
*    - zero-cross just found:                      7.2 us
*    - not looking for zero-cross (already found): 2.4 us
* 2) ADC Approximation:
*    - looking for zero-cross, not found:          5.4 us
*    - zero-cross just found:                      9.8 us
*    - not looking for zero-cross (already found): 2.64 us
*
*****************************************************************************/
interrupt VectorNumber_Vadc0conv_compl void ADC0done_ISR(void)
{

  // RIGHT justified result ADC data
  /*
  DCBusCurrent	   = ADC0ResultList[0] - DCBusCurrentOffset;
  DCBusVoltageHalf = MLIB_Mul(ADC0ResultList[1] << 1, 27307 ,F16);	// DC Bus Voltage * 2 * 10/12
  phaseVoltage 	   = ADC1ResultList[0] << 2;						// Phase Voltage * 4
  */
  // LEFT justified result ADC data
  DCBusCurrent	   = ADC0ResultList[0] - DCBusCurrentOffset;
  DCBusVoltageHalf = MLIB_Mul(ADC0ResultList[1]>>1, 13653 ,F16);	// DC Bus Voltage * 5/6 / 2
  phaseVoltage 	   = ADC1ResultList[0]>>1;							// Phase Voltage

  bemfVoltage = phaseVoltage - DCBusVoltageHalf;

	if (driveStatus.bit.AfterCMT == 1) {
		if ((timeBEMF - timeCommutation) > timeZCToff){
			driveStatus.bit.AfterCMT = 0;
		}
	}

  if ((driveStatus.bit.AfterCMT == 0) && (driveStatus.bit.NewZC == 0) && (driveStatus.bit.Sensorless == 1)) {
	  ZCdetectionAdc[ActualCmtSector]();

	  bemfVoltageOld = bemfVoltage;	// save previous Back-EMF voltage (for ADC samples approximation)
	  driveStatus.bit.AdcSaved = 1;

	  if (driveStatus.bit.NewZC) {
		  GDUPHMUX_GPHMX = BemfPhase[NextCmtSector];	// will be applied at the next PMF Reload
		  PTUC_PTULDOK = 1;
		  DCBusCurrentZC = DCBusCurrent;
		  DCBusCurrentFiltered = (tFrac16)((long) ((long)DCBusCurrentFiltered + (long)DCBusCurrentFiltered + (long)DCBusCurrentFiltered + (long)DCBusCurrentZC) >> 2);
	  }

  }

  if (driveStatus.bit.DisableAdc == 1){
	  driveStatus.bit.DisableAdc = 0;
	  driveStatus.bit.NewZC = 0;
  }

  ADC0CONIF = 1;      // Clear flag
}

/*****************************************************************************
*
* Function: void ADC1error_ISR(void)
*
* Description: ADC1 Error Interrupt Service Routine
*
*****************************************************************************/
interrupt VectorNumber_Vadc1err void ADC1error_ISR(void)
{
char tmpAdcEIF;

	tmpAdcEIF = ADC1EIF;

  if (tmpAdcEIF & ADC1EIF_LDOK_EIF_MASK) {       // Load Ok Error
      AdcErrorLDOK++;
      ADC1EIF = ADC1EIF_LDOK_EIF_MASK;
  }
  if (tmpAdcEIF & ADC1EIF_RSTAR_EIF_MASK) {       // Restart Request Error
      AdcErrorRSTAR++;
      ADC1EIF = ADC1EIF_RSTAR_EIF_MASK;
  }
  if (tmpAdcEIF & ADC1EIF_TRIG_EIF_MASK) {       // Trigger Error => Soft Reset
      AdcErrorTRIG++;
      ADC1CTL_0_ADC_SR = 1;
  }
  if (tmpAdcEIF & ADC1EIF_EOL_EIF_MASK) {       // End Of List Error => Soft Reset
      AdcErrorEOL++;
      ADC1CTL_0_ADC_SR = 1;
  }
  if (tmpAdcEIF & ADC1EIF_CMD_EIF_MASK) {       // Command Value Error => Soft Reset
      AdcErrorCMD++;
      ADC1CTL_0_ADC_SR = 1;
  }
  if (tmpAdcEIF & ADC1EIF_IA_EIF_MASK) {       // Illegal Access Error => Soft Reset
      AdcErrorIA++;
      ADC1CTL_0_ADC_SR = 1;
  }
}

/*****************************************************************************
*
* Function: void ADC1done_ISR(void) - Used only for debugging
*
* Description: ADC1 Conversion Done Interrupt Service Routine
*
*****************************************************************************/
interrupt  VectorNumber_Vadc1conv_compl void ADC1done_ISR(void)
{
	  ADC1CONIF = 1;      // Clear flag
}

/*****************************************************************************
*
* Function: void AdcCalibration(void)
*
* Description: Calibration of the DC Bus current offset function
*
*****************************************************************************/
void AdcCalibration(void)
{
int i;
tFrac16 AdcOffset;

	NextCmtPeriod = TIMER_1MS;	// 1 ms
	DCBusCurrentOffset = 0;
	driveStatus.bit.AfterCMT = 0;
	for (i = 0; i < 1000; i++) { // 1 second delay
		while (driveStatus.bit.AfterCMT == 0)
		{
			
		}
		driveStatus.bit.AfterCMT = 0;
	}
	AdcOffset = DCBusCurrent;

	for (i = 0; i < 1000; i++) { // 1 second measurement
		while (driveStatus.bit.AfterCMT == 0)
		{
			
		}
		driveStatus.bit.AfterCMT = 0;
		AdcOffset = ((long) AdcOffset + AdcOffset + AdcOffset + DCBusCurrent) >> 2;
	}
	DCBusCurrentOffset = AdcOffset;
}


/*****************************************************************************
*
* Function: void main(void)
*
* Description: main function
*
*****************************************************************************/
void main(void)
{
	initCPMU();
	initGPIO();
	initPMF();
	
	initPTU();
	initADC();
	initSCI();
	initGDU();
	initInt();
	Init_LIN();
	Init_PIM();
	l_sys_init();
	l_ifc_init(LI0);
	initTIM();


	FMSTR_Init();

	EnableInterrupts;

	AppInit();
	AdcCalibration();

	for(;;)
	{
		FMSTR_Poll();
		speed = l_u16_rd_LI0_req_speed(); /* Reads the speed send by the master */
		AppStateMachine[appState]();
		CheckFaults();
		l_u16_wr_LI0_status_current_speed(actualSpeed); /*Sends the current speed to the master */
	}
}


/*****************************************************************************
*
* Function: void AppInit(void)
*
* Description: BLDC application init. function
*
*****************************************************************************/
void AppInit(void){
	timeZCToff = TIME_TOFF;
	duty_cycle = 0;
	UpdateDutycycle();

	advanceAngle = ADVANCE_ANGLE;

	NextCmtPeriod = TIMER_1MS;	// 1 ms

	driveStatus.bit.Alignment = 0;
	driveStatus.bit.EnableCMT = 0;
	driveStatus.bit.CloseLoop = 0;
	driveStatus.bit.Sensorless = 0;
	driveStatus.bit.DisableAdc = 0;
	driveStatus.bit.NewZC = 0;

	interfaceStatus.byte = 0;
	driveDemand.byte = 0;
	driveDemandRemote.byte = 0;

	actualSpeed = 0;

	appState = APP_STOP;
}

/*****************************************************************************
*
* Function: void AppAlignment(void)
*
* Description:  function
*
*****************************************************************************/
void AppAlignment(void){
	if (alignmentTimer == 0) {
		AppAlignmentToStart();
	}
}

/*****************************************************************************
*
* Function: void AppStart(void)
*
* Description:  function
*
*****************************************************************************/
void AppStart(void){
  if (driveStatus.bit.AfterCMT == 1) {
      timeZC = TIM0TCNT - (NextCmtPeriod >> 1);

      startCMTcounter--;
      if (startCMTcounter > 0) {
      driveStatus.bit.AfterCMT = 0;

      NextCmtPeriod = MLIB_Mul(NextCmtPeriod,START_CMT_ACCELERATION,F16);
    }
  }
  if (startCMTcounter == 0) {
    AppStartToRun();
  }
}

/*****************************************************************************
*
* Function: void AppRun(void)
*
* Description:  function
*
*****************************************************************************/
void AppRun(void){
	if (StallError) {
		AppError();
		appState = APP_STOP;
		return;
	}
	
	requiredSpeed = speed;
	if (requiredSpeed > MAX_SPEED)
	{
		requiredSpeed = MAX_SPEED;
	}
	if (requiredSpeed < MIN_SPEED)
	{
		requiredSpeed = MIN_SPEED;
	}
	/*
	// Process buttons and Start/Stop switch
	// Up button
	if (interfaceStatus.bit.Up == 1)
	{
		requiredSpeed += SPEED_STEP;
		if (requiredSpeed > MAX_SPEED)
		{
			requiredSpeed = MAX_SPEED;
		}
	  interfaceStatus.bit.Up = 0;
	}
	// Down button
	if (interfaceStatus.bit.Down == 1)
	{
		requiredSpeed -= SPEED_STEP;
		if (requiredSpeed < MIN_SPEED)
		{
			requiredSpeed = MIN_SPEED;
		}
		interfaceStatus.bit.Down = 0;
	}*/
	
	// Start/Stop switch
	if(interfaceStatus.bit.Switch == 0)
	{
		if (interfaceStatus.bit.SwitchChange == 1)
		{
			driveDemandRemote.bit.StopRun = 0;
		}
	}
	if (driveDemandRemote.bit.StopRun == 0)
	{
		interfaceStatus.bit.SwitchChange = 0;
		driveDemand.bit.StopRun = 0;
	}
	if (driveDemand.bit.StopRun == 0)
	{
		AppRunToStop();
	}
}

/*****************************************************************************
*
* Function: void AppRunToStop(void)
*
* Description: Stop function, ready to run
*
*****************************************************************************/
void AppRunToStop(void)
{
	driveStatus.bit.Alignment = 0;
	driveStatus.bit.EnableCMT = 0;
	driveStatus.bit.CloseLoop = 0;
	driveStatus.bit.Sensorless = 0;
	driveStatus.bit.CloseLoop = 0;

	duty_cycle = 0;
	UpdateDutycycle();

	appState = APP_STOP;
}

/*****************************************************************************
*
* Function: void AppStop(void)
*
* Description: Stop function, ready to run
*
*****************************************************************************/
void AppStop(void){
	driveStatus.bit.CurrentLimiting = 0;

	// Process buttons and Start/Stop switch
	if(interfaceStatus.bit.Switch == 1)
	{
		if (interfaceStatus.bit.SwitchChange == 1)
		{
			driveDemandRemote.bit.StopRun = 1;
			interfaceStatus.bit.SwitchChange = 0;
		}
	}
	else
	{
		if (interfaceStatus.bit.SwitchChange == 1)
		{
			driveDemandRemote.bit.StopRun = 0;
			interfaceStatus.bit.SwitchChange = 0;
		}
	}

	driveDemand.bit.StopRun = driveDemandRemote.bit.StopRun;
	if (driveDemand.bit.StopRun == 1)
	{
		if (GDUF != 0) GDUF = GDUF;	// clear fault flags
		if (GDUF == 0) AppStopToAlignment();
	}
}


/*****************************************************************************
*
* Function: void AppError(void)
*
* Description: Error function
*
*****************************************************************************/
void AppError(void){
	driveStatus.bit.Alignment = 0;
	driveStatus.bit.EnableCMT = 0;
	driveStatus.bit.CloseLoop = 0;
	driveStatus.bit.Sensorless = 0;
	driveStatus.bit.CloseLoop = 0;

	duty_cycle = 0;
	UpdateDutycycle();
}

/*****************************************************************************
*
* Function: void AppStopToAlignment(void)
*
* Description:  function
*
*****************************************************************************/
void AppStopToAlignment(void){
	driveStatus.bit.Alignment = 1;
	driveStatus.bit.EnableCMT = 0;
	driveStatus.bit.CloseLoop = 0;
	driveStatus.bit.Sensorless = 0;
	driveStatus.bit.DisableAdc = 0;
	driveStatus.bit.NewZC = 0;

	NextCmtPeriod = TIMER_1MS;	// 1 ms

	TIM0TC0 = TIM0TCNT + NextCmtPeriod;
	timeZCToff = TIME_TOFF;
	alignmentTimer = ALIGNMENT_TIME;

    AlignCurrentPIPrms.f16UpperLimit = FRAC16(ALIGN_PI_UPPER_LIMIT);
    AlignCurrentPIPrms.f16LowerLimit = FRAC16(ALIGN_PI_LOWER_LIMIT);
    AlignCurrentPIPrms.f16CC1sc = FRAC16(ALIGN_PI_PARAM_CC1);
    AlignCurrentPIPrms.f16CC2sc = 0;
    AlignCurrentPIPrms.u16NShift = 0;
    AlignCurrentPIPrms.f32Acc = 0;
    AlignCurrentPIPrms.f16InErrK1 = 0;

	duty_cycle = 0;
	UpdateDutycycle();

	PMFCFG2 = 0x40 + 0x10;        // set commutation vector (phase A = PWM, phase B = PWM, phase C is grounded)
	PMFOUTC_OUTCTL = 0x30;
	TIM0CFORC_FOC0 = 1; // force commutation event (apply PMF MASK/Control configuration)

	appState = APP_ALIGNMENT;
}

/*****************************************************************************
*
* Function: void AppAlignmentToStart(void)
*
* Description:  function
*
*****************************************************************************/
void AppAlignmentToStart(void){
	NextCmtSector = 3; // starting sector

	driveStatus.bit.Alignment = 0;
	driveStatus.bit.EnableCMT = 1;
	driveStatus.bit.AfterCMT  = 0;

	NextCmtPeriod = START_FIRST_PERIOD;
	startCMTcounter = START_CMT_CNT;

	TIM0TC0 = TIM0TCNT + NextCmtPeriod;
	PMFCFG2 = 0x40 + MaskVal[NextCmtSector];
	PMFOUTC_OUTCTL = OutCtl[NextCmtSector];
	NextCmtSector++;
	TIM0CFORC_FOC0 = 1; // force commutation event (apply first commutation sector)

	PMFCFG2 = 0x40 + MaskVal[NextCmtSector];
	PMFOUTC_OUTCTL = OutCtl[NextCmtSector];	// save another commutation sector values to the buffered registers

	NextCmtPeriod = MLIB_Mul(NextCmtPeriod,START_CMT_ACCELERATION,F16);

	appState = APP_START;
}

/*****************************************************************************
*
* Function: void AppStartToRun(void)
*
* Description:  function
*
*****************************************************************************/
void AppStartToRun (void){
    speedPIPrms.f16UpperLimit = FRAC16(RUN_PI_UPPER_LIMIT);
    speedPIPrms.f16LowerLimit = FRAC16(RUN_PI_LOWER_LIMIT);
    speedPIPrms.f16CC1sc = FRAC16(RUN_PI_SPEED_PARAM_CC1);
    speedPIPrms.f16CC2sc = 0;
    speedPIPrms.u16NShift = 0;
    speedPIPrms.f32Acc = (((tFrac32) duty_cycle <<15)/(tFrac32) PWM_MODULO) << 16;
    speedPIPrms.f16InErrK1 = 0;

    currentPIPrms.f16UpperLimit = FRAC16(RUN_PI_UPPER_LIMIT);
    currentPIPrms.f16LowerLimit = FRAC16(RUN_PI_LOWER_LIMIT);
    currentPIPrms.f16CC1sc = FRAC16(RUN_PI_CURRENT_PARAM_CC1);
    currentPIPrms.f16CC2sc = 0;
    currentPIPrms.u16NShift = 0;
    currentPIPrms.f32Acc = (((tFrac32) duty_cycle <<15)/(tFrac32) PWM_MODULO) << 16;
    currentPIPrms.f16InErrK1 = 0;

    appState = APP_RUN;
	stallCheckCounter = 0;
	StallError = 0;

	periodZC_F_PhA = NextCmtPeriod;
	periodZC_F_PhB = NextCmtPeriod;
	periodZC_F_PhC = NextCmtPeriod;
	periodZC_R_PhA = NextCmtPeriod;
	periodZC_R_PhB = NextCmtPeriod;
	periodZC_R_PhC = NextCmtPeriod;
	actualPeriodZC = NextCmtPeriod;

	requiredSpeed = REQUIRED_RUN_SPEED;
	DCBusCurrentLimit = DC_BUS_CURRENT_LIMIT;

	interfaceStatus.bit.Down = 0;
	interfaceStatus.bit.Up = 0;

	GDUPHMUX_GPHMX = BemfPhase[ActualCmtSector];	// will be applied at the next PMF Reload
	PTUC_PTULDOK = 1;

	driveStatus.bit.Sensorless = 1;
}

/*****************************************************************************
*
* Function: void ZCdetectPhXYYYYAdc(void)
*
* Description:  Zero-cross detection functions (based on ADC)
*
*****************************************************************************/

void ZCdetectPhCfallingAdc(void)
{
tFrac16 delta;

	if (bemfVoltage <= 0)
	{
	    // Falling approximation
		delta = bemfVoltage - bemfVoltageOld;	// bemfVoltage & delta > 0 for raising BEMF, bemfVoltage & delta < 0 for falling BEMF
	    if ((driveStatus.bit.AdcSaved == 1) && (delta < bemfVoltage)) { // if delta < bemfVoltage then also delta < 0
    		timeBEMF -= MLIB_Mul(MLIB_Div(bemfVoltage, delta, F16), timeBEMF - timeOldBEMF, F16);
	    }
	    else { // middle of previous ADC sensing events
	    	timeBEMF -= ((timeBEMF - timeOldBEMF) >> 1);
	    }

		lastTimeZC = timeZC;
		timeZC = timeBEMF;

		periodZC_F_PhC = timeZC - lastTimeZC;
		actualPeriodZC = (actualPeriodZC + periodZC_F_PhC) >> 1;

		NextCmtPeriod = MLIB_Mul(actualPeriodZC, advanceAngle, F16);
		TIM0TC0 = timeZC + NextCmtPeriod;

		driveStatus.bit.NewZC = 1;
	}
}

void ZCdetectPhBraisingAdc(void)
{
tFrac16 delta;

	if (bemfVoltage >= 0)
	{
		// Raising approximation
		delta = bemfVoltage - bemfVoltageOld;	// bemfVoltage & delta > 0 for raising BEMF, bemfVoltage & delta < 0 for falling BEMF
		if ((driveStatus.bit.AdcSaved == 1) && (delta > bemfVoltage)) { // if delta > bemfVoltage then also delta > 0
    		timeBEMF -= MLIB_Mul(MLIB_Div(bemfVoltage, delta, F16), timeBEMF - timeOldBEMF, F16);
	    }
	    else { // middle of previous ADC sensing events
	    	timeBEMF -= ((timeBEMF - timeOldBEMF) >> 1);
	    }

		lastTimeZC = timeZC;
		timeZC = timeBEMF;

		periodZC_R_PhB = timeZC - lastTimeZC;
		actualPeriodZC = (actualPeriodZC + periodZC_R_PhB) >> 1;
		NextCmtPeriod = MLIB_Mul(actualPeriodZC, advanceAngle, F16);

		TIM0TC0 = timeZC + NextCmtPeriod;
		driveStatus.bit.NewZC = 1;
	}
}

void ZCdetectPhAfallingAdc(void)
{
tFrac16 delta;

	if (bemfVoltage <= 0)
	{
	    // Falling approximation
		delta = bemfVoltage - bemfVoltageOld;	// bemfVoltage & delta > 0 for raising BEMF, bemfVoltage & delta < 0 for falling BEMF
		if ((driveStatus.bit.AdcSaved == 1) && (delta < bemfVoltage)) { // if delta < bemfVoltage then also delta < 0
    		timeBEMF -= MLIB_Mul(MLIB_Div(bemfVoltage, delta, F16), timeBEMF - timeOldBEMF, F16);
	    }
	    else { // middle of previous ADC sensing events
	    	timeBEMF -= ((timeBEMF - timeOldBEMF) >> 1);
	    }

		lastTimeZC = timeZC;
		timeZC = timeBEMF;

		periodZC_F_PhA = timeZC - lastTimeZC;
		actualPeriodZC = (actualPeriodZC + periodZC_F_PhA) >> 1;

		NextCmtPeriod = MLIB_Mul(actualPeriodZC, advanceAngle, F16);
		TIM0TC0 = timeZC + NextCmtPeriod;

		driveStatus.bit.NewZC = 1;
	}
}

void ZCdetectPhCraisingAdc(void)
{
tFrac16 delta;

	if (bemfVoltage >= 0)
	{
		// Raising approximation
		delta = bemfVoltage - bemfVoltageOld;	// bemfVoltage & delta > 0 for raising BEMF, bemfVoltage & delta < 0 for falling BEMF
	    if ((driveStatus.bit.AdcSaved == 1) && (delta > bemfVoltage)) { // if delta > bemfVoltage then also delta > 0
    		timeBEMF -= MLIB_Mul(MLIB_Div(bemfVoltage, delta, F16), timeBEMF - timeOldBEMF, F16);
	    }
	    else { // middle of previous ADC sensing events
	    	timeBEMF -= ((timeBEMF - timeOldBEMF) >> 1);
	    }

		lastTimeZC = timeZC;
		timeZC = timeBEMF;

		periodZC_R_PhC = timeZC - lastTimeZC;
		actualPeriodZC = (actualPeriodZC + periodZC_R_PhC) >> 1;

		NextCmtPeriod = MLIB_Mul(actualPeriodZC, advanceAngle, F16);
		TIM0TC0 = timeZC + NextCmtPeriod;

		driveStatus.bit.NewZC = 1;
	}
}

void ZCdetectPhBfallingAdc(void)
{
tFrac16 delta;

	if (bemfVoltage <= 0)
	{
	    // Falling approximation
		delta = bemfVoltage - bemfVoltageOld;	// bemfVoltage & delta > 0 for raising BEMF, bemfVoltage & delta < 0 for falling BEMF
	    if ((driveStatus.bit.AdcSaved == 1) && (delta < bemfVoltage)) { // if delta < bemfVoltage then also delta < 0
    		timeBEMF -= MLIB_Mul(MLIB_Div(bemfVoltage, delta, F16), timeBEMF - timeOldBEMF, F16);
	    }
	    else { // middle of previous ADC sensing events
	    	timeBEMF -= ((timeBEMF - timeOldBEMF) >> 1);
	    }

		lastTimeZC = timeZC;
		timeZC = timeBEMF;

		periodZC_F_PhB = timeZC - lastTimeZC;
		actualPeriodZC = (actualPeriodZC + periodZC_F_PhB) >> 1;

		NextCmtPeriod = MLIB_Mul(actualPeriodZC, advanceAngle, F16);
		TIM0TC0 = timeZC + NextCmtPeriod;

		driveStatus.bit.NewZC = 1;
	}
}

void ZCdetectPhAraisingAdc(void)
{
tFrac16 delta;

	if (bemfVoltage >= 0)
	{
		// Raising approximation
		delta = bemfVoltage - bemfVoltageOld;	// bemfVoltage & delta > 0 for raising BEMF, bemfVoltage & delta < 0 for falling BEMF
	    if ((driveStatus.bit.AdcSaved == 1) && (delta > bemfVoltage)) { // if delta > bemfVoltage then also delta > 0
    		timeBEMF -= MLIB_Mul(MLIB_Div(bemfVoltage, delta, F16), timeBEMF - timeOldBEMF, F16);
	    }
	    else { // middle of previous ADC sensing events
	    	timeBEMF -= ((timeBEMF - timeOldBEMF) >> 1);
	    }

		lastTimeZC = timeZC;
		timeZC = timeBEMF;

		periodZC_R_PhA = timeZC - lastTimeZC;
		actualPeriodZC = (actualPeriodZC + periodZC_R_PhA) >> 1;

		NextCmtPeriod = MLIB_Mul(actualPeriodZC, advanceAngle, F16);
		TIM0TC0 = timeZC + NextCmtPeriod;

		driveStatus.bit.NewZC = 1;
	}
}

/*****************************************************************************
*
* Function: void StallCheck(void)
*
* Description:  Stall check function
*
*****************************************************************************/
void StallCheck(void)
{
tU16 max = 0, min = 65535;

  if (periodZC_F_PhA>max)
  {
    max = periodZC_F_PhA;
  }
  if (periodZC_F_PhA<min)
  {
    min = periodZC_F_PhA;
  }
  if (periodZC_F_PhB>max)
  {
    max = periodZC_F_PhB;
  }
  if (periodZC_F_PhB<min)
  {
    min = periodZC_F_PhB;
  }
  if (periodZC_F_PhC>max)
  {
    max = periodZC_F_PhC;
  }
  if (periodZC_F_PhC<min)
  {
    min = periodZC_F_PhC;
  }
  if (periodZC_R_PhA>max)
  {
    max = periodZC_R_PhA;
  }
  if (periodZC_R_PhA<min)
  {
    min = periodZC_R_PhA;
  }
  if (periodZC_R_PhB>max)
  {
    max = periodZC_R_PhB;
  }
  if (periodZC_R_PhB<min)
  {
    min = periodZC_R_PhB;
  }
  if (periodZC_R_PhC>max)
  {
    max = periodZC_R_PhC;
  }
  if (periodZC_R_PhC<min)
  {
    min = periodZC_R_PhC;
  }

  /* Save min and max commutation periods for tuning purposes */
  debugPeriodMin = min;
  debugPeriodMax = max;

  periodZcAvrg = period6ZC / 6;

  if ((max > (periodZcAvrg << 1)) || (min < (periodZcAvrg >> 1))) {
	  if (stallCheckCounter < STALLCHECK_MAX_ERRORS) {
		  stallCheckCounter++;
      } // max > (2 * periodZcAvrg) || min < (0.5 * periodZcAvrg) = period6ZC/8 * 0.5*8/6
  } else {
	  if (min < STALLCHECK_MIN_CMT_PERIOD) {
		  if (stallCheckCounter < STALLCHECK_MAX_ERRORS) {
			  stallCheckCounter++;
          }
      } else {
          if (stallCheckCounter > 0) {
        	  stallCheckCounter--;
          }
      }
  }


  if (stallCheckCounter >= STALLCHECK_MAX_ERRORS) {
	  StallError = 1;
  } else {
	  StallError = 0;
  }
}


/*****************************************************************************
*
* Function: void CheckManualInterface(void)
*
* Description:  The function checks status of user interface Up/Down Buttons
*               and START/STOP switch.
*
*****************************************************************************/
void CheckManualInterface(void)
{
static unsigned char upCounter, downCounter, switchCounter;

	// Down button
	if ((DOWN_BUTTON == 0) && (interfaceStatus.bit.Down == 0))
	{
		if (downCounter < DEBOUNCE_VALUE) downCounter++;
	}
	else
	{
		if (downCounter < DEBOUNCE_VALUE) downCounter = 0;
	}

	if ((downCounter == DEBOUNCE_VALUE) && (DOWN_BUTTON == 1))
	{
		interfaceStatus.bit.Down = 1;
		downCounter = 0;
	}

	// Up button
	if ((UP_BUTTON == 0) && (interfaceStatus.bit.Up == 0))
	{
		if (upCounter < DEBOUNCE_VALUE) upCounter++;
	}
	else
	{
		if (upCounter < DEBOUNCE_VALUE) upCounter = 0;
	}

	if ((upCounter == DEBOUNCE_VALUE) && (UP_BUTTON == 1))
	{
		interfaceStatus.bit.Up = 1;
		upCounter = 0;
	}

	// Start/Stop switch
	if (TOGGLE_SWITCH == 0)
	{
	    if (switchCounter < DEBOUNCE_VALUE) switchCounter++;
	}
	else
	{
	    if (switchCounter > 0) switchCounter--;
	}

	if (switchCounter == DEBOUNCE_VALUE)
	{
	    if (interfaceStatus.bit.Switch == 0)
	    {
	    	interfaceStatus.bit.Switch = 1;
	    	interfaceStatus.bit.SwitchChange = 1;
	    }
	}

	if (switchCounter == 0)
	{
	    if (interfaceStatus.bit.Switch == 1)
	    {
	    	interfaceStatus.bit.Switch = 0;
	    	interfaceStatus.bit.SwitchChange = 1;
	    }
	}

}

/*****************************************************************************
*
* Function: void UpdateCurrentLimitingLed(void)
*
* Description:  The function displays current limiting condition on LED
*
*****************************************************************************/
void UpdateCurrentLimitingLed(void)
{
static unsigned char Led1Counter;

	if (driveStatus.bit.CurrentLimiting == 0)
	{
		if (Led1Counter > 0)
		{
			Led1Counter--;
			if (Led1Counter == 0) CURRENT_LIMIT_LED = 0;
		}
	}
	else
	{
		Led1Counter = LED_TIMER_MS;
		CURRENT_LIMIT_LED = 1;
	}
}

/*****************************************************************************
*
* Function: void CheckFaults(void)
*
* Description:  The function checks status of faults bits
*               In case any fault bit is set application stops the operation
*
*****************************************************************************/
void CheckFaults(void)
{
	GduFlags = GDUF;		// to see in FreeMASTER
	if (GDUF != 0)
	{
		if (driveStatus.bit.Fault == 0)
		{
			driveStatus.bit.Fault = 1;
			AppError();
			appState = APP_STOP;
			driveDemandRemote.bit.StopRun = 0;
			interfaceStatus.bit.SwitchChange = 0;
			FAULT_LED = 1;
		}
	}
	else
	{
		if (driveStatus.bit.Fault == 1)
		{
			driveStatus.bit.Fault = 0;
			FAULT_LED = 0;
		}
	}
}
