/*-----------------------------------------------------------------------------
  Copyright (c) 2015 by Freescale Semiconductor, All Rights reserved.

  This is unpublished proprietary source code of Freescale.
  The copyright notice above does not imply any actual or
  intended publication of such source code.

  FREESCALE CONFIDENTIAL PROPRIETARY
  ---------------------------------------------------------------------------*/
/*-----------------------------------------------------------------------------
  FILE NAME      : hsm.c
  DEPARTMENT     : AISG
  AUTHOR         : Pradip Singh
  AUTHOR'S EMAIL : pradip.singh@freescale.com
  -----------------------------------------------------------------------------
  REVISION HISTORY
  AUTHOR       			 Date D/M/Y                DESCRIPTION              
  ---------------------- 	----------              --------------------- 
  Pradip Singh (b09147) 	 27/7/2015               Initial Version
  --------------------- 	 ----------             ---------------------
  KEYWORDS :HSM, Security
  -----------------------------------------------------------------------------
  PURPOSE :  Implements C functions for preparing data structure for different
             HSM commands. 
  ---------------------------------------------------------------------------*/

/*=============================================================================
  INCLUDE FILES
  ===========================================================================*/
#include "typedefs.h"
#include "hsm.h"
/*=============================================================================
  LOCAL CONSTANTS
  ===========================================================================*/

/*=============================================================================
  LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
  ===========================================================================*/
/* struct for command structure memory map for HSM */
struct HSM_CMD_struct HSM_CMD; 
/*=============================================================================
  LOCAL VARIABLES
  ===========================================================================*/

/*=============================================================================
  LOCAL FUNCTION PROTOTYPES
  ===========================================================================*/

/*=============================================================================
  GLOBAL VARIABLES
  ===========================================================================*/

/*=============================================================================
  LOCAL FUNCTIONS
  ===========================================================================*/

/*=============================================================================
  FUNCTION: HSM_encrypt_ECB

  DESCRIPTION:
  This function prepares the data structure for ECB encrypt command

  ARGUMENTS PASSED:
  Number of 128-bit data blocks, First Plaintext block address,
  First Cyphertext block address

  RETURN VALUE:
  None

  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None

  ===========================================================================*/
void HSM_encrypt_ECB(uint8_t nBLOCKS,
	uint32_t plain_address,
	uint32_t cypher_address)
{
    /* encrypt ECB */
    HSM_CMD.CMD = 0x1;
    /* key ID = RAM_KEY (Plain key)*/
    HSM_CMD.PARAM_1 = 0xE;            
    /* number of 128-bit data blocks */							
    HSM_CMD.PARAM_2 = nBLOCKS;
    /* first plain text block address */
    HSM_CMD.PARAM_3 = plain_address;
    /* first cypher text block address */    
    HSM_CMD.PARAM_4 = cypher_address; 							
    HSM_CMD.PARAM_5 = 0x00000000;
}
/*=============================================================================
  FUNCTION: HSM_decrypt_ECB

  DESCRIPTION:
  This function prepares the data structure for ECB decrypt command

  ARGUMENTS PASSED:
  Number of 128-bit data blocks, First Plaintext block address,
  First Cyphertext block address

  RETURN VALUE:
  None

  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None
  ===========================================================================*/
void HSM_decrypt_ECB(uint8_t nBLOCKS,
	uint32_t plain_address,
	uint32_t cypher_address)
{
    /* decrypt ECB */
    HSM_CMD.CMD = 0x3;
    /* key ID = RAM_KEY (Plain key)*/
    HSM_CMD.PARAM_1 = 0xE;            
    /* number of 128-bit data blocks */							
    HSM_CMD.PARAM_2 = nBLOCKS;
    /* first cypher text block address */
    HSM_CMD.PARAM_3 = cypher_address;
    /* first plain text block address */
    HSM_CMD.PARAM_4 = plain_address;  							
    HSM_CMD.PARAM_5 = 0x00000000;
}
/*=============================================================================
  FUNCTION: HSM_encrypt_CBC

  DESCRIPTION:
  This function prepares the data structure for CBC encrypt command

  ARGUMENTS PASSED:
  Number of 128-bit data blocks, First Plaintext block address,
  First Cyphertext block address, IV address

  RETURN VALUE:
  None

  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None
  ===========================================================================*/
void HSM_encrypt_CBC(uint8_t nBLOCKS,
	uint32_t plain_address,
	uint32_t cypher_address,
	uint32_t IV_address)
{
    /* encrypt CBC */
    HSM_CMD.CMD = 0x2;
    /* key ID = RAM_KEY (Plain key)*/
    HSM_CMD.PARAM_1 = 0xE;
    /* initialization vector address */
    HSM_CMD.PARAM_2 = IV_address;
    /* number of 128-bit data blocks */
    HSM_CMD.PARAM_3 = nBLOCKS;
    /* first plain text block address */
    HSM_CMD.PARAM_4 = plain_address;
    /* first cypher text block address */
    HSM_CMD.PARAM_5 = cypher_address; 							
}
/*=============================================================================
  FUNCTION: HSM_decrypt_CBC

  DESCRIPTION:
  This function prepares the data structure for CBC decrypt command

  ARGUMENTS PASSED:
  Number of 128-bit data blocks, First Plaintext block address,
  First Cyphertext block address, IV address

  RETURN VALUE:
  None

  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None
  ===========================================================================*/
void HSM_decrypt_CBC(uint8_t nBLOCKS,
	uint32_t plain_address,
	uint32_t cypher_address,
	uint32_t IV_address)
{
    /* decrypt CBC */
    HSM_CMD.CMD = 0x4;
    /* key ID = RAM_KEY (Plain key)*/
    HSM_CMD.PARAM_1 = 0xE;
    /* initialization vector address */
    HSM_CMD.PARAM_2 = IV_address;
    /* number of 128-bit data blocks */
    HSM_CMD.PARAM_3 = nBLOCKS;
    /* first cypher text block address */
    HSM_CMD.PARAM_4 = cypher_address;
    /* first plain text block address */
    HSM_CMD.PARAM_5 = plain_address;  							
}
/*=============================================================================
  FUNCTION: HSM_generate_mac

  DESCRIPTION:
  This function prepares the data structure for generate mac command

  ARGUMENTS PASSED:
  Message length(in bits) address, Message start address, MAC address

  RETURN VALUE:
  None

  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None
  ===========================================================================*/
void HSM_generate_mac(uint32_t msglen_address,
	uint32_t msgaddr,
	uint32_t macaddr)
{
    /* verify mac command */
    HSM_CMD.CMD = 0x05;
    /* key ID = RAM_KEY (Plain key)*/
    HSM_CMD.PARAM_1 = 0xE;
    /* Message length(in bits) address */
    HSM_CMD.PARAM_2 = msglen_address;
    /* message start address */
    HSM_CMD.PARAM_3 = msgaddr;
    /* MAC address */
    HSM_CMD.PARAM_4 = macaddr;
							
}

/*=============================================================================
  FUNCTION: HSM_verify_mac

  DESCRIPTION:
  This function prepares the data structure for verify mac command

  ARGUMENTS PASSED:
  Message length(in bits) address, Message start address, MAC address,
  MAC length(in bits) address

  RETURN VALUE:
  None

  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None
  ===========================================================================*/
void HSM_verify_mac(uint32_t msglen_address,
	uint32_t msgaddr,
	uint32_t macaddr,
	uint32_t maclen_address)
{
    /* verify mac command */
    HSM_CMD.CMD = 0x06;
    /* key ID = RAM_KEY (Plain key)*/
    HSM_CMD.PARAM_1 = 0xE;
    /* Message length(in bits) address */
    HSM_CMD.PARAM_2 = msglen_address;
    /* message start address */
    HSM_CMD.PARAM_3 = msgaddr;
    /* MAC address */
    HSM_CMD.PARAM_4 = macaddr;
    /* MAC length(in bits)/ Verification status address */
    HSM_CMD.PARAM_5 = maclen_address;  							
}
/*=============================================================================
  FUNCTION: HSM_load_plain_key

  DESCRIPTION:
  This function prepares the data structure for load plain key command

  ARGUMENTS PASSED:
  Key address

  RETURN VALUE:
  None

  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None
  ===========================================================================*/
void HSM_load_plain_key(uint32_t key_address)
{
    /* load plain key command */
    HSM_CMD.CMD = 0x8;
    /* key address */
    HSM_CMD.PARAM_1 = key_address;   							
    HSM_CMD.PARAM_2 = 0x00000000; 
    HSM_CMD.PARAM_3 = 0x00000000; 
    HSM_CMD.PARAM_4 = 0x00000000; 
    HSM_CMD.PARAM_5 = 0x00000000;
}
/*=============================================================================
  FUNCTION: hsm_done

  DESCRIPTION:
  This function clears the Command complete bit

  ARGUMENTS PASSED:
  None

  RETURN VALUE:
  None

  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None
  ===========================================================================*/
void hsm_done(void)
{
    /*Wait till CMD_COMPLETE flag is set by the SHE or HSM is busy*/
    while(HSM.HSM2HTF.B.CMD_COMPLETE == 0 | HSM.HSM2HTS.B.BUSY == 1);    
    /* clear cmd_complete */
    HSM.HSM2HTF.R = 0x00000002; 							
}
/*=============================================================================
  FUNCTION: hsm_kdf

  DESCRIPTION:
  This function calculates KDF as per the SHE firmware specifications

  ARGUMENTS PASSED:
  Plaintest address, Number of 128 bit blocks address, Destination address

  RETURN VALUE:
  None

  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None
  ===========================================================================*/
void hsm_kdf(uint32_t *plaintext,
	uint32_t *dst,
	uint32_t numblocks)
{
    uint32_t *temp;
    uint32_t iv[4],cipher[4];
    uint8_t ptr_plaintext=0;

/*Initialise output to zero*/
    temp=dst;
    temp[0] = 0UL;
    temp[1] = 0UL;
    temp[2] = 0UL;
    temp[3] = 0UL;

/*Initialise Initial Vector to zero*/
    iv[0] = 0UL;
    iv[1] = 0UL;
    iv[2] = 0UL;
    iv[3] = 0UL;

/*Initialise Cipher to zero*/
    cipher[0]=0;
    cipher[1]=0;
    cipher[2]=0;
    cipher[3]=0;

    while(numblocks)
    {
	/*Load IV as RAM key*/
	/* IV is loaded to RAM KEY */
	HSM_load_plain_key((uint32_t)&iv);						
	HSM.HT2HSMS.R         = (uint32_t) &HSM_CMD;
	HSM.HT2HSMF.B.CMD_INT = 1;  							
	hsm_done();  

	/*Calculate ECB ENC over 128 bit chunk of Plaintext*/
	/* prepares data structure for encrypt ECB command */
	HSM_encrypt_ECB(1UL,
		(uint32_t)&plaintext[4*(ptr_plaintext)],
		(uint32_t)&cipher); 
	HSM.HT2HSMS.R         = (uint32_t) &HSM_CMD;
	HSM.HT2HSMF.B.CMD_INT = 1;  							
	hsm_done();

	numblocks--;

	iv[0] = iv[0] ^ cipher[0] ^ plaintext[0+4*(ptr_plaintext)];
        iv[1] = iv[1] ^ cipher[1] ^ plaintext[1+4*(ptr_plaintext)];
        iv[2] = iv[2] ^ cipher[2] ^ plaintext[2+4*(ptr_plaintext)];
        iv[3] = iv[3] ^ cipher[3] ^ plaintext[3+4*(ptr_plaintext)];

	ptr_plaintext++;
    }

/*Copy final OUTi to dst*/
    temp[0] = iv[0];
    temp[1] = iv[1];
    temp[2] = iv[2];
    temp[3] = iv[3];
}


