/*************************************************************************//**
 * @file SIPI_HSSL_Header_v4.h                                                    
 * @copyright Freescale 2014 All Rights Reserved                             
 * @version 1.0                                                              
 * @brief This is the SW Framework to emulate and validate the
          SIPI-HSSL Protocol
 * @date 30-Jan-12
 * @author                                                                  */
/*===========================================================================*
 * UPDATE HISTORY                                                            *
 * REV      AUTHOR      DATE       	DESCRIPTION OF CHANGE                    *
 * ---   -----------  ---------    	---------------------                    *
 * 1.0                30-Jan-12     Initial version                          *
 *                                                                           *
 *===========================================================================*
 * COPYRIGHT:                                                                *
 *  Freescale Semiconductor, INC. All Rights Reserved. You are hereby        *
 *  granted a copyright license to use, modify, and distribute the           *
 *  SOFTWARE so long as this entire notice is retained without alteration    *
 *  in any modified and/or redistributed versions, and that such modified    *
 *  versions are clearly identified as such. No licenses are granted by      *
 *  implication, estoppel or otherwise under any patentsor trademarks        *
 *  of Freescale Semiconductor, Inc. This software is provided on an         *
 *  "AS IS" basis and without warranty.                                      *
 *                                                                           *
 *  To the maximum extent permitted by applicable law, Freescale             *
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,       *
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A         *
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD     *
 *  TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY        *
 *  ACCOMPANYING WRITTEN MATERIALS.                                          *
 *                                                                           *
 *  To the maximum extent permitted by applicable law, IN NO EVENT           *
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER       *
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,     *
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER            *
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.     *
 *                                                                           *
 *  Freescale Semiconductor assumes no responsibility for the                *
 *  maintenance and support of this software                                 *
 *                                                                           *
 ****************************************************************************/

#include "SIPI_API.h"
#include "..\Zipwire.h"
#include "LFAST.h"


/***************************** Function Prototypes ***************************/

/** \fn SIPI_read(DATA_TEMPLATE_t data_address, CHANNEL_t channel, uint8_t injected_error)

\brief Performs a single read transfer. Returns '0' if successful, error code otherwise. Stores read value in DATA_TEMPLATE_t passed.
\return 
	- 0 = Successful
	- 1 = Invalid Width
	- 2 = Channel Busy
	- 3 = Invalid Channel
	- 4 = Timeout Error
\param[in] data_address DATA_TEMPLATE_t structure which includes read Address and data size.
\param[in] channel SIPI Channel to use. Should be passed as 1 element of CHANNEL array. 
\param[in] injected_error injected error (if required) - Not currently implemented
*/
uint8_t SIPI_read(DATA_TEMPLATE_t data_address, CHANNEL_t channel, uint8_t injected_error);



/** \fn SIPI_read_channel_data(CHANNEL_t channel)

\brief Function which is called to read Channel data when a successful read reply / ID reply is received.
> The function will always return a 32 bit value, if 8 or 16 bit data is read it will be replicated as described in the RM.
> A relevant casting / mask operation may be required for 8 and 16 bit read replies. Need to check for command completion and channel errors before calling this function.
\param[in] channel SIPI Channel the received data used.  
\return 
	- 0 = Invalid Channel
	- 32-bit value contained in Channel data register.
*/
uint32_t SIPI_read_channel_data(CHANNEL_t channel);



 /** \fn SIPI_multiple_read(DATA_TEMPLATE_t read_array[], uint16_t array_length, CHANNEL_t channel, uint8_t injected_error, uint32_t * read_temp)

\brief Performs multiple read transfers. Returns '0' if successful, error code otherwise. Stores read values in struct pointed to. Should call SIPI_read() to process each read in array.
\param[in] read_array[] Pointer to DATA_TEMPLATE_t structure which includes read Address.
\param[in] array_length Amount of data elements in array to be sent.
\param[in] channel SIPI Channel to use. Should be passed as 1 element of CHANNEL array. 
\param[in] injected_error injected error (if required)
\param[out] read_temp provides a pointer to a data structure that will store all read data.
\return
	- 0 = Successfully Set Up
	- 1 = Invalid Data Size
	- 2 = Channel Busy
	- 3 = Invalid Channel
	- 4 = Timeout Error
*/
uint8_t SIPI_multiple_read(DATA_TEMPLATE_t * read_array, uint16_t array_length, CHANNEL_t channel, uint8_t injected_error, uint32_t * read_temp);



/** \fn SIPI_write(DATA_TEMPLATE_t write_data, CHANNEL_t channel, uint8_t injected_error)

\brief Performs a direct write transfer. Returns '0' if successful, error code otherwise.
\param[in] write_data DATA_TEMPLATE_t structure which includes write Address and Data to be written
\param[in] channel SIPI Channel to use. Should be passed as 1 element of CHANNEL array. 
\param[in] injected_error injected error (if required)
\return
	- 0 = Successfully Set Up
	- 1 = Invalid Data Size
	- 2 = Channel Busy
	- 3 = Invalid Channel
	- 4 = Timeout Error / Wrong Acknowledge
*/
uint8_t SIPI_write(DATA_TEMPLATE_t write_data, CHANNEL_t channel, uint8_t injected_error);



/** \fn SIPI_multiple_write(DATA_TEMPLATE_t write_array[], uint16_t array_length, CHANNEL_t channel, uint8_t injected_error, uint8_t DMA_Enable, uint32_t * dma_array)

\brief Performs multiple transfers. Returns '0' if successful, error code otherwise. SIPI_write should be called to
> process each separate write in array, and poll for the SIPI_write_ack function to complete before moving to next message.
\param[in] write_array[] DATA_TEMPLATE_t structure which includes array containing write Address and Data to be written for each array record.
\param[in] array_length Amount of data elements in array to be written.
\param[in] channel SIPI Channel to use. Should be passed as 1 element of CHANNEL array. 
\param[in] injected_error injected error (if required)
\param[in] DMA_Enable Selects whether DMA should be used for transfer or software. Software will form blocking function which will run until all writes complete.
\param[in] dma_array receives a pointer to an array of integers that contains DMA friendly structure
\return
	- 0 = Successfully Set Up
	- 1 = Invalid Data e
	- 2 = Channel Busy
	- 3 = Invalid Channel
	- 4 = Timeout Error / Wrong Acknowledge
*/
uint8_t SIPI_multiple_write(DATA_TEMPLATE_t write_array[], uint16_t array_length, CHANNEL_t channel, uint8_t injected_error, uint8_t DMA_Enable, uint32_t * dma_array);



/** \fn SIPI_stream_transfer(uint32_t *temp_data_stream, uint8_t initiator, uint8_t length)
\brief Performs a streaming write transfer. Returns '0' if successful, error code otherwise.
\param[in] temp_data_stream Pointer to address containing start of data to be streamed.
\param[in] initiator Decides which configuration will be taken, Initiator or Target.
\param[in] length amount of bytes to be sent
\return
	- 0 = Stream Successful
	- 1 = Invalid Length
	- 2 = Acknowledge Error
*/
uint8_t SIPI_stream_transfer(uint32_t * temp_data_stream, uint8_t initiator, uint8_t length);




/** \fn SIPI_ID(uint32_t * id_array, CHANNEL_t channel)

\brief Sends ID Request Frame to target. Stores received command in the address passed. Returns '0' if successful, error code otherwise.
\param[in] id_array used as a dummy data to set CAR to initiate the transfer
\param[in] channel SIPI Channel to use. Should be passed as 1 element of CHANNEL array. 
\return
	- 0 = Successfully Received Acknowledge and ID
	- 2 = Channel Busy
	- 3 = Invalid Channel
	- 4 = Timeout Error
*/
uint8_t SIPI_ID(uint32_t * id_array, CHANNEL_t channel);




/** \fn SIPI_init_INITIATOR(uint16_t Clock_Prescale)

\brief Initializes Initiator side of SIPI module, setting SIPIMCR with clk prescale, AID and MOEN. Returns '0' if successful, error code otherwise.
\param[in] Clock_Prescale Integer representation of Prescale for Timeout Clock. Default is 64. Can be 64, 128, 256, 512 or 1024.
\return
	- 0 = Successfully Set Up Initiator Node
	- 1 = Incorrect Clock Prescale
*/
uint8_t SIPI_init_INITIATOR(uint16_t Clock_Prescale);




/** \fn SIPI_init_TARGET(uint32_t max_count, uint32_t reload_address, uint8_t Add_Inc)

\brief Initializes Target side of SIPI module, setting TEN in SIPIMCR, MAXCR and SIPIARR. Returns '0' if successful, error code otherwise.
\param[in] max_count Maximum address count value of target node
\param[in] reload_address Reload value for the address counter at target node 
\param[in] Add_Inc Integer representation of Address Increment/Decrement bits. Can be 0,1,2 or 3 for 'No change, Increment Address by 4, Decrement Address by 4, or Not Used respectively. 
\return 
	- 0 = Successfully Set Up Target Node
	- 1 = Address Increment Error
	- 2 = Max Count Address Conflicts with Address Count
*/
uint8_t SIPI_init_TARGET(uint32_t max_count, uint32_t reload_address, uint8_t Add_Inc);




/** \fn SIPI_init_channel(CHANNEL_t channel, uint8_t mode, uint8_t error_int_enable, uint8_t data_int_enable)

\brief Initializes SIPI Channels. Sets up CIRn. Will also need to set up interrupts/events to handle received packets appropriately. Clears all errors and events associated with the channel. Returns '0' if successful, error code otherwise.
\param[in] channel SIPI Channel to use. Should be passed as 1 element of CHANNEL array. 
\param[in] mode Sets the channel mode to be used - disabled; clear error; running; or stop request;
\param[in] error_int_enable Set to 1 to enable error interrupts on the channel
\param[in] data_int_enable Set to 1 to enable data interrupts on the channel
\return 
	- 0 = Successfully Set Up Channel
	- 1 = Incorrect Channel Mode
	- 2 = Incorrect Channel
*/
uint8_t SIPI_init_channel(CHANNEL_t channel, uint8_t mode, uint8_t error_int_enable, uint8_t data_int_enable);




/** \fn SIPI_Trigger(CHANNEL_t channel)

\brief Sends Trigger Request Frame to target. Returns '0' if successful, error code otherwise.
\param[in] channel SIPI Channel to use. Should be passed as 1 element of CHANNEL array. 
\return 
	- 0 = Successfully Sent Trigger Command
	- 1 = Incorrect Channel 
	- 2 = Channel Busy
*/
uint8_t SIPI_Trigger(CHANNEL_t channel);




/** \fn SIPI_get_initiator_event(uint8_t channel_number)

\brief Returns 32 bit register showing event status for the channel. Should be polled with mask within calling function to determine if transactions have completed successfully. 
\param[in] channel_number SIPI Channel to use. 
\return
	- 0 = Incorrect Channel
	- SW Channel Status Register
*/
uint32_t SIPI_get_initiator_event(uint8_t channel_number);




/** \fn SIPI_reset()

\brief Performs soft reset of module. Clears all status and error registers, returning the module to 'Disabled'. Any transfers in progress when reset is called will immediately end. Returns '0' if successful, error code otherwise.
\return
	- 0 = Successfully Reset on Module
*/ 
uint8_t SIPI_reset(void);


/** \fn SIPI_module_mode(uint8_t Mode)

\brief Puts the SIPI module into the required mode. Must be used to place module into 'INIT' mode before calling the SIPI_init functions. Returns '0' if successful, error code otherwise.
\param[in] Mode Integer representation of required mode. 0 = Disabled, 1 = Enabled/Init, 2 = Enabled/Run
\return
	- 0 = Successfully Set Up Module Mode
	- 1 = Invalid Mode Selected
*/ 
uint8_t SIPI_module_mode(uint8_t Mode);
