/******************************************************************************
 *
 * Freescale Semiconductor Inc.
 * (c) Copyright 2004-2013 Freescale Semiconductor, Inc.
 * ALL RIGHTS RESERVED.
 *
 ******************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 **************************************************************************//*!
 *
 * @file SwTimer.h
 *
 * @author Samuel Quiroz 
 *
 * @version 1.0
 *
 * @date SEP-11-2009
 *
 * @brief This file handles an array of software timers and trigger the timer event when
 * 		  the timer has elapsed.
 *****************************************************************************/

#ifndef __SWTIMER
#define __SWTIMER

/********************************************************************************************
 * Header files
 *******************************************************************************************/
#include "derivative.h"
#include "fsl_types.h"
#include "lptmr.h"

/********************************************************************************************
 * Definitions
 *******************************************************************************************/
#define MAX_TIMER_OBJECTS				10			/**< This is the max number of timers that the application will have */
#define MAX_COUNTER_OBJECTS				5

#define HW_TIMER_INIT					vfnLptTimerInit(HW_TIMER_DECREMENT_VALUE_MS,LPT_CLK_LPO,LPT_PRE_1,TRUE)	
#define HW_TIMER_IRQ					LPT_INTERRUPT		
#define HW_TIMER_IRQ_CLEAR				LPT_INTERRUPT_CLEAR
#define HW_TIMER_DECREMENT_VALUE_MS		10

/********************************************************************************************
 * Typedefs
 *******************************************************************************************/
typedef struct
{
	uint16_t timerCount;
	pFunc_t timerEvent;
} SwTimerObj_t;

typedef struct
{
	uint16_t timerCount;
} SwCounter_t;

typedef uint8_t SwTimerId_t;

/********************************************************************************************
 * Function prototypes
 *******************************************************************************************/
/** Initializes SwTimer module. Disables all timers. */
void SwTimer_Init(void);

/** This function must be called periodically in the main loop */
void SwTimer_PeriodicTask(void);

/** Starts a timer with a given period 
	\param timerId Number of the timer to start
	\param tickPeriod_ms Timer period in ms
*/
void SwTimer_StartTimer(uint8_t timerId, uint16_t tickPeriod_ms);

/** Stops a timer 
	\param timerId Number of the timer to stop
*/
void SwTimer_StopTimer(uint8_t timerId);

/**
	Creates a timer and assigns it call-back function.
	\param callBackFunc Function to be executed when timer has elapsed
	\return timerId The ID of the timer that was created. It returns INVALID_TIMER_ID (0xFF)
	if the timer was not created (because MAX_TIMER_OBJECTS was reached)
*/
uint8_t SwTimer_CreateTimer(pFunc_t callBackFunc);

/**
	Creates a counter.
	\return The counter ID. This ID will be used to start, stop and read the counter.
			Returns INVALID_TIMER_ID (0xFF= if the counter was not created due to 
			memory limitations (because MAX_TIMER_OBJECTS was reached)
*/
uint8_t SwTimer_CreateCounter(void);

/**
	Starts a counter.
	\param counterId Id of the counter to start.
*/
void SwTimer_StartCounter(uint8_t counterId);

/**
	Stops a counter.
	\param counterId Id of the counter to stop.
*/
void SwTimer_StopCounter(uint8_t counterId);


/**
	Reads a counter.
	\param counterId Id of the counter to read.
*/
uint16_t SwTimer_ReadCounter(uint8_t counterId);

#endif      //SWTIMER
