/*******************************************************************************
*
*  FREESCALE SEMICONDUCTOR INC. 2009-2014
*  ALL RIGHTS RESERVED
*  COPYRIGHT (c)
*
********************************************************************************
*
*  FILE NAME:  etpudef.h
*
*  DESCRIPTION:
*    This file includes standard eTPU(2) definitions, 
*    enabling access to the eTPU(2) hardware features from C language.
*    The usage of these definitions goes with the usage of Freescale 
*    CodeWarrior eTPU(2) compiler.
*    This file is a replacement of the etpu.h included in Freescale CodeWarrior
*    eTPU(2) compiler installation. It includes similar definitions and 
*    is MISRA 2004 compliant.
*
*******************************************************************************/
#ifndef __ETPUDEF_H
#define __ETPUDEF_H


#ifndef __ETPU_H
/*******************************************************************************
*  eTPU(2) Channel Structure
*    The channel structure enables access to the eTPU(2) channel hardware.
*    Only a constant can be assigned to a structure member. Such assignments
*    are compiled into appropriate microinstructions. 
*    See "eTPU(2) Channel Constants" section which defines applicable constants
*    for each structure member.
*******************************************************************************/
typedef struct chan_struct
{
	int24_t CIRC;    /* Channel Interrupt and Data Transfer Requests */
	int24_t ERWA;    /* Write Channel Match A Register */
	int24_t ERWB;    /* Write Channel Match B Register */
	int24_t FLC;     /* Channel Flags Operation */
	int24_t IPACA;   /* Input Pin Action Control A */
	int24_t IPACB;   /* Input Pin Action Control B */
	int24_t OPACA;   /* Output Pin Action Control A */
	int24_t OPACB;   /* Output Pin Action Control B */
	int24_t MRLA;    /* Match Recognition Latch A */
	int24_t MRLB;    /* Match Recognition Latch B */
	int24_t MRLE;    /* Match Recognition Latch Enable */
	int24_t LSR;     /* Link Service Request */
	int24_t MTD;     /* Match and Transition Detection */
	int24_t PDCM;    /* Pre-Defined Channel mode */
	int24_t PIN;     /* Pin State Control */
	int24_t TBSA;    /* Time Base Select A */
	int24_t TBSB;    /* Time Base Select B */
	int24_t TDL;     /* Transition Detection Latch */
	int24_t FLAG0;   /* Channel Flag 0 */
	int24_t FLAG1;   /* Channel Flag 1 */
	int24_t FM0;     /* Function Mode bit 0 */
	int24_t FM1;     /* Function Mode bit 1 */
	int24_t PSS;     /* Pin State at thread start */
	int24_t PSTI;    /* Input Pin State */
	int24_t PRSS;    /* Input Pin State at service request (eTPU2 only) */
	int24_t PSTO;    /* Output Pin State */
	int24_t TDLA;    /* Transition Detection Latch A (eTPU2 only) */
	int24_t TDLB;    /* Transition Detection Latch B (eTPU2 only) */
	int24_t UDCMRWA; /* Write User-Defined Channel Mode (eTPU2 only) */
	int24_t SMPR;    /* Semaphore */
	int24_t MRLEA;   /* Match Recognition Latch A Enable (eTPU2 only) */
	int24_t MRLEB;   /* Match Recognition Latch B Enable (eTPU2 only) */
} chan_struct;
extern chan_struct channel;

/*******************************************************************************
*  eTPU(2) CC Structure
*    The CC structure includes all hardware branch conditions and enables
*    access to them from the C language.
*    All the cc structure members can only have a value of 
*    TRUE (1) of FALSE (0).
*******************************************************************************/
typedef struct register_cc
{
	int24_t C;      /* "Carry" ALU flag */
	int24_t N;      /* "Negative" ALU flag */
	int24_t Z;      /* "Zero" ALU flag */
	int24_t V;      /* "Overflow" ALU flag */
	int24_t MB;     /* "Busy" MDU flag */
	int24_t MC;     /* "Carry" MDU flag */
	int24_t MN;     /* "Negative" MDU flag */
	int24_t MV;     /* "Overflow" MDU flag */
	int24_t MZ;     /* "Zero" MDU flag */
	int24_t TDLA;   /* Transition Detection Latch A channel flag (eTPU2 only) */
	int24_t TDLB;   /* Transition Detection Latch B channel flag (eTPU2 only) */
	int24_t MRLA;   /* Match Recognition Latch A channel flag (eTPU2 only) */
	int24_t MRLB;   /* Match Recognition Latch B channel flag (eTPU2 only) */
	int24_t LSR;    /* Link Service Request channel flag */
	int24_t FM1;    /* Function Mode bit 1 channel flag */
	int24_t FM0;    /* Function Mode bit 0 channel flag */
	int24_t PSS;    /* Pin State at Thread Start channel flag */
	int24_t LT;     /* "Less Then" ALU flag (signed) */
	int24_t LS;     /* "Lower or Equal" ALU flag (unsigned) */
	int24_t P_24;   /* P register bit 24 */
	int24_t P_25;   /* P register bit 25 */
	int24_t P_26;   /* P register bit 26 */
	int24_t P_27;   /* P register bit 27 */
	int24_t P_28;   /* P register bit 28 */
	int24_t P_29;   /* P register bit 29 */
	int24_t P_30;   /* P register bit 30 */
	int24_t P_31;   /* P register bit 31 */
	int24_t PSTO;   /* Output Pin State channel flag */
	int24_t PSTI;   /* Input Pin State channel flag */
	int24_t SMLCK;  /* Semaphore Locked semaphore flag */
	int24_t GE;     /* "Greater Equal" ALU flag (signed) */
	int24_t GT;     /* "Greater Then" ALU flag (unsigned) */
	int24_t FLAG0;  /* Channel Flag 0 */
	int24_t FLAG1;  /* Channel Flag 1 */
/*	int24_t PRSS; */  /* Input Pin State at Service Request channel flag (eTPU2 only) */
} register_cc;
#endif
extern register_cc cc;
#ifndef __ETPU_H

/*******************************************************************************
*  eTPU(2) Tooth Program Register (TPR) Structure
*    The TPR structure is used to control the eTPU(2) Angle Counter (EAC) logic.
*******************************************************************************/
struct tpr_struct {
  unsigned int TICKS    : 10;  /* Angle Ticks Number in the Current Tooth */
  unsigned int MISSCNT2 : 1;   /* Missing Tooth Counter [2] - eTPU2 only */
  unsigned int HOLD     : 1;   /* Force EAC Hold */
  unsigned int IPH      : 1;   /* Insert Physical Tooth */
  unsigned int MISSCNT  : 2;   /* Missing Tooth Counter [1:0] */
  unsigned int LAST     : 1;   /* Last Tooth Indication */
};

#define TPR_LAST   0x8000
#define TPR_IPH    0x1000
#define TPR_HOLD   0x0800

/*******************************************************************************
*  eTPU(2) Registers
*******************************************************************************/
extern register_chan       chan;          /* CHAN Register */
extern register_chan_base  chan_base;     /* CHAN_BASE Register */
#ifdef __ETPU2__
extern register_engine_base engine_base;  /* ENGINE_BASE Register */
#endif
extern register_erta       erta;          /* ERTA Register */
extern register_ertb       ertb;          /* ERTB Register */
extern register_link       link;          /* LINK Register*/
extern register_tcr1       tcr1;          /* Time Count Register 1 (TCR1) */
extern register_tcr2       tcr2;          /* Time (Angle) Count Register 2 (TCR2) */
extern register_tpr        tpr;           /* Tooth Program Register (TPR) */
extern register_trr        trr;           /* Tick Rate Register (TRR) */
#endif

/*******************************************************************************
*  eTPU(2) Channel Constants
*    The following constants are used for channel structure assignments:
*      channel.<member> = <constant>;
*    For example:
*      channel.CIRC = CIRC_INT_FROM_SELECTED;.
*******************************************************************************/
/* CIRC - Channel Interrupt and Data Transfer Requests */
#ifdef __ETPU2__
#define CIRC_INT_FROM_SELECTED  0
#define CIRC_DATA_FROM_SELECTED 1
#define CIRC_BOTH_FROM_SELECTED 2
#define CIRC_BOTH_FROM_SERVICED 3
#define CIRC_INT_FROM_SERVICED  4
#define CIRC_DATA_FROM_SERVICED 5
#define CIRC_GLOBAL_EXCEPTION   6
#define CIRC_NO_REQ             7
#else
#define CIRC_INT_FROM_SERVICED  0
#define CIRC_DATA_FROM_SERVICED 1
#define CIRC_GLOBAL_EXCEPTION   2
#define CIRC_NO_REQ             3
#endif

/* ERWA/ERWB - Write Channel Match A/B Register */
#define ERW_WRITE_ERT_TO_MATCH    0

/* FLC - P Flags Operation */
#define FLC_COPY_FLAGS_1_0_FROM_P_25_24  4
#define FLC_COPY_FLAGS_1_0_FROM_P_27_26  5
#define FLC_COPY_FLAGS_1_0_FROM_P_29_28  6

/* IPACA/IPACB - Input Pin Action Control A/B */
#define IPAC_NO_DETECT            0
#define IPAC_RISING               1
#define IPAC_FALLING              2
#define IPAC_EITHER               3
#define IPAC_INPUT_0_ON_MATCH     4
#define IPAC_INPUT_1_ON_MATCH     5
#define IPAC_NO_CHANGE            7

/* OPACA/OPACB - Output Pin Action Control A/B */
#define OPAC_NO_CHANGE      0
#define OPAC_MATCH_HIGH     1
#define OPAC_MATCH_LOW      2
#define OPAC_MATCH_TOGGLE   3
#define OPAC_TRANS_LOW      4
#define OPAC_TRANS_HIGH     5
#define OPAC_TRANS_TOGGLE   6

/* MRLA/MRLB - Match Recognition Latch A/B */
#define MRL_CLEAR     0

/* MRLE/MRLEA/MRLEB - Match Recognition Latch Enable both/A/B */
#define MRLE_DISABLE  0

/* LSR - Link Service Request */
#define LSR_CLEAR     0

/* MTD - Match and Transition Detection */
#define MTD_ENABLE    0
#define MTD_DISABLE   1

/* PIN - Pin State Control */
#define PIN_AS_OPACA  0
#define PIN_AS_OPACB  1
#define PIN_SET_HIGH  2
#define PIN_SET_LOW   4

/* TBSA/TBSB - Time Base Select A/B */
#define TBS_M1C1GE    0
#define TBS_M2C1GE    1
#define TBS_M1C2GE    2
#define TBS_M2C2GE    3
#define TBS_M1C1EQ    4
#define TBS_M2C1EQ    5
#define TBS_M1C2EQ    6
#define TBS_M2C2EQ    7
/* TBSA - OBE - Output Buffer Enable */
#define TBSA_SET_OBE  8
#define TBSA_CLR_OBE  9

/* TDL/TDLA/TDLB - Transition Detection Latch both/A/B */
#define TDL_CLEAR        0

/* SMPR - Semaphore */
#define SMPR_LOCK_0      0
#define SMPR_LOCK_1      1
#define SMPR_LOCK_2      2
#define SMPR_LOCK_3      3
#define SMPR_FREE      (-1)

/* PDCM - Pre-Defined Channel Mode */
#define PDCM_EM_B_ST  0
#define PDCM_EM_B_DT  1
#define PDCM_EM_NB_ST 2
#define PDCM_EM_NB_DT 3
#define PDCM_M2_ST    4
#define PDCM_M2_DT    5
#define PDCM_BM_ST    6
#define PDCM_BM_DT    7
#define PDCM_M2_O_ST  8
#define PDCM_M2_O_DT  9
#ifdef __ETPU2__
#define PDCM_UDCM     10
#endif
#define PDCM_SM_ST    12
#define PDCM_SM_DT    13
#define PDCM_SM_ST_E  14

#ifdef __ETPU2__
/* UDCM - User-Defined Channel Mode */
/* - Match Service Requests */
#define UDCM_MSR_NO_REQUEST                  (0 << 12)
#define UDCM_MSR_MATCH_B_ONLY                (1 << 12)
#define UDCM_MSR_2ND_MATCH                   (2 << 12)
#define UDCM_MSR_BOTH_MATCHES                (3 << 12)
/* - Match Captures */
#define UDCM_MCAP_OWN                        (0 << 11)  /* a match captures own time */
#define UDCM_MCAP_BOTH                       (1 << 11)  /* either match captures both */
/* - Transition Service Request */
#define UDCM_TSR_ON_1ST_TRANS                (0 << 1)
#define UDCM_TSR_ON_2ND_TRANS                (1 << 1)
/* - Transition Captures */
#define UDCM_TCAP_BOTH                       (0 << 0)  /* 1st transition captures both */
#define UDCM_TCAP_OWN                        (1 << 0)  /* transitions captures own time */
/* - Match A Enable Transitions */
#define UDCM_M1ET_DISABLED                   (0 << 10)
#define UDCM_M1ET_MATCH_A_ENABLES_TRANS      (1 << 10)
/* - Match A Enables Match B */
#define UDCM_M1EM2_DISABLED                  (0 << 9)
#define UDCM_M1EM2_MATCH_A_ENABLES_MATCH_B   (1 << 9)
/* - Match A Blocks Match B */
#define UDCM_M1BM2_DISABLED                  (0 << 8)
#define UDCM_M1BM2_MATCH_A_BLOCKS_MATCH_B    (1 << 8)
/* - Match B Blocks Match A */
#define UDCM_M2BM1_DISABLED                  (0 << 7)
#define UDCM_M2BM1_MATCH_B_BLOCKS_MATCH_A    (1 << 7)
/* - Match B Blocks Transitions */
#define UDCM_M2BT_DISABLED                   (0 << 6)
#define UDCM_M2BT_MATCH_B_BLOCKS_TRANS       (1 << 6)
/* - Transition A Blocks Match A */
#define UDCM_T1BM1_DISABLED                  (0 << 5)
#define UDCM_T1BM1_TRANS_A_BLOCKS_MATCH_A    (1 << 5)
/* - Transition B Blocks Match A */
#define UDCM_T2BM1_DISABLED                  (0 << 4)
#define UDCM_T2BM1_TRANS_B_BLOCKS_MATCH_A    (1 << 4)
/* - Transition Blocks Match B */
#define UDCM_TBM2_TRANS_A_BLOCKS_MATCH_B     (0 << 3)
#define UDCM_TBM2_TRANS_B_BLOCKS_MATCH_B     (1 << 3)
/* - Transition A Enables Transition B */
#define UDCM_T1ET2_DISABLED                  (0 << 2)
#define UDCM_T1ET2_TRANS_A_ENABLES_TRANS_B   (1 << 2)

/* UDCMRWA - Write User-Defined Channel Mode */
#define UDCMRWA_WRITE_ERTA_TO_UDCM   0

/********************************************************************
*  UDCM Usage Example:
*    erta = UDCM_MSR_NO_REQUEST | UDCM_MSR_MATCH_B_ONLY | UDCM_MSR_2ND_MATCH | UDCM_MSR_BOTH_MATCHES
*         + UDCM_MCAP_OWN | UDCM_MCAP_BOTH
*         + UDCM_TSR_ON_1ST_TRANS | UDCM_TSR_ON_2ND_TRANS
*         + UDCM_TCAP_BOTH | UDCM_TCAP_OWN
*         + UDCM_M1ET_DISABLED | UDCM_M1ET_MATCH_A_ENABLES_TRANS
*         + UDCM_M1EM2_DISABLED | UDCM_M1EM2_MATCH_A_ENABLES_MATCH_B
*         + UDCM_M1BM2_DISABLED | UDCM_M1BM2_MATCH_A_BLOCKS_MATCH_B
*         + UDCM_M2BM1_DISABLED | UDCM_M2BM1_MATCH_B_BLOCKS_MATCH_A
*         + UDCM_M2BT_DISABLED | UDCM_M2BT_MATCH_B_BLOCKS_TRANS
*         + UDCM_T1BM1_DISABLED | UDCM_T1BM1_TRANS_A_BLOCKS_MATCH_A
*         + UDCM_T2BM1_DISABLED | UDCM_T2BM1_TRANS_B_BLOCKS_MATCH_A
*         + UDCM_TBM2_TRANS_A_BLOCKS_MATCH_B | UDCM_TBM2_TRANS_B_BLOCKS_MATCH_B
*         + UDCM_T1ET2_DISABLED | UDCM_T1ET2_TRANS_A_ENABLES_TRANS_B;
*    channel.UDCMRWA = UDCMRWA_WRITE_ERTA_TO_UDCM;
*    channel.PDCM = PDCM_UDCM;
********************************************************************/
#endif

#ifndef __ETPU_H
/*******************************************************************************
*  General Definitions
*******************************************************************************/
typedef char Bool;

#ifndef NULL
#define NULL 0L
#endif

#ifndef TRUE
#define TRUE        1
#endif

#ifndef FALSE
#define FALSE       0
#endif

/*******************************************************************************
*  Math Intrinsics
*******************************************************************************/
int24_t _s32_div_f(register_b int24_t dividend, register_a int24_t divisor);
int24_t _smod(register_b int24_t dividend, register_a int24_t divisor);
void __clear(register_a int8_t* buf, register_b int24_t len);

/*******************************************************************************
*  Math supported by eTPU ALU but not naturally available in C language
*******************************************************************************/
inline int f24i24muls(fract24_t f, int24_t i)
{
	int ret;
	asm{.nonvolatile;
		muls.f f@Rn, i@Rn
	L1:	jmp.if (mb) L1
		addi   ret@Rn, mach, 0
	}
	return ret;
}

inline int f24i24mulu(ufract24_t f, uint24_t i)
{
	int ret;
	asm{.nonvolatile;
		mulu.f f@Rn, i@Rn
	L2:	jmp.if (mb) L2
		addi   ret@Rn, mach, 0
	}
	return ret;
}
#endif

#endif /* __ETPUDEF_H */

/*******************************************************************************
 *
 * Copyright:
 *	Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ******************************************************************************/
/*******************************************************************************
*
*  REVISION HISTORY:
*
*  FILE OWNER: Milan Brejl [r54529]
*  Revision 1.1  2014/10/27  r54529
*  TPR Structure updated for eTPU2+ (MISSCNT2 added).
*
*  Revision 1.0  2014/03/16  r54529
*  User-Defined Channel Mode definitions added.
*  Minor comment and formating improvements.
*  Ready for release 1.0.
*
*  Revision 0.1  2012/06/13  r54529
*  Initial version. Parallel use with etpu.h from CW enabled.
* 
*******************************************************************************/
